package cn.wjee.commons.thread;

import cn.wjee.commons.constants.LogVar;
import cn.wjee.commons.lang.RandomUtils;
import lombok.NonNull;
import org.slf4j.MDC;

import java.util.Map;
import java.util.Objects;
import java.util.concurrent.*;

/**
 * 线程池
 *
 * @author listening
 */
public class MyThreadPoolExecutor extends ThreadPoolExecutor {

    public MyThreadPoolExecutor(int corePoolSize,
                                int maximumPoolSize,
                                long keepAliveTime,
                                TimeUnit unit,
                                BlockingQueue<Runnable> workQueue,
                                ThreadFactory threadFactory,
                                RejectedExecutionHandler handler) {
        super(corePoolSize, maximumPoolSize, keepAliveTime, unit, workQueue, threadFactory, handler);
    }

    /**
     * 构建简单线程池
     *
     * @param name 名称
     * @return ThreadPoolExecutor
     */
    public static ThreadPoolExecutor getSimplePool(String name) {
        // IO密集型: 线程数 =2N cpu, 计算密集型: 线程数 = N cpu + 1
        int processors = Runtime.getRuntime().availableProcessors();

        ThreadPoolExecutor threadPoolExecutor = new MyThreadPoolExecutor(
            processors * 2,
            processors * 5,
            300,
            TimeUnit.SECONDS,
            new ArrayBlockingQueue<>(1000, true),
            new MyThreadFactory(name),
            new ThreadPoolExecutor.CallerRunsPolicy());

        Runtime.getRuntime().addShutdownHook(new Thread(threadPoolExecutor::shutdown));
        return threadPoolExecutor;
    }

    @Override
    public void execute(@NonNull Runnable command) {
        super.execute(wrap(command, MDC.getCopyOfContextMap()));
    }

    @NonNull
    @Override
    public <T> Future<T> submit(@NonNull Callable<T> task) {
        return super.submit(wrap(task, MDC.getCopyOfContextMap()));
    }

    /**
     * 用于父线程向线程池中提交任务时，将自身MDC中的数据赋值给子线程
     *
     * @param callable 线程任务
     * @param context  MDC Context
     * @param <T>      类型
     * @return Runnable
     */
    public static <T> Callable<T> wrap(final Callable<T> callable, final Map<String, String> context) {
        return () -> {
            if (Objects.isNull(context)) {
                MDC.clear();
            } else {
                MDC.setContextMap(context);
            }
            if (Objects.isNull(MDC.get(LogVar.TRACE_ID))) {
                MDC.put(LogVar.TRACE_ID, RandomUtils.getUuid());
            }
            try {
                return callable.call();
            } finally {
                MDC.clear();
            }
        };
    }

    /**
     * 用于父线程向线程池中提交任务时，将自身MDC中的数据赋值给子线程
     *
     * @param runnable 线程任务
     * @param context  MDC Context
     * @return Runnable
     */
    public static Runnable wrap(final Runnable runnable, final Map<String, String> context) {
        return () -> {
            if (Objects.isNull(context)) {
                MDC.clear();
            } else {
                MDC.setContextMap(context);
            }
            if (Objects.isNull(MDC.get(LogVar.TRACE_ID))) {
                MDC.put(LogVar.TRACE_ID, RandomUtils.getUuid());
            }
            try {
                runnable.run();
            } finally {
                MDC.clear();
            }
        };
    }


}
