package co.faraboom.framework.calendar;

import co.faraboom.framework.exception.ResponseCodes;
import co.faraboom.framework.exception.ServiceException;
import co.faraboom.framework.exception.ServiceExceptionType;
import co.faraboom.framework.util.GeneralUtil;
import org.apache.commons.lang3.time.DateUtils;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.Locale;
import java.util.TimeZone;

public class DateUtil {
    public static final long SECOND_MILLIS = 1000;
    public static final String DATE_DASH_T_TIME_TIMEZONE_PATTERN_BOOM = "yyyy-MM-dd'T'HH:mm:ssZ";
    public static final String DATE_DASH_T_ZERO_TIME_PATTERN = "yyyy-MM-dd'T'00:00:00";
    public static final String DATE_DASH_T_TIME_Z_PATTERN_BOOM_OLD = "yyyy-MM-dd'T'HH:mm:ss'Z'";
    public static final String DATE_DASH_T_TIME_MILLISECONDS_Z_PATTERN = "yyyy-MM-dd'T'HH:mm:ss.SSSSSSS'Z'";
    public static final String DATE_SLASH_SPACE_TIME_MILLISECONDS_PATTERN = "yyyy/MM/dd HH:mm:s:SSS";
    public static final String DATE_DASH_T_TIME_MILLISECONDS_PATTERN = "yyyy-MM-dd'T'HH:mm:ss.SSS'Z'";
    public static final String DATE_SLASH_SPACE_TIME_PATTERN = "yyyy/MM/dd HH:mm:ss";
    public static final String DATE_SLASH_SPACE_TIME_PATTERN_MONTH_FIRST = "MM/dd/yyyy HH:mm:ss";
    public static final String DATE_SLASH_SPACE_TIME_PATTERN_DAY_IN_LEFT = "dd/MM/yyyy HH:mm:ss";
    public static final String DATE_TIME_PATTERN = "yyMMddHHmmss";
    //    public static final String LONG_DATE_TIME_PATTERN = "yyyyMMddHHmmss";
    public static final String LONG_DATE_TIME_PATTERN = "YYMMDDhhmmss";
    public static final String DATE_DASH_SPACE_TIME_PATTERN = "yyyy-MM-dd HH:mm:ss";
    public static final String START_TIME_Z_PATTERN = "T00:00:00'Z'";
    public static final String END_TIME_Z_PATTERN = "T23:59:59'Z'";
    public static final String DATE_DASH_PATTERN = "yyyy-MM-dd";
    public static final String DATE_SLASH_PATTERN = "yyyy/MM/dd";
    public static final String TIME_TIMEZONE_PATTERN = "HH:mm:ssZ";
    public static final String TIME_Z_PATTERN_OLD = "HH:mm:ss'Z'";
    public static final String TIME_PATTERN = "HHmmss";
    public static final String DATE_DASH_T_TIME_PATTERN = "yyyy-MM-dd'T'HH:mm:ss";
    public static final String DATE_SLASH_SPACE_TIME_12HOUR_FORMAT = "M/d/yyyy hh:mm:ss aa";
    //    public static final String DATE_ONLY_DATE_PATTERN = "yyMMdd";
    public static final String DATE_ONLY_DATE_PATTERN = "YYMMDD";

    public static String convertDateNowToString() {
        return convertDateToString(getCurrentDateTime());
    }

    public static String convertDateNowUtcToString() {
        DateFormat dateFormat = new SimpleDateFormat(DATE_DASH_T_TIME_Z_PATTERN_BOOM_OLD, Locale.ENGLISH);
        dateFormat.setTimeZone(TimeZone.getTimeZone("UTC"));
        return dateFormat.format(new Date());
    }

    public static String getTimeFromDate(Date date) {
        DateFormat dateFormat = new SimpleDateFormat(TIME_PATTERN, Locale.ENGLISH);
        dateFormat.setTimeZone(TimeZone.getTimeZone(getDefaultTimeZone()));
        return dateFormat.format(date);
    }

    public static String convertDateToString(Date date) {
        DateFormat dateFormat = new SimpleDateFormat(DATE_DASH_T_TIME_TIMEZONE_PATTERN_BOOM, Locale.ENGLISH);
        dateFormat.setTimeZone(TimeZone.getTimeZone(getDefaultTimeZone()));
        return dateFormat.format(date);
    }

    public static String convertDateToStringWithCustomePattern(Date date, String pattern) {
        if (GeneralUtil.isNull(date)) {
            return null;
        }
        DateFormat dateFormat = new SimpleDateFormat(pattern, Locale.ENGLISH);
        dateFormat.setTimeZone(TimeZone.getTimeZone(getDefaultTimeZone()));
        return dateFormat.format(date);
    }

    public static Date convertStringToDate(String dateInString, String pattern) throws ServiceException {
        SimpleDateFormat formatter = new SimpleDateFormat(pattern);

        Date date = null;
        try {
            if (dateInString != null) {
                date = formatter.parse(dateInString);
            }
            return date;
        } catch (Exception exception) {
            throw new ServiceException(ResponseCodes.INVALID_DATE, ServiceExceptionType.Bad_Request);
        }
    }

    public static Date getCurrentDate() {
        TimeZone.setDefault(TimeZone.getTimeZone(getDefaultTimeZone()));
        return DateUtils.truncate(new Date(), Calendar.DAY_OF_MONTH);
    }

    public static Date getCurrentDateTime() {
        TimeZone.setDefault(TimeZone.getTimeZone(getDefaultTimeZone()));
        return new Date();
    }

    public static String getCurrentDateTime(String pattern) {
        return convertDateToStringWithCustomePattern(getCurrentDateTime(), pattern);
    }

    public static Date getFirstDateOfWeek(Date date) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        calendar.set(Calendar.DAY_OF_WEEK, calendar.getActualMinimum(Calendar.DAY_OF_WEEK));

        return calendar.getTime();
    }

    public static Date addOrRemoveSecondsForDate(Date date, int timeModifyBySeconds) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);

        calendar.add(Calendar.SECOND, timeModifyBySeconds);
        return calendar.getTime();
    }

    public static Date addOrRemoveDaysForDate(Date date, int timeModifyByDays) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);

        calendar.add(Calendar.DAY_OF_MONTH, timeModifyByDays);
        return calendar.getTime();
    }

    public static int secondsDiff(Date earlierDate, Date laterDate) {
        if (earlierDate == null || laterDate == null) return 0;

        return (int) ((laterDate.getTime() / SECOND_MILLIS) - (earlierDate.getTime() / SECOND_MILLIS));
    }

    public static Long calculateDuration(Long startTime, Long endTime) {
        // Backend never got called, we're better off signaling this scenario with a null return value
        if (startTime == null) return null;

        endTime = (endTime != null) ? endTime : System.currentTimeMillis();

        return endTime - startTime;
    }

    public static int calculateSolarCalendar() {

        return calendarCalculator(getCurrentDateTime());
    }

    public static int solarCalendarForDate(String date) throws ServiceException {
        return calendarCalculator(convertStringToDate(date, DATE_DASH_PATTERN));
    }

    public static int solarCalendarForDate(Date date) {
        return calendarCalculator(date);
    }

    private static int calendarCalculator(Date date) {
        StringBuilder finalId = new StringBuilder();
        SolarCalendar calendar = new SolarCalendar(date);

        finalId.append(calendar.getYear());
        int length = String.valueOf(calendar.getMonth()).length();

        if (length == 1)
            finalId.append(0);
        finalId.append(calendar.getMonth());
        length = String.valueOf(calendar.getDate()).length();

        if (length == 1)
            finalId.append(0);
        finalId.append(calendar.getDate());

        return Integer.parseInt(finalId.toString());
    }

    public static String getDefaultTimeZone() {
        String timeZone = GeneralUtil.getEnvironmentVariable("user.timezone");
        if (GeneralUtil.isNullOrEmpty(timeZone))
            timeZone = "UTC";
        return timeZone;
    }
}