package co.faraboom.framework.util.security;

import co.faraboom.framework.exception.ResponseCodes;
import co.faraboom.framework.exception.ServiceException;
import co.faraboom.framework.exception.ServiceExceptionType;
import org.bouncycastle.cert.jcajce.JcaCertStore;
import org.bouncycastle.cms.CMSProcessableByteArray;
import org.bouncycastle.cms.CMSSignedData;
import org.bouncycastle.cms.CMSSignedDataGenerator;
import org.bouncycastle.cms.CMSTypedData;
import org.bouncycastle.cms.jcajce.JcaSignerInfoGeneratorBuilder;
import org.bouncycastle.crypto.engines.DESEngine;
import org.bouncycastle.crypto.macs.ISO9797Alg3Mac;
import org.bouncycastle.crypto.params.KeyParameter;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.operator.ContentSigner;
import org.bouncycastle.operator.jcajce.JcaContentSignerBuilder;
import org.bouncycastle.operator.jcajce.JcaDigestCalculatorProviderBuilder;
import org.bouncycastle.util.Store;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import co.faraboom.framework.util.GeneralUtil;

import javax.crypto.*;
import javax.crypto.spec.IvParameterSpec;
import javax.crypto.spec.PBEKeySpec;
import javax.crypto.spec.SecretKeySpec;
import java.io.*;
import java.nio.charset.StandardCharsets;
import java.security.*;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.KeySpec;
import java.util.ArrayList;
import java.util.Base64;


public class CryptoUtil {

    private static final Logger LOGGER = LoggerFactory.getLogger(CryptoUtil.class);

    private static final char[] STORE_PASS = "DezLCZtvdxCfuLRPO7RWACXhir9p7XRkCyIxp".toCharArray();
    private static final char[] KEY_PASS = "Textgs79zBmquj6DzewCBoMz3LmLPqtPS9".toCharArray();
    private static final String ALIAS = "encks";
    private static final String IV_BASE64 = "T86El4mqHmfFBOUbbWFT7w==";
    private static final String INTERNAL_PASSWORD = "T0s@Nb00m";
    private static final String INTERNAL_SALT = "3c8ca8c0-d3e8-4f60-afc2-7c3ed27c11af";

    public static final String AES_CBC_WITH_RANDOM_PADDING = "AES/CBC/PKCS5Padding";
    public static final String AES_ECB_WITH_RANDOM_PADDING = "AES/ECB/PKCS5Padding";
    public static final String DESEDE_ECB_ZERO_BYTE_PADDING = "DESede/ECB/ZeroBytePadding";

    public static String encryptAES(String clearText) {
        return encryptAES(clearText, INTERNAL_PASSWORD);
    }

    public static String encryptAES(String clearText, String password) {
        try {
            return encryptAES(clearText, password, INTERNAL_SALT);
        } catch (ServiceException e) {
            return clearText;
        }
    }

    public static String encryptAES(String clearText, String password, String salt) throws ServiceException {
        try {
            LOGGER.info("start encrypt in AesCryptoService to encrypt the credentials using AES 256");
            if (GeneralUtil.isNullOrEmpty(clearText))
                return "";

            SecretKey secretKey = generateSecretKey(password, salt);

            Cipher cipher = Cipher.getInstance(AES_CBC_WITH_RANDOM_PADDING);
            cipher.init(Cipher.ENCRYPT_MODE, new SecretKeySpec(secretKey.getEncoded(), AES_CBC_WITH_RANDOM_PADDING.split("/")[0]),
                    new IvParameterSpec(secretKey.getEncoded(), 0, 16));

            ByteArrayOutputStream out = new ByteArrayOutputStream();
            CipherOutputStream cipherOut = new CipherOutputStream(out, cipher);
            OutputStreamWriter writer = new OutputStreamWriter(cipherOut, StandardCharsets.UTF_8);

            writer.write(clearText);
            writer.close();

            LOGGER.info("The clear text just got encrypted");

            return Base64.getEncoder().encodeToString(out.toByteArray());
        } catch (Exception ex) {
            LOGGER.error("Failed to encrypt: ", ex);
            throw new ServiceException(ResponseCodes.FAILED_TO_ENCRYPT, ServiceExceptionType.Internal_Server_Error);
        }
    }

    public static String decryptAES(String encryptedText) {
        return decryptAES(encryptedText, INTERNAL_PASSWORD);
    }

    public static String decryptAES(String encryptedText, String password) {
        try {
            return decryptAES(encryptedText, password, INTERNAL_SALT);
        } catch (ServiceException e) {
            return encryptedText;
        }
    }

    public static String decryptAES(String encryptedText, String password, String salt) throws ServiceException {
        try {
            LOGGER.info("start decrypt in AesCryptoService  to decrypt the given cipher text using AES 256");
            if (GeneralUtil.isNullOrEmpty(encryptedText))
                return "";

            SecretKey secretKey = generateSecretKey(password, salt);

            Cipher cipher = Cipher.getInstance(AES_CBC_WITH_RANDOM_PADDING);
            cipher.init(Cipher.DECRYPT_MODE, new SecretKeySpec(secretKey.getEncoded(), AES_CBC_WITH_RANDOM_PADDING.split("/")[0]),
                    new IvParameterSpec(secretKey.getEncoded(), 0, 16));

            ByteArrayInputStream in = new ByteArrayInputStream(Base64.getDecoder().decode(encryptedText));
            CipherInputStream cipherIn = new CipherInputStream(in, cipher);
            BufferedReader reader = new BufferedReader(new InputStreamReader(cipherIn, StandardCharsets.UTF_8));

            LOGGER.info("The passed cipher text just got decrypted");

            return reader.readLine();
        } catch (Exception exc) {
            LOGGER.error("Failed to decrypt: ", exc);
            throw new ServiceException(ResponseCodes.FAILED_TO_DECRYPT, ServiceExceptionType.Internal_Server_Error);
        }
    }

    public static byte[] encryptSHA1(String clearText) throws ServiceException {
        try {
            MessageDigest sha1 = MessageDigest.getInstance("SHA1");
            return sha1.digest(clearText.getBytes(StandardCharsets.UTF_8));
        } catch (NoSuchAlgorithmException exc) {
            LOGGER.error("Failed to encrypt: ", exc);
            throw new ServiceException(ResponseCodes.FAILED_TO_ENCRYPT, ServiceExceptionType.Internal_Server_Error);
        }
    }

    public static String encryptTripleDES(byte[] value, byte[] key) throws ServiceException {
        try {
            Cipher cipher = Cipher.getInstance(DESEDE_ECB_ZERO_BYTE_PADDING, new BouncyCastleProvider());
            cipher.init(Cipher.ENCRYPT_MODE, new SecretKeySpec(GetKeyBytes(key), DESEDE_ECB_ZERO_BYTE_PADDING.split("/")[0]));
            return SecurityUtil.encodeHexStr(cipher.doFinal(value));
        } catch (Exception exc) {
            LOGGER.error("Failed to encrypt: ", exc);
            throw new ServiceException(ResponseCodes.FAILED_TO_ENCRYPT, ServiceExceptionType.Internal_Server_Error);
        }
    }

    public static byte[] decryptTripleDES(byte[] value, byte[] key) throws ServiceException {
        try {
            Cipher cipher = Cipher.getInstance(DESEDE_ECB_ZERO_BYTE_PADDING, new BouncyCastleProvider());
            cipher.init(Cipher.DECRYPT_MODE, new SecretKeySpec(key, DESEDE_ECB_ZERO_BYTE_PADDING.split("/")[0]));
            return cipher.doFinal(value);
        } catch (Exception exc) {
            LOGGER.error("Failed to decrypt: ", exc);
            throw new ServiceException(ResponseCodes.FAILED_TO_DECRYPT, ServiceExceptionType.Internal_Server_Error);
        }
    }

    public static String generateISO9797Alg3Mac(String data, String key) {
        ISO9797Alg3Mac iso9797Alg3Mac = new ISO9797Alg3Mac(new DESEngine());
        iso9797Alg3Mac.init(new KeyParameter(SecurityUtil.HexFromString(key)));
        iso9797Alg3Mac.update(data.getBytes(StandardCharsets.UTF_8), 0, data.length());
        byte[] out = new byte[8];
        iso9797Alg3Mac.doFinal(out, 0);

        return SecurityUtil.encodeHexStr(out);
    }

    private static byte[] GetKeyBytes(byte[] key) throws Exception {
        if (null == key || key.length < 1)
            throw new Exception("key is null or empty!");
        int start = key.length;
        byte[] key24 = new byte[24];
        for (int i = 0; i < start; i++) {
            key24[i] = key[i];
        }
        for (int i = start; i < 24; i++) {// for compatibility with .net16-bit keys
            key24[i] = key[i - start];
        }
        return key24;
    }

    private static SecretKey getSecretKey() throws ServiceException {
        KeyStore keyStore = getKeyStore();

        try {
            return (SecretKey) keyStore.getKey(ALIAS, KEY_PASS);
        } catch (Exception ex) {
            LOGGER.error("Failed to load secret key: ", ex);
            throw new ServiceException(ResponseCodes.FAILED_TO_LOAD_SECRET_KEY, ServiceExceptionType.Internal_Server_Error);
        }
    }

    private static KeyStore getKeyStore() throws ServiceException {
        final InputStream in = Thread.currentThread().getContextClassLoader().getResourceAsStream("/encryption-ks.jks");
        try {
            KeyStore keyStore = KeyStore.getInstance("JCEKS");
            keyStore.load(in, STORE_PASS);

            return keyStore;
        } catch (Exception ex) {
            LOGGER.error("Failed to load keystore");
            throw new ServiceException(ResponseCodes.FAILED_TO_LOAD_KEYSTORE, ServiceExceptionType.Internal_Server_Error);
        }
    }

    private static IvParameterSpec getInitializationVector() {
        return new IvParameterSpec(Base64.getDecoder().decode(IV_BASE64));
    }

    private static byte[] encrypt(String clearText) throws ServiceException {
        try {
            LOGGER.info("start encrypt in AesCryptoService to encrypt the credentials using AES 256");

            Cipher cipher = Cipher.getInstance(AES_CBC_WITH_RANDOM_PADDING);
            cipher.init(Cipher.ENCRYPT_MODE, getSecretKey(), getInitializationVector());

            ByteArrayOutputStream out = new ByteArrayOutputStream();
            CipherOutputStream cipherOut = new CipherOutputStream(out, cipher);
            OutputStreamWriter writer = new OutputStreamWriter(cipherOut, StandardCharsets.UTF_8);

            writer.write(clearText);
            writer.close();

            LOGGER.info("The clear text just got encrypted");

            return out.toByteArray();
        } catch (Exception ex) {
            LOGGER.error("Failed to encrypt: ", ex);
            throw new ServiceException(ResponseCodes.FAILED_TO_ENCRYPT, ServiceExceptionType.Internal_Server_Error);
        }
    }

    private static String decrypt(byte[] cipherByteArray) throws ServiceException {
        try {
            LOGGER.info("start decrypt in AesCryptoService  to decrypt the given cipher text using AES 256");

            Cipher cipher = Cipher.getInstance(AES_CBC_WITH_RANDOM_PADDING);
            cipher.init(Cipher.DECRYPT_MODE, getSecretKey(), getInitializationVector());

            ByteArrayInputStream in = new ByteArrayInputStream(cipherByteArray);
            CipherInputStream cipherIn = new CipherInputStream(in, cipher);
            BufferedReader reader = new BufferedReader(new InputStreamReader(cipherIn, StandardCharsets.UTF_8));

            LOGGER.info("The passed cipher text just got decrypted");

            return reader.readLine();
        } catch (Exception ex) {
            LOGGER.error("Failed to decrypt: ", ex);
            throw new ServiceException(ResponseCodes.FAILED_TO_DECRYPT, ServiceExceptionType.Internal_Server_Error);
        }
    }

    private static SecretKey generateSecretKey(String password, String salt) {
        byte[] saltBytes = salt.getBytes(StandardCharsets.US_ASCII);

        SecretKeyFactory factory = null;
        try {
            factory = SecretKeyFactory.getInstance("PBKDF2WithHmacSHA1");
        } catch (NoSuchAlgorithmException e2) {
            e2.printStackTrace();
        }
        KeySpec spec = new PBEKeySpec(password.toCharArray(), saltBytes, 1024, 256);
        SecretKey secretKey = null;
        try {
            secretKey = (factory != null) ? factory.generateSecret(spec) : null;
        } catch (InvalidKeySpecException e2) {
            e2.printStackTrace();
        }
        return secretKey;
    }

    public static KeyStore loadKeyStore(byte[] storeFile, String instance, String password) throws ServiceException {
        KeyStore keystore;
        try {
            keystore = KeyStore.getInstance(instance);
            InputStream is = new ByteArrayInputStream(storeFile);
            keystore.load(is, password.toCharArray());
            return keystore;
        } catch (KeyStoreException | CertificateException | NoSuchAlgorithmException | IOException e) {
            LOGGER.error("Failed to load keystore");
            throw new ServiceException(ResponseCodes.SERVICE, ServiceExceptionType.Internal_Server_Error);
        }

    }

    public static String cmsSign(String b64Message, PrivateKey pk, X509Certificate cert, boolean attachMessage) throws ServiceException {
        try {
            Security.addProvider(new BouncyCastleProvider());

            ArrayList certList = new ArrayList();
            certList.add(cert);

            CMSTypedData data = new CMSProcessableByteArray(org.bouncycastle.util.encoders.Base64.decode(b64Message));

            Store certs = new JcaCertStore(certList);

            CMSSignedDataGenerator gen = new CMSSignedDataGenerator();

            ContentSigner shaSigner = (new JcaContentSignerBuilder("SHA256withRSA")).setProvider("BC").build(pk);

            gen.addSignerInfoGenerator((new JcaSignerInfoGeneratorBuilder((new JcaDigestCalculatorProviderBuilder()).setProvider("BC").build())).build(shaSigner, cert));
            gen.addCertificates(certs);

            CMSSignedData sigData = gen.generate(data, attachMessage);

            String signedCms = Base64.getEncoder().encodeToString(sigData.getEncoded());

            return signedCms;
        } catch (Exception exc) {
            LOGGER.error("Failed to sign date: ", exc);
            throw new ServiceException(ResponseCodes.SERVICE, ServiceExceptionType.Internal_Server_Error);
        }
    }
}