package co.faraboom.framework.validation;

import co.faraboom.framework.exception.ResponseCodes;
import co.faraboom.framework.exception.ServiceException;
import co.faraboom.framework.exception.ServiceExceptionType;

import java.io.Serializable;

public final class IbanCheckDigit implements CheckDigit, Serializable {

    private static final int MIN_CODE_LEN = 5;

    private static final long serialVersionUID = -3600191725934382801L;

    private static final int MAX_ALPHANUMERIC_VALUE = 35; // Character.getNumericValue('Z')
    public static final CheckDigit IBAN_CHECK_DIGIT = new IbanCheckDigit();

    private static final long MAX = 999999999;

    private static final long MODULUS = 97;

    public IbanCheckDigit() {
    }

    public boolean isValid(String code) {
        if (code == null || code.length() < MIN_CODE_LEN) {
            return false;
        }
        String check = code.substring(2, 4); // CHECKSTYLE IGNORE MagicNumber
        if ("00".equals(check) || "01".equals(check) || "99".equals(check)) {
            return false;
        }
        try {
            int modulusResult = calculateModulus(code);
            return (modulusResult == 1);
        } catch (ServiceException ex) {
            return false;
        }
    }

    @Override
    public String calculate(String code) throws ServiceException{
        if (code == null || code.length() < MIN_CODE_LEN) {
            throw new ServiceException(ResponseCodes.INVALID_IBAN, ServiceExceptionType.Bad_Request);
        }
        code = code.substring(0, 2) + "00" + code.substring(4); // CHECKSTYLE IGNORE MagicNumber
        int modulusResult = calculateModulus(code);
        int charValue = (98 - modulusResult); // CHECKSTYLE IGNORE MagicNumber
        String checkDigit = Integer.toString(charValue);
        return (charValue > 9 ? checkDigit : "0" + checkDigit); // CHECKSTYLE IGNORE MagicNumber
    }

    private int calculateModulus(String code) throws ServiceException {
        String reformattedCode = code.substring(4) + code.substring(0, 4); // CHECKSTYLE IGNORE MagicNumber
        long total = 0;
        for (int i = 0; i < reformattedCode.length(); i++) {
            int charValue = Character.getNumericValue(reformattedCode.charAt(i));
            if (charValue < 0 || charValue > MAX_ALPHANUMERIC_VALUE) {
                throw new ServiceException(ResponseCodes.INVALID_IBAN, ServiceExceptionType.Bad_Request);
            }
            total = (charValue > 9 ? total * 100 : total * 10) + charValue; // CHECKSTYLE IGNORE MagicNumber
            if (total > MAX) {
                total = total % MODULUS;
            }
        }
        return (int) (total % MODULUS);
    }
}