package co.faraboom.framework.util;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.MapperFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import co.faraboom.framework.exception.ResponseCodes;
import co.faraboom.framework.exception.ServiceException;
import co.faraboom.framework.exception.ServiceExceptionType;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.interceptor.InvocationContext;
import javax.ws.rs.PathParam;
import javax.ws.rs.QueryParam;
import java.lang.annotation.Annotation;
import java.lang.reflect.Field;
import java.math.BigDecimal;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class GeneralUtil {
    public static final String DELIMITER = "___";
    public static final String DELIMITER_ALTERNATE = "\\|";
    public static final String TOSAN_BOOM_REGISTRY_KEY = "tosanboomregistry";
    public static final String ALPHABETIC_PATTERN = "^(?U)[\\p{Alnum}\\-. ]*";
    public static final String MASK_SENSITIVE_RESPONSE = "******";
    public static final String MOBILE_REGEX = "(\\+989|00989|09|989|9)(\\d{9})";
    public static final Pattern MOBILE_PATTERN = Pattern.compile(MOBILE_REGEX, Pattern.CASE_INSENSITIVE);
    private static final Logger LOGGER = LoggerFactory.getLogger(GeneralUtil.class);
    private static final int IPV4_MAX_OCTET_VALUE = 255;
    private static final int MAX_UNSIGNED_SHORT = 0xffff;
    private static final int BASE_16 = 16;
    private static final String IPV4_REGEX = "^(\\d{1,3})\\.(\\d{1,3})\\.(\\d{1,3})\\.(\\d{1,3})$";
    private static final int IPV6_MAX_HEX_GROUPS = 8;
    private static final int IPV6_MAX_HEX_DIGITS_PER_GROUP = 4;
    private static final Pattern IP4_PATTERN = Pattern.compile(IPV4_REGEX, Pattern.CASE_INSENSITIVE);
    private static final Set<Class<?>> WRAPPER_TYPES = getWrapperTypes();

    public static void requireNonNull(Object value, String message) throws ServiceException {
        if (value == null)
            throw new ServiceException(message, ServiceExceptionType.Bad_Request);
    }

    public static void requireNonNullOrEmpty(String value, String message) throws ServiceException {
        if (value == null || value.trim().isEmpty())
            throw new ServiceException(message, ServiceExceptionType.Bad_Request);
    }

    public static void requireGreaterThanOrEqualZero(Long value, String message) throws ServiceException {
        if (value == null || value < 0)
            throw new ServiceException(message, ServiceExceptionType.Bad_Request);
    }

    public static <T extends Number> void requireGreaterThanZero(T value, String message) throws ServiceException {
        if (value == null || value.longValue() <= 0)
            throw new ServiceException(message, ServiceExceptionType.Bad_Request);
    }

    public static void requireGreaterThanZero(BigDecimal value, String message) throws ServiceException {
        if (value == null || value.signum() <= 0)
            throw new ServiceException(message, ServiceExceptionType.Bad_Request);
    }

    public static void validateAmount(BigDecimal value, String message) throws ServiceException {
        if (value != null && value.signum() <= 0)
            throw new ServiceException(message, ServiceExceptionType.Bad_Request);
    }

    public static String getBan(String iban) {
        return iban.substring(4, 7);
    }

    public static boolean isNullOrEmpty(String str) {
        return str == null || str.trim().isEmpty();
    }

    public static boolean isNullOrEmpty(Iterable iterable) {
        if (iterable == null)
            return true;

        if (iterable instanceof Collection)
            return ((Collection) iterable).isEmpty();

        return !iterable.iterator().hasNext();
    }

    public static boolean isNullOrEmpty(Map map) {
        if (map == null)
            return true;

        return map.isEmpty();
    }

    public static boolean isNullOrEmpty(Object[] array) {
        return array == null || array.length == 0;
    }

    public static boolean isNull(Object object) {
        return object == null;
    }

    public static Long parseLong(String value) {
        try {
            return Long.parseLong(value);
        } catch (NumberFormatException exception) {
            LOGGER.error("utilities.GeneralUtil, parseLong error is:", exception);
            return null;
        }
    }

    public static Boolean parseBoolean(String value) {
        try {
            return Boolean.parseBoolean(value);
        } catch (NumberFormatException exception) {
            LOGGER.error("utilities.GeneralUtil, parseBoolean error is:", exception);
            return false;
        }
    }

    public static void setOrValidatePagingInformation(Pageable pageable) throws ServiceException {
        Objects.requireNonNull(pageable, "utilities.Pageable can't be null");

        if (pageable.getLength() == null)
            pageable.setLength(1L);
        else
            //requireGreaterThanZero(pageable.getLength(), exeptions.ResponseCodes.INVALID_LENGTH);
            requireGreaterThanOrEqualZero(pageable.getLength(), ResponseCodes.INVALID_LENGTH);

        if (pageable.getOffset() == null)
            pageable.setOffset(0L);
        else
            requireGreaterThanOrEqualZero(pageable.getOffset(), ResponseCodes.INVALID_OFFSET);
    }

    public static String getServerLocalIpAddress() {
        InetAddress ip;
        try {
            ip = InetAddress.getLocalHost();
            return ip.getHostAddress();
        } catch (UnknownHostException e2) {

            LOGGER.error("error in getActiveBanks ip {} ", e2.getMessage());
            return "";
        }
    }

    public static String convertToJson(InvocationContext invocationContext) throws JsonProcessingException {
        ObjectMapper mapper = createMapper();
        Object[] parameters = invocationContext.getParameters();

        if (parameters.length == 0)
            return null;
        else if (parameters.length <= 1) {
            return mapper.writeValueAsString(parameters[0]);

        }
        Annotation[][] annotations = invocationContext.getMethod().getParameterAnnotations();

        Map<String, String> pathParameters = new HashMap<>();
        Map<String, String> bodyParameters = new HashMap<>();
        int i = 0;
        while (i < parameters.length) {
            Object parameter = parameters[i];
            if (parameter != null) {
                String clazz = parameter.getClass().getSimpleName();
                if ("String".equals(clazz) || "Boolean".equals(clazz)
                        || "Integer".equals(clazz) || "Date.".equals(clazz)
                        || "Long".equals(clazz) || "BigDecimal".equals(clazz)) {
                    if (annotations[i].length > 0) {
                        String property = annotations[i][0] instanceof PathParam
                                ? ((PathParam) annotations[i][0]).value()
                                : ((QueryParam) annotations[i][0]).value();
                        pathParameters.put(property, mapper.writeValueAsString(parameter));
                    }
                } else {
                    bodyParameters.put(parameter.getClass().getSimpleName(), mapper.writeValueAsString(parameter));
                }
            }
            i++;
        }

        for (Map.Entry<String, String> pathParam : pathParameters.entrySet()) {
            String pathParamKey = pathParameters.get(pathParam.getKey());
            boolean remove = false;
            for (Map.Entry<String, String> bodyParameter : bodyParameters.entrySet()) {
                String bodyValue = bodyParameters.get(bodyParameter.getKey());
                String pattern = pathParam.getKey() + "\":null";
                String pattern2 = pathParam.getKey() + "\":" + pathParamKey;
                if (bodyValue.contains(pattern) || bodyValue.contains(pattern2)) {
                    bodyParameters.put(bodyParameter.getKey(),
                            bodyValue.replace(pattern, pathParam.getKey() + "\":" + pathParamKey));
                    remove = true;
                }
            }
            if (remove)
                pathParameters.remove(pathParam.getKey());
        }

        if (pathParameters.size() + bodyParameters.size() == 1) {
            return pathParameters.size() == 1 ?
                    pathParameters.values().toArray()[0].toString()
                    : bodyParameters.values().toArray()[0].toString();
        } else {
            StringBuilder stringBuilder = new StringBuilder();
            stringBuilder.append("{");

            buildJson(pathParameters, stringBuilder);
            buildJson(bodyParameters, stringBuilder);

            String value = stringBuilder.toString();
            if (value.endsWith(","))
                value = value.substring(0, value.length() - 1);

            value += "}";
            return value;
        }
    }

    public static String convertToJsonForHmac(InvocationContext invocationContext) throws JsonProcessingException {
        ObjectMapper mapper = createMapper();

        Object[] parameters = invocationContext.getParameters();

        if (parameters.length == 0)
            return null;
        Map<String, String> bodyParameters = new HashMap<>();
        int i = 0;
        while (i < parameters.length) {
            Object parameter = parameters[i];
            if (parameter != null) {
                String clazz = parameter.getClass().getSimpleName();
                if (!"String".equals(clazz) && !"Boolean".equals(clazz)
                        && !"Integer".equals(clazz) && !"Date.".equals(clazz)
                        && !"Long".equals(clazz) && !"BigDecimal".equals(clazz)) {
                    bodyParameters.put(parameter.getClass().getSimpleName(), mapper.writeValueAsString(parameter));
                }
            }
            i++;
        }

        if (bodyParameters.size() == 0) {
            return null;
        } else if (bodyParameters.size() == 1) {
            return bodyParameters.values().toArray()[0].toString();
        } else {
            StringBuilder stringBuilder = new StringBuilder();
            stringBuilder.append("{");

            buildJson(bodyParameters, stringBuilder);

            String value = stringBuilder.toString();
            if (value.endsWith(","))
                value = value.substring(0, value.length() - 1);

            value += "}";
            return value;
        }
    }

    public static Boolean validateIpAddress(String ipAddress) {
        return isValidInet4Address(ipAddress) || isValidInet6Address(ipAddress);
    }

    public static String getEnvironmentVariable(String envName) {
        String envVal = System.getProperty(envName);
        LOGGER.debug("Environment variable is: {} and value is: {}", envName, envVal);
        return envVal;
    }

    public static String normalizeString(String input) {
        if (isNullOrEmpty(input))
            return input;

        return input.replace("ﮎ", "ک").replace("ﮏ", "ک")
                .replace("ﮐ", "ک").replace("ﮑ", "ک").replace("ك", "ک")
                .replace("ي", "ی").replace("ھ", "ه");
    }


    public static ObjectMapper createMapper() {
        ObjectMapper jsonMapper = new ObjectMapper();
        jsonMapper.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
        jsonMapper.configure(MapperFeature.ACCEPT_CASE_INSENSITIVE_PROPERTIES, true);
        jsonMapper.setSerializationInclusion(JsonInclude.Include.NON_NULL);
        return jsonMapper;
    }


    public static DataType getDataType(Field field) {
        Class<?> clazz = field.getType();
        boolean isArray = isArray(clazz);
        if (isArray) {
            String typeName = field.getAnnotatedType().getType().getTypeName();
            String className = typeName.contains("<") && typeName.contains(">") ? typeName.substring(typeName.indexOf("<") + 1, typeName.indexOf(">"))
                    : typeName.contains("[]") ? typeName.substring(0, typeName.indexOf("[]")) : typeName;
            try {
                clazz = Class.forName(convertPrimitiveToWrapperName(className));
            } catch (ClassNotFoundException e) {
            }
        }

        return getDataType(clazz, isArray);
    }

    public static DataType getDataType(Class<?> clazz) {
        boolean isArray = isArray(clazz);
        return getDataType(clazz, isArray);
    }

    public static boolean isPrimitiveOrWrapperType(Class<?> clazz) {
        return clazz.isPrimitive() || WRAPPER_TYPES.contains(clazz);
    }

    public static String convertPrimitiveToWrapperName(String name) {
        switch (name) {
            case "boolean":
                return "java.lang.Boolean";
            case "char":
                return "java.lang.Character";
            case "byte":
                return "java.lang.Byte";
            case "short":
                return "java.lang.Short";
            case "int":
                return "java.lang.Integer";
            case "long":
                return "java.lang.Long";
            case "float":
                return "java.lang.Float";
            case "double":
                return "java.lang.Double";
            default:
                return name;
        }
    }

    public static String normalizePhoneNumber(String phoneNumber) {
        if (isNullOrEmpty(phoneNumber))
            return phoneNumber;

        if (phoneNumber.startsWith("00980"))
            phoneNumber = "98" + phoneNumber.substring(5);
        else if (phoneNumber.startsWith("+980"))
            phoneNumber = "98" + phoneNumber.substring(4);
        else if (phoneNumber.startsWith("0098"))
            phoneNumber = phoneNumber.substring(2);
        else if (phoneNumber.startsWith("+98"))
            phoneNumber = phoneNumber.substring(1);
        else if (phoneNumber.startsWith("0"))
            phoneNumber = "98" + phoneNumber.substring(1);
        else if (!phoneNumber.startsWith("98"))
            phoneNumber = "98" + phoneNumber;

        return phoneNumber;
    }

    public static boolean isValidMobile(String input) {
        return GeneralUtil.MOBILE_PATTERN.matcher(input).matches();
    }

    public static boolean isValidCard(String input) {
        if (isNullOrEmpty(input) || input.length() != 16 || !StringUtils.isNumeric(input))
            return false;
        char[] chars = input.toCharArray();
        int total = 0;
        for (int i = 0; i < chars.length; i++) {
            int factor = Integer.valueOf(String.valueOf(chars[i])) * ((i % 2 == 0) ? 2 : 1);
            total += factor > 9 ? factor - 9 : factor;
        }
        return ((total % 10) == 0);
    }

    public static AbstractMap.SimpleEntry<String, String> extractName(String value) {
        String firstName = null;
        String lastName = null;
        if (!isNullOrEmpty(value)) {
            String[] lst = value.split(" ");
            if (lst.length == 2) {
                firstName = lst[0];
                lastName = lst[1];
            } else {
                firstName = value;
            }
        }

        return new AbstractMap.SimpleEntry<>(firstName, lastName);
    }

    public static boolean isArray(Class<?> clazz) {
        String name = clazz.getName();
        return "java.co.faraboom.framework.util.Arrays".equals(name) || "java.co.faraboom.framework.util.List".equals(name) || "java.co.faraboom.framework.util.ArrayList".equals(name) || clazz.isArray();
    }

    public static void sameBankAssertion(String sourceIban, String destinationIban) throws ServiceException {
        String sourceBan = getBan(sourceIban);
        String destinationBan = getBan(destinationIban);
        if (sourceBan.equalsIgnoreCase(destinationBan))
            throw new ServiceException(ResponseCodes.MUST_DIFFERENT_BAN, ServiceExceptionType.Bad_Request);
    }

    private static DataType getDataType(Class<?> clazz, boolean isArray) {
        if (clazz.isEnum()) {
            return isArray ? DataType.ArrayOfEnum : DataType.Enum;
        }

        switch (clazz.getName()) {
            case "java.lang.Boolean":
            case "boolean":
                return isArray ? DataType.ArrayOfBoolean : DataType.Boolean;
            case "java.lang.Character":
            case "java.lang.String":
            case "char":
                return isArray ? DataType.ArrayOfString : DataType.String;
            case "java.lang.Byte":
            case "byte":
                return isArray ? DataType.ArrayOfByte : DataType.Byte;
            case "java.lang.Short":
            case "short":
                return isArray ? DataType.ArrayOfInt16 : DataType.Int16;
            case "java.lang.Integer":
            case "java.lang.Number":
            case "java.math.BigInteger":
            case "int":
                return isArray ? DataType.ArrayOfInt32 : DataType.Int32;
            case "java.lang.Long":
            case "long":
                return isArray ? DataType.ArrayOfInt64 : DataType.Int64;
            case "java.lang.Float":
            case "float":
                return isArray ? DataType.ArrayOfFloat : DataType.Float;
            case "java.lang.Double":
            case "double":
                return isArray ? DataType.ArrayOfDouble : DataType.Double;
            case "java.math.BigDecimal":
                return isArray ? DataType.ArrayOfDecimal : DataType.Decimal;
            case "java.co.faraboom.framework.util.Date":
                return isArray ? DataType.ArrayOfDateTime : DataType.DateTime;
            default:
                return isArray ? DataType.ArrayOfObject : DataType.Object;
        }
    }

    private static Set<Class<?>> getWrapperTypes() {
        Set<Class<?>> ret = new HashSet<>();
        ret.add(Boolean.class);

        ret.add(Character.class);
        ret.add(String.class);

        ret.add(Byte.class);

        ret.add(Short.class);

        ret.add(Integer.class);
        ret.add(Number.class);
        ret.add(java.math.BigInteger.class);

        ret.add(Long.class);

        ret.add(Float.class);

        ret.add(Double.class);

        ret.add(System.class);

        ret.add(Enum.class);
        ret.add(Void.class);
        ret.add(BigDecimal.class);

        ret.add(Date.class);
        return ret;
    }

    private static boolean isValidInet4Address(String inet4Address) {
        // verify that address conforms to generic IPv4 format
        String[] groups = null;
        Matcher matcher = IP4_PATTERN.matcher(inet4Address);
        if (matcher.matches()) {
            int count = matcher.groupCount();
            groups = new String[count];
            for (int i = 0; i < count; i++) {
                groups[i] = matcher.group(i + 1);
            }
        }

        if (groups == null)
            return false;

        // verify that address subgroups are legal
        for (String ipSegment : groups) {
            if (ipSegment == null || ipSegment.length() == 0) {
                return false;
            }

            int iIpSegment = 0;

            try {
                iIpSegment = Integer.parseInt(ipSegment);
            } catch (NumberFormatException e) {
                return false;
            }

            if (iIpSegment > IPV4_MAX_OCTET_VALUE) {
                return false;
            }

            if (ipSegment.length() > 1 && ipSegment.startsWith("0")) {
                return false;
            }

        }

        return true;
    }

    private static boolean isValidInet6Address(String inet6Address) {
        boolean containsCompressedZeroes = inet6Address.contains("::");
        if (containsCompressedZeroes && (inet6Address.indexOf("::") != inet6Address.lastIndexOf("::"))) {
            return false;
        }
        if ((inet6Address.startsWith(":") && !inet6Address.startsWith("::"))
                || (inet6Address.endsWith(":") && !inet6Address.endsWith("::"))) {
            return false;
        }
        String[] octets = inet6Address.split(":");
        if (containsCompressedZeroes) {
            List<String> octetList = new ArrayList<String>(Arrays.asList(octets));
            if (inet6Address.endsWith("::")) {
                // String.split() drops ending empty segments
                octetList.add("");
            } else if (inet6Address.startsWith("::") && !octetList.isEmpty()) {
                octetList.remove(0);
            }
            octets = octetList.toArray(new String[octetList.size()]);
        }
        if (octets.length > IPV6_MAX_HEX_GROUPS) {
            return false;
        }
        int validOctets = 0;
        int emptyOctets = 0; // consecutive empty chunks
        for (int index = 0; index < octets.length; index++) {
            String octet = octets[index];
            if (octet.length() == 0) {
                emptyOctets++;
                if (emptyOctets > 1) {
                    return false;
                }
            } else {
                emptyOctets = 0;
                // Is last chunk an IPv4 address?
                if (index == octets.length - 1 && octet.contains(".")) {
                    if (!isValidInet4Address(octet)) {
                        return false;
                    }
                    validOctets += 2;
                    continue;
                }
                if (octet.length() > IPV6_MAX_HEX_DIGITS_PER_GROUP) {
                    return false;
                }
                int octetInt = 0;
                try {
                    octetInt = Integer.parseInt(octet, BASE_16);
                } catch (NumberFormatException e) {
                    return false;
                }
                if (octetInt < 0 || octetInt > MAX_UNSIGNED_SHORT) {
                    return false;
                }
            }
            validOctets++;
        }
        if (validOctets > IPV6_MAX_HEX_GROUPS || (validOctets < IPV6_MAX_HEX_GROUPS && !containsCompressedZeroes)) {
            return false;
        }
        return true;
    }

    private static void buildJson(Map<String, String> map, StringBuilder stringBuilder) {
        for (Map.Entry<String, String> mapEntry : map.entrySet()) {
            stringBuilder.append("\"");
            stringBuilder.append(mapEntry.getKey());
            stringBuilder.append("\":");
            stringBuilder.append(map.get(mapEntry.getKey()));
            stringBuilder.append(",");
        }
    }

    public static Long extractBillAmountFromPayId(String payId, String fieldName) throws ServiceException {
        if (payId.length() < 6 || payId.length() > 13) {
            throw new ServiceException(ResponseCodes.ILLEGAL_PAY_ID, ServiceExceptionType.Bad_Request)
                    .setValue(payId)
                    .setReference(StringUtil.toSnakeCase(fieldName));
        }
        return Long.parseLong(payId.substring(0, payId.length() - 5)) * 1000;
    }

    public static boolean amountValidation(BigDecimal value, int maxIntegral, int maxFractional, BigDecimal minValue) {
        if (isNull(value)) {
            return true;
        }
        if (maxIntegral < 0 || maxFractional < 0) {
            throw new IllegalArgumentException(
                    "Negative value for Integer and fraction is prevent");
        }
        boolean minValidate = value.compareTo(minValue) != -1;
        if (minValidate) {
            value = value.stripTrailingZeros();

            int intLength = value.precision() - value.scale();
            if (intLength <= maxIntegral) {
                return value.scale() <= maxFractional;
            } else {
                return false;
            }
        }
        return false;
    }

    public static boolean isValidPostalCode(String postalCode) {
        if (postalCode.length() != 10
                || !org.apache.commons.lang.StringUtils.isNumeric(postalCode)
                || postalCode.substring(0, 6).contains("0")
                || postalCode.substring(0, 5).contains("2")
                || postalCode.charAt(4) == '5'
                || postalCode.substring(6).equals("0000")) {
            return false;
        }

        for (int i = 0; i <= 9; i++) {
            int finalI = i;
            long count = postalCode.chars().filter(ch -> ch == (char) finalI + '0').count();
            if (count > 6)
                return false;
        }

        return true;
    }

    public static boolean isValidNationalId(String nationalId) {
        if (!org.apache.commons.lang.StringUtils.isNumeric(nationalId) || nationalId.length() < 8
                || nationalId.length() > 10) {
            return false;
        }
        nationalId = org.apache.commons.lang.StringUtils.leftPad(nationalId, 10, "0");

        String[] portions = nationalId.split("");

        int j = 10;
        int sum = 0;

        for (int i = 0; i < portions.length - 1; i++) {
            sum += Integer.parseInt(portions[i]) * j --;
        }

        int remainder = sum % 11;

        int controlNumber = Integer.parseInt(portions[portions.length - 1]);
        return remainder < 2 && controlNumber == remainder
                || remainder >= 2 && controlNumber == 11 - remainder;
    }

}
