package co.faraboom.framework.util.security;

import co.faraboom.framework.exception.ResponseCodes;
import co.faraboom.framework.exception.ServiceException;
import co.faraboom.framework.exception.ServiceExceptionType;
import org.apache.commons.lang3.RandomStringUtils;
import org.apache.commons.text.StringEscapeUtils;
import org.apache.cxf.common.util.Base64Exception;
import org.apache.cxf.common.util.Base64Utility;
import org.apache.openjpa.persistence.ArgumentException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Element;
import org.xml.sax.SAXException;
import co.faraboom.framework.util.FilterFlag;
import co.faraboom.framework.util.GeneralUtil;
import co.faraboom.framework.validation.IbanValidator;

import javax.crypto.*;
import javax.crypto.spec.IvParameterSpec;
import javax.crypto.spec.SecretKeySpec;
import javax.servlet.http.HttpServletRequest;
import javax.xml.bind.DatatypeConverter;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.math.BigInteger;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.security.*;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.RSAPrivateCrtKeySpec;
import java.security.spec.RSAPublicKeySpec;
import java.util.Arrays;
import java.util.Base64;
import java.util.EnumSet;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class SecurityUtil {
    public final static String BEARER_TOKEN_TYPE = "Bearer";
    public final static String HMAC_TOKEN_TYPE = "bmx";
    public final static String BASIC_TOKEN_TYPE = "Basic";

    private final static Logger LOGGER = LoggerFactory.getLogger(SecurityUtil.class);
    private final static String names = "\"(?i)(username|password|pin|pin2|depositNumber|cvv2|expDate|iban|sessionId|accountId|counterpartyAccount|availableBalance|ledgerBalance|cardPassword|expireDate|sourceNumber|destinationNumber|sourceCardNumber|destinationCardNumber|affectedAccount|affectedDestinationAccount|SrcDepositNumber|SrcCardNumber|DestDepositNumber|DestCardNumber|cardDestination|cardSource|srccardno|destcardno|TargetPan|SourcePan|Pan|nonFractionalRemainder|file|reportData|sourceCard|destinationCard|destination|cardHolderNonFractionalAmount|actualAmount|availableAmount|balance|extAccNo|extCustId|diban|sIban|tokenId|nationalId|accountNumber|customerNumber|sourceDepositNumber|ibanNumber|customer-id|session)\"";
    private final static String stringRegex = names + "\\s*:\\s*\"(.*?)\"";
    private final static String baseNumericRegex = names + "\\s*:\\s*([0-9]*?)";
    private final static String numericCommaRegex = baseNumericRegex + ",";
    private final static String numericSemicolonRegex = baseNumericRegex + ";";
    private final static String numericCurlyBracketRegex = baseNumericRegex + "}";
    private final static String numericSquareBracketRegex = baseNumericRegex + "]";

    private final static Pattern stringPattern = Pattern.compile(stringRegex, Pattern.CASE_INSENSITIVE);
    private final static Pattern numericCommaPattern = Pattern.compile(numericCommaRegex, Pattern.CASE_INSENSITIVE);
    private final static Pattern numericSemicolonPattern = Pattern.compile(numericSemicolonRegex, Pattern.CASE_INSENSITIVE);
    private final static Pattern numericCurlyBracketPattern = Pattern.compile(numericCurlyBracketRegex, Pattern.CASE_INSENSITIVE);
    private final static Pattern numericSquareBracketPattern = Pattern.compile(numericSquareBracketRegex, Pattern.CASE_INSENSITIVE);

    private final static Pattern stripTagsRegex = Pattern.compile("<[^<>]*>", Pattern.CASE_INSENSITIVE);
    private final static List<Pattern> regexList =
            Arrays.asList(
                    Pattern.compile("<script[^>]*>.*?</script[^><]*>", Pattern.CASE_INSENSITIVE),
                    Pattern.compile("<script", Pattern.CASE_INSENSITIVE),
                    Pattern.compile("javascript:", Pattern.CASE_INSENSITIVE),
                    Pattern.compile("vbscript:", Pattern.CASE_INSENSITIVE),
                    Pattern.compile("unescape", Pattern.CASE_INSENSITIVE),
                    Pattern.compile("alert[\\s(&nbsp;)]*\\([\\s(&nbsp;)]*'?[\\s(&nbsp;)]*[\"(&quot;)]?", Pattern.CASE_INSENSITIVE),
                    Pattern.compile("eval*.\\(", Pattern.CASE_INSENSITIVE)
            );

    private static final char[] DIGITS_LOWER = {'0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'a', 'b', 'c', 'd', 'e', 'f'};
    private static final char[] DIGITS_UPPER = {'0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'A', 'B', 'C', 'D', 'E', 'F'};

    public static String generateHash(String secretKey, String raw) throws ServiceException {
        if (GeneralUtil.isNullOrEmpty(raw))
            return null;

        byte[] secretKeyByteArray = createSecretKey(secretKey);

        byte[] signature;

        signature = raw.getBytes(StandardCharsets.UTF_8);

        Mac sha256HMAC;
        try {
            sha256HMAC = Mac.getInstance("HmacSHA256");
        } catch (NoSuchAlgorithmException e) {
            throw new ServiceException(ResponseCodes.SERVICE, ServiceExceptionType.Internal_Server_Error);

        }
        SecretKeySpec secretKeySpec = new SecretKeySpec(secretKeyByteArray, "HmacSHA256");
        try {
            sha256HMAC.init(secretKeySpec);
        } catch (InvalidKeyException e) {
            throw new ServiceException(ResponseCodes.SERVICE, ServiceExceptionType.Internal_Server_Error);

        }
        return Base64Utility.encode(sha256HMAC.doFinal(signature));
    }

    private static byte[] createSecretKey(String appSecret) throws ServiceException {
        try {
            return Base64Utility.decode(appSecret);
        } catch (Base64Exception e) {
            throw new ServiceException(ResponseCodes.SERVICE, ServiceExceptionType.Internal_Server_Error);
        }
    }

    public static String generateHashForHmac(String secretKey, String raw) throws ServiceException {
        LOGGER.info("request string body to generate Hmac :" + raw);
        if (GeneralUtil.isNullOrEmpty(raw))
            return null;

        byte[] secretKeyByteArray = createSecretKeyForHmac(secretKey);

        byte[] signature;

        signature = raw.getBytes(StandardCharsets.UTF_8);


        Mac sha256HMAC;
        try {
            sha256HMAC = Mac.getInstance("HmacSHA256");
        } catch (NoSuchAlgorithmException e) {
            throw new ServiceException(ResponseCodes.SERVICE, ServiceExceptionType.Internal_Server_Error,
                    e.getLocalizedMessage());

        }
        SecretKeySpec secretKeySpec = new SecretKeySpec(secretKeyByteArray, "HmacSHA256");
        try {
            sha256HMAC.init(secretKeySpec);
        } catch (InvalidKeyException e) {
            throw new ServiceException(ResponseCodes.SERVICE, ServiceExceptionType.Internal_Server_Error,
                    e.getLocalizedMessage());

        }
        return Base64Utility.encode(sha256HMAC.doFinal(signature));
    }

    private static byte[] createSecretKeyForHmac(String secretKey) {
        return secretKey.getBytes(StandardCharsets.UTF_8);
    }

    public static String signSHA1withRSA(String plainText, PrivateKey privateKey)
            throws InvalidKeyException, NoSuchAlgorithmException, SignatureException {
        Signature privateSignature = Signature.getInstance("SHA1withRSA");
        privateSignature.initSign(privateKey);
        privateSignature.update(plainText.getBytes(StandardCharsets.UTF_8));

        byte[] signature = privateSignature.sign();

        return Base64.getEncoder().encodeToString(signature);
    }

    public static String signSHA256withRSA(String staString, PrivateKey pk) throws NoSuchAlgorithmException, InvalidKeyException, SignatureException {
        Signature generator = Signature.getInstance("SHA256withRSA");
        generator.initSign(pk);
        generator.update(staString.getBytes(StandardCharsets.UTF_8));
        byte[] result = generator.sign();
        return BinToHex(result);
    }

    public static boolean verifySHA256withRSA(String plainText, String signature, PublicKey publicKey) throws Exception {
        Signature publicSignature = Signature.getInstance("SHA256withRSA");
        publicSignature.initVerify(publicKey);
        publicSignature.update(plainText.getBytes(StandardCharsets.UTF_8));

        byte[] signatureBytes = Base64.getDecoder().decode(signature);

        return publicSignature.verify(signatureBytes);
    }

    public static String encryptRSA(String plainText, PublicKey publicKey) throws NoSuchPaddingException, NoSuchAlgorithmException, InvalidKeyException, BadPaddingException, IllegalBlockSizeException {
        Cipher encryptCipher = Cipher.getInstance("RSA");
        encryptCipher.init(Cipher.ENCRYPT_MODE, publicKey);

        byte[] cipherText = encryptCipher.doFinal(plainText.getBytes(StandardCharsets.UTF_8));

        return Base64.getEncoder().encodeToString(cipherText);
    }

    public static String decryptRSA(String cipherText, PrivateKey privateKey) throws Exception {
        byte[] bytes = Base64.getDecoder().decode(cipherText);

        Cipher decryptCipher = Cipher.getInstance("RSA");
        decryptCipher.init(Cipher.DECRYPT_MODE, privateKey);

        return new String(decryptCipher.doFinal(bytes), StandardCharsets.UTF_8);
    }

    public static String encryptAES(String plainText, byte[] key, byte[] iv) throws NoSuchAlgorithmException,
            NoSuchPaddingException, InvalidKeyException, InvalidAlgorithmParameterException, BadPaddingException,
            IllegalBlockSizeException {
        if (GeneralUtil.isNullOrEmpty(plainText))
            return plainText;
        IvParameterSpec ivParameterSpec = new IvParameterSpec(iv);
        Cipher cipher = Cipher.getInstance(CryptoUtil.AES_CBC_WITH_RANDOM_PADDING);
        SecretKeySpec secretKey = new SecretKeySpec(key, CryptoUtil.AES_CBC_WITH_RANDOM_PADDING.split("/")[0]);
        cipher.init(Cipher.ENCRYPT_MODE, secretKey, ivParameterSpec);
        byte[] cipherText = cipher.doFinal(plainText.getBytes(StandardCharsets.UTF_8));
        return new String(Base64.getEncoder().encode(cipherText), StandardCharsets.UTF_8);
    }

    public static String decryptAes(String encryptedText, byte[] key, byte[] iv) throws NoSuchAlgorithmException, NoSuchPaddingException, InvalidAlgorithmParameterException, InvalidKeyException, IOException, BadPaddingException, IllegalBlockSizeException {

        if (GeneralUtil.isNullOrEmpty(encryptedText))
            return encryptedText;
        Cipher cipher = Cipher.getInstance(CryptoUtil.AES_CBC_WITH_RANDOM_PADDING);
        cipher.init(Cipher.DECRYPT_MODE, new SecretKeySpec(key, CryptoUtil.AES_CBC_WITH_RANDOM_PADDING.split("/")[0]), new IvParameterSpec(iv));
        byte[] cipherText = cipher.doFinal(Base64.getDecoder().decode(encryptedText));
        return new String(cipherText, StandardCharsets.UTF_8);
    }

    public static String getAuthorizationHeader(HttpServletRequest request, String tokenType) {
        if (GeneralUtil.isNull(request) || GeneralUtil.isNullOrEmpty(tokenType))
            throw new ArgumentException("request", null, true, true);

        String authorization = request.getHeader("Authorization");
        if (GeneralUtil.isNullOrEmpty(authorization))
            return null;

        String[] lst = authorization.split(",");
        for (String aLst : lst) {
            String[] lst2 = aLst.trim().split("\\s", 2);
            if (tokenType.equals(lst2[0].trim())) {
                return lst2[1].trim();
            }
        }

        return null;
    }

    public static PrivateKey loadPrivateKeyFromXMl(String key) throws ParserConfigurationException, NoSuchAlgorithmException,
            IOException, SAXException, InvalidKeySpecException {

        DocumentBuilder documentBuilder = DocumentBuilderFactory.newInstance().newDocumentBuilder();
        Element element = documentBuilder.parse(new ByteArrayInputStream(key
                .getBytes(StandardCharsets.UTF_8))).getDocumentElement();

        String[] names = {"Modulus", "Exponent", "D", "P", "Q", "DP", "DQ", "InverseQ"};
        BigInteger[] values = new BigInteger[names.length];
        for (int i = 0; i < names.length; i++) {
            String pkValue = element.getElementsByTagName(names[i]).item(0).getTextContent();
            values[i] = new BigInteger(1, DatatypeConverter.parseBase64Binary(pkValue));
        }
        return KeyFactory.getInstance("RSA")
                .generatePrivate(new RSAPrivateCrtKeySpec
                        (values[0], values[1], values[2], values[3],
                                values[4], values[5], values[6], values[7]));

    }

    public static PublicKey loadPublicKeyFromXMl(String key) throws ParserConfigurationException, NoSuchAlgorithmException,
            IOException, SAXException, InvalidKeySpecException {

        DocumentBuilder documentBuilder = DocumentBuilderFactory.newInstance().newDocumentBuilder();
        Element element = documentBuilder.parse(new ByteArrayInputStream(key
                .getBytes(StandardCharsets.UTF_8))).getDocumentElement();

        String[] names = {"Modulus", "Exponent"};
        BigInteger[] values = new BigInteger[names.length];
        for (int i = 0; i < names.length; i++) {
            String pkValue = element.getElementsByTagName(names[i]).item(0).getTextContent();
            values[i] = new BigInteger(1, DatatypeConverter.parseBase64Binary(pkValue));
        }
        return KeyFactory.getInstance("RSA")
                .generatePublic(new RSAPublicKeySpec(values[0], values[1]));

    }

    public static String eraseSensitiveInformation(String text) {
        if (GeneralUtil.isNullOrEmpty(text))
            return text;

        String str = stringPattern.matcher(text).replaceAll("\"$1\":\"***\"");
        str = numericCommaPattern.matcher(str).replaceAll("\"$1\":111,");
        str = numericSemicolonPattern.matcher(str).replaceAll("\"$1\":111;");
        str = numericCurlyBracketPattern.matcher(str).replaceAll("\"$1\":111}");
        str = numericSquareBracketPattern.matcher(str).replaceAll("\"$1\":111]");

        return str;
    }

    public static String maskAccount(String inputValue) {
        if (GeneralUtil.isNullOrEmpty(inputValue))
            return "";

        if (GeneralUtil.isValidCard(inputValue)) {  //card
            return inputValue.substring(0, 6) + "******" + inputValue.substring(12);
        } else if (IbanValidator.getInstance().isValid(inputValue)) {    //iban
            return inputValue.substring(0, 10) + "******" + inputValue.substring(16);
        } else { //deposit
            String requiredMask = "";
            String lastDigits = "";

            if (inputValue.length() > 3) {
                lastDigits = inputValue.substring(inputValue.length() - 4);

                for (int i = 0; i <= inputValue.length() - lastDigits.length(); i++) {
                    requiredMask = requiredMask + "*";
                }
            }
            return requiredMask + lastDigits;
        }
    }

    public static String BinToHex(byte[] bytes) {
        char[] hexArray = "0123456789ABCDEF".toCharArray();
        char[] hc = new char[bytes.length * 2];
        for (int j = 0; j < bytes.length; j++) {
            int v = bytes[j] & 0xFF;
            hc[j * 2] = hexArray[v >>> 4];
            hc[j * 2 + 1] = hexArray[v & 0x0F];
        }
        return new String(hc);
    }

    public static byte[] HexFromString(String hex) {
        int len = hex.length();
        byte[] buf = new byte[((len + 1) / 2)];

        int i = 0, j = 0;
        if ((len % 2) == 1)
            buf[j++] = (byte) fromDigit(hex.charAt(i++));

        while (i < len) {
            buf[j++] = (byte) ((fromDigit(hex.charAt(i++)) << 4) | fromDigit(hex
                    .charAt(i++)));
        }
        return buf;
    }

    private static int fromDigit(char ch) {
        if (ch >= '0' && ch <= '9')
            return ch - '0';
        if (ch >= 'A' && ch <= 'F')
            return ch - 'A' + 10;
        if (ch >= 'a' && ch <= 'f')
            return ch - 'a' + 10;

        throw new IllegalArgumentException("invalid hex digit '" + ch + "'");
    }

    public static String inputFilter(String userInput, EnumSet<FilterFlag> filters) {

        if (userInput == null) {
            return "";
        }

        if (filters.contains(FilterFlag.NO_ANGL_EBRACKETS)) {
            userInput = formatAngleBrackets(userInput);
        }

        if (filters.contains(FilterFlag.NO_MARKUP) && includesMarkup(userInput)) {
            userInput = StringEscapeUtils.escapeHtml4(userInput);
            userInput = StringEscapeUtils.escapeHtml4(userInput);
        }

        if (filters.contains(FilterFlag.NO_SCRIPTING)) {
            userInput = formatDisableScripting(userInput);
        }

        if (filters.contains(FilterFlag.MULTI_LINE)) {
            userInput = formatMultiLine(userInput);
        }
        return userInput;
    }

    private static String formatAngleBrackets(String userInput) {

        return userInput.replace("<", "").replace(">", "");
    }

    private static String formatDisableScripting(String strInput) {

        return GeneralUtil.isNullOrEmpty(strInput) ? strInput : filterStrings(strInput);
    }

    private static String formatMultiLine(String userInput) {

        String lbreak = "<br />";
        return userInput.replaceAll("\r\n", lbreak).replaceAll("\n", lbreak).replaceAll("\r", lbreak);
    }

    private static boolean includesMarkup(String strInput) {

        Matcher m = stripTagsRegex.matcher(strInput);
        return m.matches();
    }

    private static String filterStrings(String userInput) {

        String replacement = " ";
        if (userInput.contains("&gt;") || userInput.contains("&lt;")) {
            //text is encoded, so decode and try again
            userInput = StringEscapeUtils.unescapeHtml4(userInput);
            for (Pattern p : regexList) {
                Matcher m = p.matcher(userInput);
                userInput = m.replaceAll(replacement);
            }
            userInput = StringEscapeUtils.escapeHtml4(userInput);
        } else {
            for (Pattern p : regexList) {
                Matcher m = p.matcher(userInput);
                userInput = m.replaceAll(replacement);
            }
        }
        return userInput;
    }

    public static String getBasicAuthorization(String username, String password) {
        String format = username + ":" + password;
        return "Basic " + Base64.getEncoder().encodeToString(format.getBytes(StandardCharsets.UTF_8));
    }

    public static String encrypt(byte[] input, byte[] key, String algoritm, String transformation) throws IllegalBlockSizeException, BadPaddingException, NoSuchAlgorithmException, NoSuchPaddingException, InvalidKeyException {

        SecretKey keySpec = new SecretKeySpec(key, algoritm);

        Cipher encrypt = Cipher.getInstance(transformation);
        encrypt.init(Cipher.ENCRYPT_MODE, keySpec);

        return Base64.getEncoder().encodeToString(encrypt.doFinal(input));
    }

    public static String convertToSHA256Hash(String inputText) throws NoSuchAlgorithmException {
        if (GeneralUtil.isNullOrEmpty(inputText))
            return "";
        MessageDigest digest = MessageDigest.getInstance("SHA-256");
        byte[] hash = digest.digest(inputText.getBytes(Charset.defaultCharset()));
        return Base64.getEncoder().encodeToString(hash);
    }

    public static char[] encodeHex(byte[] data, char[] toDigits) {
        int l = data.length;
        char[] out = new char[l << 1];
        // two characters form the hex value.
        for (int i = 0, j = 0; i < l; i++) {
            out[j++] = toDigits[(0xF0 & data[i]) >>> 4];
            out[j++] = toDigits[0x0F & data[i]];
        }
        return out;
    }

    public static String encodeHexStr(byte[] data) {
        return encodeHexStr(data, false);
    }

    public static String encodeHexStr(byte[] data, boolean toLowerCase) {
        return encodeHexStr(data, toLowerCase ? DIGITS_LOWER : DIGITS_UPPER);
    }

    public static String encodeHexStr(byte[] data, char[] toDigits) {
        return new String(encodeHex(data, toDigits));
    }

    public static String generateRandomNumber(int count) {
        return RandomStringUtils.random(count, "0123456789");
    }
}