/*
 * Copyright   Loy Fu. 付厚俊
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
*/
package com.loy.e.data.permission.service.impl;

import java.io.IOException;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.servlet.http.HttpServletResponse;

import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RestController;

import com.loy.e.common.annotation.Author;
import com.loy.e.common.tree.TreeNode;
import com.loy.e.common.vo.DefaultRespone;
import com.loy.e.core.annotation.ControllerLogExeTime;
import com.loy.e.core.util.TableToExcelUtil;
import com.loy.e.data.permission.api.DataStrategyService;
import com.loy.e.data.permission.data.DataStrategyParam;
import com.loy.e.data.permission.domain.DataStrategyQueryParam;
import com.loy.e.data.permission.domain.entity.DataStrategyEntity;
import com.loy.e.data.permission.repository.DataStrategyRepository;
import com.loy.upm.sys.domain.entity.RoleEntity;
import com.loy.upm.sys.repository.RoleRepository;

@Author(author = "Loy Fu", website = "http://www.17jee.com", contact = "qq群 540553957")
@RestController(value = "dataStrategyService")
@RequestMapping(value = "**/strategy", method = { RequestMethod.POST, RequestMethod.GET })
@Transactional

public class DataStrategyServiceImpl implements DataStrategyService {

	@Autowired
	DataStrategyRepository dataStrategyRepository;
	@Autowired
	RoleRepository roleRepository;

	@RequestMapping(value = "/data/user", method = { RequestMethod.POST })

	public DefaultRespone<List<String>> getDataStrategyByUserId(@RequestBody DataStrategyParam dataStrategyParam) {
		List<DataStrategyEntity> list = dataStrategyRepository.findByUserId(dataStrategyParam.getUserId(),
				dataStrategyParam.getUniqueKey());
		DefaultRespone<List<String>> respone = new DefaultRespone<List<String>>();
		if (list != null && !list.isEmpty()) {
			List<String> result = new ArrayList<String>();
			for (DataStrategyEntity e : list) {
				if (!e.getEffective()) {
					continue;
				}
				String ql = e.getQl();
				if (StringUtils.isNotEmpty(ql)) {
					result.add("(" + ql + ")");
				}
			}
			respone.setData(result);
		}
		return respone;
	}

	@RequestMapping(value = "/user/all", method = { RequestMethod.POST, RequestMethod.GET })
	public List<String> findDataStrategyByUserId(String userId) {
		List<DataStrategyEntity> list = dataStrategyRepository.findByUserId(userId);
		if (list != null && !list.isEmpty()) {
			List<String> result = new ArrayList<String>();
			for (DataStrategyEntity e : list) {
				if (!e.getEffective()) {
					continue;
				}
				result.add(e.getName());

			}
			return result;
		}
		return null;
	}

	@RequestMapping({ "/save" })
	@ControllerLogExeTime(description = "保存权限策略")
	public DataStrategyEntity save(DataStrategyEntity dataStrategyEntity) {
		dataStrategyEntity.setId(null);
		this.dataStrategyRepository.save(dataStrategyEntity);
		return dataStrategyEntity;
	}

	@RequestMapping({ "/update" })
	@ControllerLogExeTime(description = "修改权限策略")
	public void update(DataStrategyEntity dataStrategyEntity) {
		String id = dataStrategyEntity.getId();
		DataStrategyEntity entity = this.dataStrategyRepository.get(id);
		entity.setEffective(dataStrategyEntity.getEffective());
		entity.setName(dataStrategyEntity.getName());
		entity.setDescription(dataStrategyEntity.getDescription());
		entity.setQl(dataStrategyEntity.getQl());
		entity.setUniqueKey(dataStrategyEntity.getUniqueKey());
		this.dataStrategyRepository.save(entity);
	}

	@ControllerLogExeTime(description = "获取权限策略", log = false)
	@RequestMapping({ "/get" })
	public DataStrategyEntity get(String id) {
		DataStrategyEntity dataStrategyEntity = this.dataStrategyRepository.get(id);
		return dataStrategyEntity;
	}

	@ControllerLogExeTime(description = "查看权限策略", log = false)
	@RequestMapping({ "/detail" })
	public DataStrategyEntity detail(String id) {
		DataStrategyEntity dataStrategyEntity = this.dataStrategyRepository.get(id);
		return dataStrategyEntity;
	}

	@RequestMapping(value = "/page")
	public Page<DataStrategyEntity> queryPage(DataStrategyQueryParam dataStrategyQueryParam, Pageable pageable) {
		Page<DataStrategyEntity> page = dataStrategyRepository.queryDataStrategyPage(dataStrategyQueryParam, pageable);
		return page;
	}

	@SuppressWarnings("rawtypes")
	@ControllerLogExeTime(description = "权限策略角色", log = false)
	@RequestMapping("/roles")
	public List<TreeNode> roles(String strategyId) {
		List<RoleEntity> list = roleRepository.findAll();
		List<TreeNode> treeList = new ArrayList<TreeNode>();
		if (list != null && !list.isEmpty()) {
			Set<String> tempSet = new HashSet<String>();
			if (StringUtils.isNotEmpty(strategyId)) {
				List<RoleEntity> roles = dataStrategyRepository.findRolesByStrategyId(strategyId);
				for (RoleEntity r : roles) {
					tempSet.add(r.getId());
				}
			}
			for (RoleEntity role : list) {
				TreeNode treeNode = new TreeNode();
				String roleId = role.getId();
				treeNode.setId(roleId);
				treeNode.setText(role.getName());
				treeNode.setType("item");
				if (tempSet.contains(roleId)) {
					treeNode.setSelected(true);
				}
				treeList.add(treeNode);
			}
		}
		return treeList;
	}

	@SuppressWarnings("rawtypes")
	@ControllerLogExeTime(description = "角色权限策略", log = false)
	@RequestMapping("/strategies")
	public List<TreeNode> strategies(String roleId) {
		List<DataStrategyEntity> list = dataStrategyRepository.findAll();
		List<TreeNode> treeList = new ArrayList<TreeNode>();
		if (list != null && !list.isEmpty()) {
			Set<String> tempSet = new HashSet<String>();

			List<DataStrategyEntity> strategyOfRole = dataStrategyRepository.findByRoleId(roleId);
			for (DataStrategyEntity r : strategyOfRole) {
				tempSet.add(r.getId());
			}

			for (DataStrategyEntity strategy : list) {
				if (!strategy.getEffective()) {
					continue;
				}
				TreeNode treeNode = new TreeNode();
				String strategyId = strategy.getId();
				treeNode.setId(strategyId);
				treeNode.setText(strategy.getName());
				treeNode.setType("item");
				if (tempSet.contains(strategyId)) {
					treeNode.setSelected(true);
				}
				treeList.add(treeNode);
			}
		}
		return treeList;
	}

	@ControllerLogExeTime(description = "数据授权于角色")
	@RequestMapping(value = "/authority/role", method = { RequestMethod.POST })
	public void authorityToRole(String strategyId, String roleIds) {
		DataStrategyEntity dataStrategyEntity = dataStrategyRepository.get(strategyId);
		Set<RoleEntity> resources = new HashSet<RoleEntity>();

		if (StringUtils.isNotEmpty(roleIds)) {
			String[] rIds = roleIds.split(",");
			for (String roleId : rIds) {
				RoleEntity roleEntity = this.roleRepository.get(roleId);
				if (roleEntity != null) {
					resources.add(roleEntity);
				}
			}
		}
		dataStrategyEntity.setRoles(resources);
		dataStrategyRepository.save(dataStrategyEntity);
	}

	@ControllerLogExeTime(description = "角色授数据权限")
	@RequestMapping(value = "/role/data/authority", method = { RequestMethod.POST })
	public void roleAuthority(String roleId, String strategyIds) {
		RoleEntity roleEntity = this.roleRepository.get(roleId);
		List<DataStrategyEntity> roleHasStrategy = dataStrategyRepository.findByRoleId(roleId);
		if (roleHasStrategy != null) {
			for (DataStrategyEntity dataStrategyEntity : roleHasStrategy) {
				dataStrategyEntity.getRoles().remove(roleEntity);
			}
		}
		if (StringUtils.isNotEmpty(strategyIds)) {
			String[] rIds = strategyIds.split(",");
			for (String strategyId : rIds) {
				DataStrategyEntity dataStrategyEntity = this.dataStrategyRepository.get(strategyId);
				if (dataStrategyEntity != null) {
					dataStrategyEntity.getRoles().add(roleEntity);
				}
			}
		}

	}

	@ControllerLogExeTime(description = "删除权限策略")
	@RequestMapping({ "/del" })
	public void del(String id) {
		if (StringUtils.isNotEmpty(id)) {
			String[] idsArr = id.split(",");
			if (idsArr != null) {
				for (String idd : idsArr) {
					DataStrategyEntity dataStrategyEntity = (DataStrategyEntity) this.dataStrategyRepository.get(idd);
					if (dataStrategyEntity != null) {
						this.dataStrategyRepository.delete(dataStrategyEntity);
					}
				}
			}
		}
	}

	@RequestMapping(value = { "/excel" }, method = { RequestMethod.POST })
	@ControllerLogExeTime(description = "导出权限策略", log = false)
	public void excel(String html, HttpServletResponse response) throws IOException {
		response.setContentType("application/msexcel;charset=UTF-8");
		response.addHeader("Content-Disposition", "attachment;filename=strategies.xls");
		OutputStream out = response.getOutputStream();
		TableToExcelUtil.createExcelFormTable("company", html, 1, out);
		out.flush();
		out.close();
	}

}
