package com.aallam.openai.api.chat

import com.aallam.openai.api.BetaOpenAI
import com.aallam.openai.api.OpenAIDsl
import com.aallam.openai.api.model.ModelId
import kotlinx.serialization.SerialName
import kotlinx.serialization.Serializable

/**
 * Creates a completion for the chat message.
 */
@BetaOpenAI
@Serializable
public class ChatCompletionRequest(
    /**
     * ID of the model to use.
     */
    @SerialName("model") public val model: ModelId,

    /**
     * The messages to generate chat completions for.
     */
    @SerialName("messages") public val messages: List<ChatMessage>,

    /**
     * What sampling temperature to use, between 0 and 2. Higher values like 0.8 will make the output more random,
     * while lower values like 0.2 will make it more focused and deterministic.
     *
     * We generally recommend altering this or [topP] but not both.
     */
    @SerialName("temperature") public val temperature: Double? = null,

    /**
     * An alternative to sampling with temperature, called nucleus sampling, where the model considers the results
     * of the tokens with top_p probability mass. So 0.1 means only the tokens comprising the top 10% probability mass
     * are considered.
     *
     * We generally recommend altering this or [temperature] but not both.
     */
    @SerialName("top_p") public val topP: Double? = null,

    /**
     * How many chat completion choices to generate for each input message.
     */
    @SerialName("n") public val n: Int? = null,

    /**
     * Up to 4 sequences where the API will stop generating further tokens.
     */
    @SerialName("stop") public val stop: List<String>? = null,

    /**
     * The maximum number of tokens allowed for the generated answer. By default, the number of tokens the model can
     * return will be (4096 - prompt tokens).
     */
    @SerialName("max_tokens") public val maxTokens: Int? = null,

    /**
     * Number between -2.0 and 2.0. Positive values penalize new tokens based on whether they appear in the text so far,
     * increasing the model's likelihood to talk about new topics.
     *
     * [Read more](https://platform.openai.com/docs/api-reference/parameter-details)
     */
    @SerialName("presence_penalty") public val presencePenalty: Double? = null,

    /**
     * Number between -2.0 and 2.0. Positive values penalize new tokens based on their existing frequency in the text so
     * far, decreasing the model's likelihood to repeat the same line verbatim.
     *
     * [Read more](https://platform.openai.com/docs/api-reference/parameter-details)
     */
    @SerialName("frequency_penalty") public val frequencyPenalty: Double? = null,

    /**
     * Modify the likelihood of specified tokens appearing in the completion.
     *
     * Accepts a json object that maps tokens (specified by their token ID in the tokenizer) to an associated bias value
     * from -100 to 100. Mathematically, the bias is added to the logits generated by the model prior to sampling.
     * The exact effect will vary per model, but values between -1 and 1 should decrease or increase likelihood of
     * selection; values like -100 or 100 should result in a ban or exclusive selection of the relevant token.
     */
    @SerialName("logit_bias") public val logitBias: Map<String, Int>? = null,

    /**
     * A unique identifier representing your end-user, which can help OpenAI to monitor and detect abuse.
     */
    @SerialName("user") public val user: String? = null,

    /**
     * A list of functions the model may generate JSON inputs for.
     */
    @SerialName("functions") public val functions: List<ChatCompletionFunction>? = null,

    /**
     * Controls how the model responds to function calls. [FunctionMode.None] means the model does not call a function,
     * and responds to the end-user.
     * [FunctionMode.Auto] means the model can pick between an end-user or calling a function.
     * Specifying a particular function via [FunctionMode.Named] forces the model to call that function.
     * [FunctionMode.None] is the default when no functions are present.
     * [FunctionMode.Auto] is the default if functions are present.
     */
    @SerialName("function_call") public val functionCall: FunctionMode? = null,
)

/**
 * The messages to generate chat completions for.
 */
@BetaOpenAI
public fun chatCompletionRequest(block: ChatCompletionRequestBuilder.() -> Unit): ChatCompletionRequest =
    ChatCompletionRequestBuilder().apply(block).build()

/**
 * Creates a completion for the chat message.
 */
@BetaOpenAI
@OpenAIDsl
public class ChatCompletionRequestBuilder {
    /**
     * ID of the model to use.
     */
    public var model: ModelId? = null

    /**
     * The messages to generate chat completions for.
     */
    public var messages: List<ChatMessage>? = null

    /**
     * What sampling temperature to use, between 0 and 2. Higher values like 0.8 will make the output more random,
     * while lower values like 0.2 will make it more focused and deterministic.
     *
     * We generally recommend altering this or [topP] but not both.
     */
    public var temperature: Double? = null

    /**
     * An alternative to sampling with temperature, called nucleus sampling, where the model considers the results
     * of the tokens with top_p probability mass. So 0.1 means only the tokens comprising the top 10% probability mass
     * are considered.
     *
     * We generally recommend altering this or [temperature] but not both.
     */
    public var topP: Double? = null

    /**
     * How many chat completion choices to generate for each input message.
     */
    public var n: Int? = null

    /**
     * Up to 4 sequences where the API will stop generating further tokens.
     */
    public var stop: List<String>? = null

    /**
     * The maximum number of tokens allowed for the generated answer. By default, the number of tokens the model can
     * return will be (4096 - prompt tokens).
     */
    public var maxTokens: Int? = null

    /**
     * Number between -2.0 and 2.0. Positive values penalize new tokens based on whether they appear in the text so far,
     * increasing the model's likelihood to talk about new topics.
     *
     * [Read more](https://platform.openai.com/docs/api-reference/parameter-details)
     */
    public var presencePenalty: Double? = null

    /**
     * Number between -2.0 and 2.0. Positive values penalize new tokens based on their existing frequency in the text so
     * far, decreasing the model's likelihood to repeat the same line verbatim.
     *
     * [Read more](https://platform.openai.com/docs/api-reference/parameter-details)
     */
    public var frequencyPenalty: Double? = null

    /**
     * Modify the likelihood of specified tokens appearing in the completion.
     *
     * Accepts a json object that maps tokens (specified by their token ID in the tokenizer) to an associated bias value
     * from -100 to 100. Mathematically, the bias is added to the logits generated by the model prior to sampling.
     * The exact effect will vary per model, but values between -1 and 1 should decrease or increase likelihood of
     * selection; values like -100 or 100 should result in a ban or exclusive selection of the relevant token.
     */
    public var logitBias: Map<String, Int>? = null

    /**
     * A unique identifier representing your end-user, which can help OpenAI to monitor and detect abuse.
     */
    public var user: String? = null

    /**
     * A list of functions the model may generate JSON inputs for.
     */
    public var functions: List<ChatCompletionFunction>? = null

    /**
     * Controls how the model responds to function calls. [FunctionMode.None] means the model does not call a function,
     * and responds to the end-user.
     * [FunctionMode.Auto] means the model can pick between an end-user or calling a function.
     * Specifying a particular function via [FunctionMode.Named] forces the model to call that function.
     * [FunctionMode.None] is the default when no functions are present.
     * [FunctionMode.Auto] is the default if functions are present.
     */
    public var functionCall: FunctionMode? = null

    /**
     * The messages to generate chat completions for.
     */
    public fun messages(block: ChatMessagesBuilder.() -> Unit) {
        messages = ChatMessagesBuilder().apply(block).messages
    }

    /**
     * A list of functions the model may generate JSON inputs for.
     */
    public fun functions(block: FunctionsBuilder.() -> Unit) {
        functions = FunctionsBuilder().apply(block).functions
    }

    /**
     * Builder of [ChatCompletionRequest] instances.
     */
    public fun build(): ChatCompletionRequest = ChatCompletionRequest(
        model = requireNotNull(model) { "model is required" },
        messages = requireNotNull(messages) { "messages is required" },
        temperature = temperature,
        topP = topP,
        n = n,
        stop = stop,
        maxTokens = maxTokens,
        presencePenalty = presencePenalty,
        frequencyPenalty = frequencyPenalty,
        logitBias = logitBias,
        user = user,
        functions = functions,
        functionCall = functionCall,
    )
}

/**
 * Creates a list of [ChatMessage].
 */
@BetaOpenAI
public class ChatMessagesBuilder {
    internal val messages = mutableListOf<ChatMessage>()

    /**
     * Creates a [ChatMessage] instance.
     */
    public fun message(block: ChatMessageBuilder.() -> Unit) {
        messages += ChatMessageBuilder().apply(block).build()
    }
}

/**
 * Creates a list of [ChatCompletionFunction].
 */
@BetaOpenAI
public class FunctionsBuilder {
    internal val functions = mutableListOf<ChatCompletionFunction>()

    /**
     * Creates a [ChatMessage] instance.
     */
    public fun function(block: ChatCompletionFunctionBuilder.() -> Unit) {
        functions += ChatCompletionFunctionBuilder().apply(block).build()
    }
}
