/**
 * Licensed to Abiquo Holdings S.L. (Abiquo) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package com.abiquo.model.adapter;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.ObjectCodec;
import com.fasterxml.jackson.databind.DeserializationContext;
import com.fasterxml.jackson.databind.JsonDeserializer;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.ArrayNode;

public class MetadataJsonAdapter extends JsonDeserializer<Map<String, Object>>
{
    private static final String NULL = "null";

    @Override
    public Map<String, Object> deserialize(final JsonParser jp, final DeserializationContext ctxt)
        throws IOException, JsonProcessingException
    {
        ObjectCodec oc = jp.getCodec();
        JsonNode node = oc.readTree(jp);
        Map<String, Object> metadata = unmarshal(node.fields(), new HashMap<String, Object>());
        Map<String, Object> json = new HashMap<>();
        json.put("metadata", metadata);
        return json;
    }

    public Map<String, Object> unmarshal(final Iterator<Entry<String, JsonNode>> nodes,
        final Map<String, Object> map)
    {
        while (nodes.hasNext())
        {
            final Entry<String, JsonNode> entry = nodes.next();
            if (entry.getValue().isValueNode())
            {
                if (entry.getValue().asText() == null || NULL.equals(entry.getValue().asText())
                    || "".equals(entry.getValue().asText()))
                {
                    map.put(entry.getKey(), null);
                    continue;
                }
                map.put(entry.getKey(), entry.getValue().asText());
                continue;
            }
            if (entry.getValue().isObject())
            {
                map.put(entry.getKey(),
                    unmarshal(entry.getValue().fields(), new HashMap<String, Object>()));
                continue;
            }
            if (entry.getValue().isArray())
            {
                map.put(entry.getKey(), toList((ArrayNode) entry.getValue()));
                continue;
            }
        }
        return map;
    }

    private List<Object> toList(final ArrayNode nodes)
    {
        List<Object> elements = new ArrayList<>();

        int size = nodes.size();
        if (size > 0)
        {
            for (int i = 0; i < size; i++) // node do not return java.lang.Iterable
            {
                JsonNode node = nodes.get(i);
                if (node.isValueNode())
                {
                    elements.add(node.asText());
                }
                else if (node.isNull())
                {
                    elements.add(NULL);
                }
                else
                {
                    elements.add(unmarshal(node.fields(), new HashMap<String, Object>()));
                }
            }
        }
        return elements;
    }
}
