/**
 * Licensed to Abiquo Holdings S.L. (Abiquo) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package com.abiquo.model.adapter;

import static com.abiquo.model.util.StringSplitter.commaSplit;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;

import javax.xml.bind.annotation.adapters.XmlAdapter;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;

import com.abiquo.server.core.cloud.HypervisorTypeDto;

/**
 * XmlAdapter implementation to generate the XML representation of a Map<String, String>. It is used
 * to un/marshal the constraints map of an hypervisor type.
 * 
 * @see HypervisorTypeDto#getConstraints()
 */
public class PluginConstraintsXmlAdapter extends XmlAdapter<Element, Map<String, Object>>
{
    private static final String OPERATIONS = "operations";

    private DocumentBuilder documentBuilder;

    protected String getRootElementName()
    {
        return "constraints";
    }

    private DocumentBuilder getDocumentBuilder() throws Exception
    {
        // Lazy load the DocumentBuilder as it is not used for unmarshalling.
        if (null == documentBuilder)
        {
            DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
            documentBuilder = dbf.newDocumentBuilder();
        }
        return documentBuilder;
    }

    @Override
    public Element marshal(final Map<String, Object> map) throws Exception
    {
        DocumentBuilder db = getDocumentBuilder();
        Document document = db.newDocument();
        Element rootElement = document.createElement(getRootElementName());
        marshalMap(map, rootElement, document);
        document.appendChild(rootElement);
        return rootElement;
    }

    /**
     * Convert the given map in a child element of the given root element. Each key is a child
     * element with his value as text content.
     * 
     * @param map the map to marshal
     * @param rootElement the root element
     * @param document the XML document that we are building
     * @return the given root element with the new child element added
     */
    private Element marshalMap(final Map<String, Object> map, final Element rootElement,
        final Document document)
    {
        for (Entry<String, Object> entry : map.entrySet())
        {
            Element element = document.createElement(entry.getKey().toLowerCase());
            Object value = entry.getValue();
            element.setTextContent(value == null ? "" : value.toString());
            rootElement.appendChild(element);
        }

        return rootElement;
    }

    @Override
    public Map<String, Object> unmarshal(final Element element) throws Exception
    {
        if (null == element)
        {
            return null;
        }

        return unmarshalMap(element.getFirstChild(), new HashMap<String, Object>(),
            element.getNodeName());
    }

    /**
     * Recursive function to traverse the given XML node and fill the given map
     * 
     * @param node the node to traverse
     * @param map the map to fill
     * @return the given map
     */
    private Map<String, Object> unmarshalMap(final Node node, final Map<String, Object> map,
        final String rootName)
    {
        if (node.getNodeType() == Node.ELEMENT_NODE && node.getFirstChild() == null)
        {
            // <node_element />
            map.put(node.getNodeName(), OPERATIONS.equals(rootName) ? new ArrayList<String>()
                : null);
        }
        else if (node.getNodeType() == Node.ELEMENT_NODE
            && node.getFirstChild().getNodeType() == Node.TEXT_NODE)
        {
            // <node_element>text_node</node_element>
            map.put(node.getNodeName(), node.getFirstChild().getNodeValue());
        }
        else if (node.getNodeType() == Node.ELEMENT_NODE)
        {
            if (node.getFirstChild().getNodeType() == Node.ELEMENT_NODE)
            {
                map.put(node.getNodeName(),
                    extractInner(node.getFirstChild(), new HashMap<String, Object>()));
            }
        }

        if (node.getNextSibling() != null)
        {
            // <node_element>text_node</node_element><next_sibling>...
            unmarshalMap(node.getNextSibling(), map, rootName);
        }
        return map;
    }

    private Map<String, Object> extractInner(final Node node, final Map<String, Object> map)
    {
        if (node.getFirstChild() == null)
        {
            map.put(node.getNodeName(), new ArrayList<String>());
        }
        else if (node.getFirstChild().getNodeType() == Node.TEXT_NODE)
        {
            map.put(node.getNodeName(), commaSplit(node.getFirstChild().getTextContent()));
        }

        if (node.getNextSibling() != null)
        {
            extractInner(node.getNextSibling(), map);
        }
        return map;
    }

}
