/**
 * Licensed to Abiquo Holdings S.L. (Abiquo) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package com.abiquo.model.adapter;

import static com.abiquo.model.util.StringSplitter.commaSplit;
import static java.util.Objects.requireNonNull;

import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import javax.xml.bind.annotation.adapters.XmlAdapter;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;

import com.abiquo.server.core.cloud.HypervisorTypeDto;

/**
 * XmlAdapter implementation to generate the XML representation of a Map<String, String>. It is used
 * to un/marshal the constraints map of an hypervisor type.
 * 
 * @see HypervisorTypeDto#getConstraints()
 */
public class PluginOperationsXmlAdapter extends
    XmlAdapter<Element, Map<String, Map<String, List<String>>>>
{
    private DocumentBuilder documentBuilder;

    protected String getRootElementName()
    {
        return "operations";
    }

    private DocumentBuilder getDocumentBuilder() throws Exception
    {
        // Lazy load the DocumentBuilder as it is not used for unmarshalling.
        if (null == documentBuilder)
        {
            DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
            documentBuilder = dbf.newDocumentBuilder();
        }
        return documentBuilder;
    }

    @Override
    public Element marshal(final Map<String, Map<String, List<String>>> map) throws Exception
    {
        DocumentBuilder db = getDocumentBuilder();
        Document document = db.newDocument();
        Element rootElement = document.createElement(getRootElementName());
        marshalMap(map, rootElement, document);
        document.appendChild(rootElement);
        return rootElement;
    }

    /**
     * Convert the given map in a child element of the given root element. Each key is a child
     * element with his value as text content.
     * 
     * @param map the map to marshal
     * @param rootElement the root element
     * @param document the XML document that we are building
     * @return the given root element with the new child element added
     */
    private Element marshalMap(final Map<String, Map<String, List<String>>> map,
        final Element rootElement, final Document document)
    {
        for (Entry<String, Map<String, List<String>>> outer : map.entrySet())
        {
            Element element = document.createElement(outer.getKey().toLowerCase());
            for (Entry<String, List<String>> inner : outer.getValue().entrySet())
            {
                Element innerElement = document.createElement(inner.getKey());
                StringBuilder content = new StringBuilder("");
                List<String> value = inner.getValue();
                if (value != null)
                {
                    content.append(join(value));
                }
                innerElement.setTextContent(content.toString());
                element.appendChild(innerElement);
            }
            rootElement.appendChild(element);
        }

        return rootElement;
    }

    private String join(final List<String> value)
    {
        requireNonNull(value, "values is null");
        Iterator<String> parts = value.iterator();
        StringBuilder join = new StringBuilder("");
        if (parts.hasNext())
        {
            join.append(parts.next());
            while (parts.hasNext())
            {
                join.append(",");
                join.append(parts.next());
            }
        }
        return join.toString();
    }

    @Override
    public Map<String, Map<String, List<String>>> unmarshal(final Element element) throws Exception
    {
        if (null == element)
        {
            return null;
        }

        return unmarshalMap(element.getFirstChild(),
            new HashMap<String, Map<String, List<String>>>());
    }

    /**
     * Recursive function to traverse the given XML node and fill the given map
     * 
     * @param node the node to traverse
     * @param map the map to fill
     * @return the given map
     */
    private Map<String, Map<String, List<String>>> unmarshalMap(final Node node,
        final Map<String, Map<String, List<String>>> map)
    {
        if (node.getNodeType() == Node.ELEMENT_NODE && node.getFirstChild() == null)
        {
            // <node_element />
            map.put(node.getNodeName(), null);
        }
        else if (node.getNodeType() == Node.ELEMENT_NODE)
        {
            Map<String, List<String>> inner = extractInner(node.getFirstChild());
            map.put(node.getNodeName(), inner);
        }

        if (node.getNextSibling() != null)
        {
            // <node_element>text_node</node_element><next_sibling>...
            unmarshalMap(node.getNextSibling(), map);
        }
        return map;
    }

    private Map<String, List<String>> extractInner(final Node node)
    {
        if (node.getFirstChild() != null)
        {
            extractInner(node.getFirstChild());

        }
        Map<String, List<String>> inner = new HashMap<>();
        if (node.getFirstChild().getNodeType() == Node.TEXT_NODE)
        {
            // <node_element>text_node1</node_element>
            inner.put(node.getNodeName(), commaSplit(node.getFirstChild().getNodeValue()));
        }
        return inner;
    }
}
