/**
 * Licensed to Abiquo Holdings S.L. (Abiquo) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package com.abiquo.model.enumerator;

import java.util.LinkedList;
import java.util.List;

import javax.xml.bind.annotation.XmlEnum;

/**
 * Operating system list (from OVF CIM_OperatingSystem supported systems)
 * <p>
 * 
 * @see <a
 *      href="http://dmtf.org/sites/default/files/cim/cim_schema_v2280/cim_schema_2.28.0Final-Doc.zip"
 *      /> <a
 *      https://raw.github.com/jclouds/jclouds/master/compute/src/main/java/org/jclouds/cim/OSType
 *      .java" />
 */
@XmlEnum
public enum OSType
{
    OTHER(1, "Other", false), //
    MACOS(2, "MACOS", false), //

    SOLARIS(29, "Solaris", false), //

    LINUX(36, "LINUX", false), //

    FREEBSD(42, "FreeBSD", false), //
    NETBSD(43, "NetBSD", false), //

    OPENBSD(65, "OpenBSD", false), //
    NOT_APPLICABLE(66, "Not Applicable", false), //
    WINDOWS(67, "Windows", true), //

    WINDOWS_SERVER_2003(69, "Microsoft Windows Server 2003", false), //
    WINDOWS_SERVER_2003_64(70, "Microsoft Windows Server 2003 64-Bit", true), //

    WINDOWS_SERVER_2008(76, "Microsoft Windows Server 2008", false), //
    WINDOWS_SERVER_2008_64(77, "Microsoft Windows Server 2008 64-Bit", true), //
    FREEBSD_64(78, "FreeBSD 64-Bit", true), //
    RHEL(79, "RedHat Enterprise Linux", false), //
    RHEL_64(80, "RedHat Enterprise Linux 64-Bit", true), //
    SOLARIS_64(81, "Solaris 64-Bit", true), //
    SUSE(82, "SUSE", false), //
    SUSE_64(83, "SUSE 64-Bit", true), //
    SLES(84, "SLES", false), //
    SLES_64(85, "SLES 64-Bit", true), //
    NOVELL_OES(86, "Novell OES", true), //

    MANDRIVA(89, "Mandriva", false), //
    MANDRIVA_64(90, "Mandriva 64-Bit", true), //
    TURBOLINUX(91, "TurboLinux", false), //
    TURBOLINUX_64(92, "TurboLinux 64-Bit", true), //
    UBUNTU(93, "Ubuntu", false), //
    UBUNTU_64(94, "Ubuntu 64-Bit", true), //
    DEBIAN(95, "Debian", false), //
    DEBIAN_64(96, "Debian 64-Bit", false), //
    LINUX_2_4(97, "Linux 2.4.x", false), //
    LINUX_2_4_64(98, "Linux 2.4.x 64-Bit", true), //
    LINUX_2_6(99, "Linux 2.6.x", false), //
    LINUX_2_6_64(100, "Linux 2.6.x 64-Bit", true), //
    LINUX_64(101, "Linux 64-Bit", true), //
    OTHER_64(102, "Other 64-Bit", true), //
    WINDOWS_SERVER_2008_R2(103, "Microsoft Windows Server 2008 R2", true), //
    ESXI(104, "VMware ESXi", true), //
    WINDOWS_7(105, "Microsoft Windows 7", false), //
    CENTOS(106, "CentOS 32-bit", false), //
    CENTOS_64(107, "CentOS 64-bit", true), //
    ORACLE_ENTERPRISE_LINUX(108, "Oracle Enterprise Linux 32-bit", false), //
    ORACLE_ENTERPRISE_LINUX_64(109, "Oracle Enterprise Linux 64-bit", true), //
    ECOMSTATION_32(110, "eComStation 32-bit", false), //
    WINDOWS_SERVER_2011(111, "Microsoft Windows Server 2011", true), //

    WINDOWS_SERVER_2012(113, "Microsoft Windows Server 2012", true), //
    WINDOWS_8(114, "Microsoft 8", false), //
    WINDOWS_8_64(115, "Microsoft 8 64-bit", true), //
    UNRECOGNIZED(Integer.MAX_VALUE, "UNRECOGNIZED", true);

    public final static List<OSType> arch64 = computeArch(true);

    public final static List<OSType> arch32 = computeArch(false);

    public static List<OSType> computeArch(final boolean arch64)
    {
        List<OSType> filtered = new LinkedList<OSType>();
        for (OSType value : values())
        {
            if (arch64 == value.is64Bit())
            {
                filtered.add(value);
            }
        }
        return filtered;
    }

    /**
     * Operating system type as indicated in the CIM_OperatingSystem specification (used in OVF
     * representation).
     */
    final int code;

    /** Human readable operating system description */
    final String description;

    /** Check if OS is 64 bit based (otherwise assume 32) */
    final boolean is64b;

    OSType(final int code, final String description, final boolean is64b)
    {
        this.code = code;
        this.description = description;
        this.is64b = is64b;
    }

    /** return UNRECOGNIZED if not found */
    public static OSType fromCode(final int code)
    {
        for (OSType value : values())
        {
            if (code == value.getCode())
            {
                return value;
            }
        }
        return UNRECOGNIZED;
    }

    public int getCode()
    {
        return code;
    }

    public String getDescription()
    {
        return description;
    }

    public boolean is64Bit()
    {
        return is64b;
    }
}
