/**
 * Licensed to Abiquo Holdings S.L. (Abiquo) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package com.abiquo.model.enumerator;

/**
 * Abiquo community edition
 * cloud management application for hybrid clouds
 * Copyright (C) 2008-2010 - Abiquo Holdings S.L.
 *
 * This application is free software; you can redistribute it and/or
 * modify it under the terms of the GNU LESSER GENERAL PUBLIC
 * LICENSE as published by the Free Software Foundation under
 * version 3 of the License
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * LESSER GENERAL PUBLIC LICENSE v.3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

import java.net.URI;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;

public enum RemoteServiceType
{
    VIRTUAL_FACTORY("Virtualization Manager", "virtualfactory", "http://", 80), //
    STORAGE_SYSTEM_MONITOR("Storage Manager", "ssm", "http://", 80), //
    VIRTUAL_SYSTEM_MONITOR("Monitor Manager", "vsm", "http://", 80), //
    NODE_COLLECTOR("Discovery Manager", "nodecollector", "http://", 80), //
    DHCP_SERVICE("DHCP Service", "dhcp", "omapi://", 7911), //
    BPM_SERVICE("Business Process Manager", "bpm-async", "http://", 80), //
    APPLIANCE_MANAGER("Appliance Manager", "am", "http://", 80), //
    DHCPv6("DHCPv6", "dhcpv6", "omapi://", 7911), //
    CLOUD_PROVIDER_PROXY("Cloud provider proxy", "cpp", "http://", 80);

    public static List<RemoteServiceType> CLOUD_PROVIDER_REMOTE_SERVICES = Collections
        .unmodifiableList(Arrays.asList(VIRTUAL_FACTORY, VIRTUAL_SYSTEM_MONITOR, NODE_COLLECTOR,
            CLOUD_PROVIDER_PROXY));

    private String name;

    private String serviceMapping;

    private String defaultProtocol;

    private Integer defaultPort;

    public Integer getDefaultPort()
    {
        return defaultPort;
    }

    public String getDefaultProtocol()
    {
        return defaultProtocol;
    }

    public String getName()
    {
        return name;
    }

    public String getServiceMapping()
    {
        return serviceMapping;
    }

    RemoteServiceType(final String name, final String serviceMapping, final String defaultProtocol,
        final Integer defaultPort)
    {
        this.name = name;
        this.serviceMapping = serviceMapping;
        this.defaultProtocol = defaultProtocol;
        this.defaultPort = defaultPort;
    }

    RemoteServiceType(final String name, final String serviceMapping)
    {
        this.name = name;
        this.serviceMapping = serviceMapping;
    }

    public boolean canBeChecked()
    {
        return this != DHCP_SERVICE && this != DHCPv6;
    }

    public boolean checkUniqueness()
    {
        return this == APPLIANCE_MANAGER || this == BPM_SERVICE || this == VIRTUAL_FACTORY;
    }

    public boolean checkDatacenterId()
    {
        return this == BPM_SERVICE || this == VIRTUAL_FACTORY;
    }

    public String fixUri(final URI uri)
    {
        String protocol = uri.getScheme();
        String domainName = uri.getHost();
        Integer port = uri.getPort();
        String path = uri.getPath();

        String domainHost = domainName + (port != null ? ":" + port : "");

        String fullURL = fixProtocol(protocol) + domainHost;

        if (path != null && !path.isEmpty())
        {
            fullURL = appendPathToBaseUri(fullURL, path);
        }

        return fullURL;
    }

    public static RemoteServiceType valueFromName(final String name)
    {
        for (RemoteServiceType element : RemoteServiceType.values())
        {
            if (name.replaceAll("_", "").equalsIgnoreCase(element.name().replaceAll("_", "")))
            {
                return element;
            }
        }

        return valueOf(name);
    }

    private String fixProtocol(String protocol)
    {
        if (!protocol.endsWith("://"))
        {
            protocol += "://";
        }
        return protocol;
    }

    public String getIdentifierName()
    {

        return this.name().replaceAll("_", "");
    }

    public static boolean isCloudProviderRemoteService(final RemoteServiceType type)
    {
        return CLOUD_PROVIDER_REMOTE_SERVICES.contains(type);
    }

    // from wink UriHelper
    private static String appendPathToBaseUri(String baseUri, String path)
    {
        if (baseUri == null)
        {
            throw new NullPointerException("baseUri");
        }
        if (path == null || path.length() == 0)
        {
            return baseUri;
        }
        boolean endSlash = baseUri.endsWith("/");
        boolean startSlash = path.charAt(0) == '/';
        if (startSlash && endSlash)
        {
            return baseUri + path.substring(1);
        }
        else if (startSlash || endSlash)
        {
            return baseUri + path;
        }
        else
        {
            return baseUri + '/' + path;
        }
    }

}
