/**
 * Licensed to Abiquo Holdings S.L. (Abiquo) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package com.abiquo.model.transport;

import java.io.Serializable;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;

import javax.xml.bind.annotation.XmlElement;
import javax.xml.bind.annotation.XmlTransient;

import com.abiquo.model.rest.RESTLink;
import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonProperty;

public abstract class SingleResourceTransportDto implements Serializable
{
    public static final String API_VERSION = "3.1";

    public static final String API_VERSION_30 = "3.0";

    public static final String API_VERSION_29 = "2.9";

    public static final String API_VERSION_28 = "2.8";

    public static final String API_VERSION_27 = "2.7";

    public static final String API_VERSION_26 = "2.6";

    public static final String API_VERSION_24 = "2.4";

    public static final String API_VERSION_23 = "2.3";

    public static final String API_VERSION_22 = "2.2";

    public static final String API_VERSION_20 = "2.0";

    public static final String APPLICATION = "application";

    public static final String VERSION_PARAM = "; version=";

    public static final String XML = "xml";

    public static final String JSON = "json";

    private static final long serialVersionUID = 1L;

    protected RESTLink editLink;

    protected List<RESTLink> links;

    @XmlElement(name = "link")
    @JsonProperty("links")
    public List<RESTLink> getLinks()
    {
        if (links == null)
        {
            links = new ArrayList<RESTLink>();
        }
        return links;
    }

    public void setLinks(final List<RESTLink> links)
    {
        this.links = links;
    }

    public void addLink(final RESTLink link)
    {
        if (this.links == null)
        {
            this.links = new ArrayList<RESTLink>();
        }
        this.links.add(link);
    }

    public void addLinks(final List<RESTLink> links)
    {
        if (this.links == null)
        {
            this.links = new ArrayList<RESTLink>();
        }
        this.links.addAll(links);
    }

    @JsonIgnore
    public RESTLink getEditLink()
    {
        if (editLink == null)
        {
            editLink = searchLink("edit");
        }
        return editLink;
    }

    public void addEditLink(final RESTLink edit)
    {
        editLink = edit;
        RESTLink currentEdit = searchLink("edit");
        if (currentEdit != null)
        {
            links.remove(currentEdit);
        }
        links.add(editLink);
    }

    /**
     * Returns a {@link RESTLink} with the relation attribute value or null if no such link is
     * found. The comparison is <b>case sensitive</b>. If rel is null a null is returned.
     * 
     * @see #getLinks()
     * @param rel relation attribute value of the link that we are looking for.
     * @return link if found, <code>null</code> otherwise.
     */
    public RESTLink searchLink(final String rel)
    {
        for (RESTLink link : getLinks())
        {
            if (rel != null && rel.equals(link.getRel()))
            {
                return link;
            }
        }
        return null;
    }

    /**
     * Returns a {@link List} of {@link RESTLink} with the relation attribute value or an empty one
     * if no such link is found. If rel is null result an empty list.
     * 
     * @see #getLinks()
     * @param rel relation attribute value of the link that we are looking for.
     * @return list with the links if found, or an empty otherwise.
     */
    public List<RESTLink> searchLinks(final String rel)
    {
        List<RESTLink> links = new ArrayList<RESTLink>();
        if (rel == null)
        {
            return links;
        }
        for (RESTLink link : getLinks())
        {
            if (rel.equals(link.getRel()))
            {
                links.add(link);
            }
        }
        return links;
    }

    public RESTLink searchLink(final String rel, final String title)
    {
        if (getLinks() == null)
        {
            setLinks(new ArrayList<RESTLink>());
        }

        for (RESTLink link : getLinks())
        {
            if (link.getRel().equals(rel) && link.getTitle().equals(title))
            {
                return link;
            }
        }
        return null;
    }

    public List<RESTLink> searchLinksByHref(final String href)
    {
        if (getLinks() == null)
        {
            setLinks(new ArrayList<RESTLink>());
        }

        List<RESTLink> links = new ArrayList<RESTLink>();
        for (RESTLink link : getLinks())
        {
            if (sameHref(link.getHref(), href))
            {
                links.add(link);
            }
        }

        return links;
    }

    private boolean sameHref(final String one, final String other)
    {
        try
        {
            final URI first = new URI(one);
            final URI second = new URI(other);

            final HashSet<String> firstQueryParams = new HashSet<String>();
            final String firstQueryStr = nullToEmpty(first.getQuery());
            firstQueryParams.addAll(Arrays.asList(firstQueryStr.split("&")));

            final HashSet<String> secondQueryParams = new HashSet<String>();
            final String secondQuerStr = nullToEmpty(second.getQuery());
            secondQueryParams.addAll(Arrays.asList(secondQuerStr.split("&")));

            return nullToEmpty(first.getScheme()).equals(nullToEmpty(second.getScheme())) //
                && nullToEmpty(first.getHost()).equals(nullToEmpty(second.getHost())) //
                && nullToEmpty(first.getPath()).equals(nullToEmpty(second.getPath())) //
                && firstQueryParams.size() == secondQueryParams.size() //
                && firstQueryParams.containsAll(secondQueryParams);
        }
        catch (URISyntaxException e)
        {
            return false;
        }
    }

    private String nullToEmpty(final String value)
    {
        return value == null ? "" : value;
    }

    public void modifyLink(final String rel, final String href)
    {
        RESTLink link = searchLink(rel);
        if (link == null)
        {
            addLink(new RESTLink(rel, href));
        }
        else
        {
            link.setHref(href);
        }
    }

    public Integer getIdFromLink(final String rel)
    {
        RESTLink restLink = this.searchLink(rel);
        if (restLink == null)
        {
            return null;
        }
        return restLink.getId();
    }

    @XmlTransient
    @JsonIgnore
    public Integer getEntityId()
    {
        Integer id = getIdFromLink("edit");
        if (id == null)
        {
            id = getIdFromLink("self");
        }
        return id;
    }

    @JsonIgnore
    public abstract String getMediaType();

    @JsonIgnore
    public abstract String getBaseMediaType();
}
