/**
 * Licensed to Abiquo Holdings S.L. (Abiquo) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package com.abiquo.model.util;

import java.lang.reflect.Array;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;

@SuppressWarnings("unchecked")
public class ModelTransformer
{
    public static <T> T transportFromPersistence(final Class<T> clazz, final Object persistent)
        throws Exception
    {
        return transform(clazz, clazz, persistent);
    }

    public static <T> T persistenceFromTransport(final Class<T> clazz, final Object transport)
        throws Exception
    {
        return transform(transport.getClass(), clazz, transport);
    }

    public static <T> T transform(final Class sourceClass, final Class<T> targetClass,
        final Object template) throws Exception
    {
        T instance = targetClass.newInstance();
        transform(sourceClass, targetClass, template, instance);
        return instance;
    }

    public static <T> void transform(final Class sourceClass, final Class<T> targetClass,
        final Object source, final T target) throws Exception
    {
        Field[] transportFields = sourceClass.getDeclaredFields();
        Class superClass = sourceClass.getSuperclass();
        while (superClass != null
            && !superClass.getSimpleName().equalsIgnoreCase("SingleResourceTransportDto"))
        {
            transportFields = (Field[]) addAll(transportFields, superClass.getDeclaredFields());
            superClass = superClass.getSuperclass();
        }

        for (Field field : transportFields)
        {
            int modifiers = field.getModifiers();
            if (!Modifier.isTransient(modifiers) && !Modifier.isStatic(modifiers))
            {
                String name = field.getName();
                try
                {
                    if (fieldExist(name, targetClass) && fieldExist(name, source.getClass())
                        || getterExist(name, source.getClass())
                        && setterExist(name, targetClass, field.getType()))
                    {
                        Object value =
                            getter(name, source.getClass()).invoke(source, new Object[0]);

                        if (setterExist(name, targetClass, field.getType()))
                        {
                            try
                            {
                                setter(name, targetClass, field.getType()).invoke(target,
                                    new Object[] {value});
                            }
                            catch (Exception ie)
                            {
                                setter(name, targetClass, value.getClass()).invoke(target,
                                    new Object[] {value});
                            }

                        }
                    }
                }
                catch (InvocationTargetException e)
                {
                    // Ignore invalid field
                }
            }
        }
    }

    private static Method getter(final String fieldName, final Class clazz) throws Exception
    {
        try
        {
            return getter("get", fieldName, clazz);
        }
        catch (NoSuchMethodException ex)
        {
            return getter("is", fieldName, clazz);
        }
    }

    private static Method getter(final String prefix, final String fieldName, final Class clazz)
        throws Exception
    {
        String name = prefix + capitalize(fieldName);
        return clazz.getMethod(name, new Class[0]);
    }

    private static Method setter(final String fieldName, final Class clazz, final Class type)
        throws Exception
    {
        String name = "set" + capitalize(fieldName);
        Method method = clazz.getMethod(name, new Class[] {type});

        if (method != null)
        {
            method.setAccessible(true);
        }
        return method;
    }

    private static boolean fieldExist(final String fieldName, final Class clazz) throws Exception
    {
        try
        {
            return clazz.getDeclaredField(fieldName) != null;
        }
        catch (NoSuchFieldException ex)
        {
            return false;
        }
    }

    private static boolean getterExist(final String fieldName, final Class clazz) throws Exception
    {
        try
        {
            return getter(fieldName, clazz) != null;
        }
        catch (NoSuchMethodException ex)
        {
            return false;
        }
    }

    private static boolean setterExist(final String fieldName, final Class clazz, final Class type)
        throws Exception
    {
        try
        {
            return setter(fieldName, clazz, type) != null;
        }
        catch (NoSuchMethodException ex)
        {
            return false;
        }
    }

    // from commons StringUtils
    private static String capitalize(String str)
    {
        int strLen;
        if (str == null || (strLen = str.length()) == 0)
        {
            return str;
        }
        return new StringBuilder(strLen).append(Character.toTitleCase(str.charAt(0)))
            .append(str.substring(1)).toString();
    }

    // from commons ArraysUtils
    public static Object[] addAll(Object[] array1, Object[] array2)
    {
        if (array1 == null)
        {
            if (array2 == null)
            {
                return null;
            }
            return (Object[]) array2.clone();

        }
        else if (array2 == null)
        {
            return (Object[]) array1.clone();
        }
        Object[] joinedArray =
            (Object[]) Array.newInstance(array1.getClass().getComponentType(), array1.length
                + array2.length);
        System.arraycopy(array1, 0, joinedArray, 0, array1.length);
        try
        {
            System.arraycopy(array2, 0, joinedArray, array1.length, array2.length);
        }
        catch (ArrayStoreException ase)
        {
            // Check if problem was due to incompatible types
            /*
             * We do this here, rather than before the copy because: - it would be a wasted check
             * most of the time - safer, in case check turns out to be too strict
             */
            final Class type1 = array1.getClass().getComponentType();
            final Class type2 = array2.getClass().getComponentType();
            if (!type1.isAssignableFrom(type2))
            {
                throw new IllegalArgumentException("Cannot store " + type2.getName()
                    + " in an array of " + type1.getName());
            }
            throw ase; // No, so rethrow original
        }
        return joinedArray;
    }
}
