/**
 * Licensed to Abiquo Holdings S.L. (Abiquo) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package com.abiquo.server.core.cloud;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import javax.xml.bind.annotation.adapters.XmlAdapter;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

public class ParameterAdapter extends XmlAdapter<Element, Map<String, Object>>
{

    private static final String DISKS = "disks";

    private static final String PATHS = "paths";

    private static final String RESULTS = "results";

    private DocumentBuilder documentBuilder;

    private DocumentBuilder getDocumentBuilder() throws Exception
    {
        // Lazy load the DocumentBuilder as it is not used for unmarshalling.
        if (null == documentBuilder)
        {
            DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
            documentBuilder = dbf.newDocumentBuilder();
        }
        return documentBuilder;
    }

    @Override
    public Element marshal(final Map<String, Object> parameter) throws Exception
    {
        if (null == parameter)
        {
            // No <metadata> tag
            return null;
        }
        DocumentBuilder db = getDocumentBuilder();

        Document document = db.newDocument();
        Element rootElement = document.createElement("metadata");
        getValue(parameter, rootElement, document);
        // Marshaller adds a <metadata> tag already
        return (Element) rootElement.getFirstChild();
    }

    /**
     * Recursive function to get an Element to represent an XML document i.e.
     * 
     * <pre>
     * <metadata><backup><full><monthly/><daily>10</daily></full></backup></metadata>
     * </pre>
     * 
     * @return Element
     */
    public Element getValue(final Map<String, Object> map, final Element rootElement,
        final Document document)
    {

        for (Entry<String, Object> entry : map.entrySet())
        {
            Element element = document.createElement(entry.getKey());
            Object value = entry.getValue();
            if (value instanceof Map)
            {
                getValue((Map) entry.getValue(), element, document);
            }
            else if (value instanceof List)
            {
                for (int i = 0; i < ((List) value).size(); i++)
                {
                    String currentName = element.getNodeName();
                    Object elementValue = ((List) value).get(i);
                    Element element2 =
                        document.createElement(currentName.substring(0, currentName.length() - 1));
                    if (elementValue instanceof Map)
                    {
                        getValue((Map) elementValue, element2, document);
                    }
                    else
                    {
                        element2.setTextContent(String.valueOf(elementValue));
                    }
                    element.appendChild(element2);
                }
            }
            else
            {
                element.setTextContent(value == null ? "" : value.toString());
            }
            rootElement.appendChild(element);
        }

        return rootElement;
    }

    @Override
    public Map<String, Object> unmarshal(final Element element) throws Exception
    {
        if (null == element)
        {
            return null;
        }
        return unmarshalMap(element, new HashMap<String, Object>());
    }

    /**
     * Recursive function to go over nested xml elements // i.e
     * 
     * <pre>
     * <backupSchedule><complete><monthly/><daily><time>10:30 -0300</time></daily></complete></backupSchedule>
     * </pre>
     * 
     * @return Map<String, Object>
     */
    public Map<String, Object> unmarshalMap(final Node node, final Map<String, Object> parameters)
    {
        if (node.getNodeType() == Node.ELEMENT_NODE && node.getFirstChild() == null)
        {
            parameters.put(node.getNodeName(), null);
        }
        else if (node.getNodeType() == Node.ELEMENT_NODE
            && node.getFirstChild().getNodeType() == Node.ELEMENT_NODE)
        {
            if (DISKS.equals(node.getNodeName()) || PATHS.equals(node.getNodeName()))
            {
                List<String> nodes = new ArrayList<String>();
                NodeList childs = node.getChildNodes();
                for (int i = 0; i < childs.getLength(); i++)
                {
                    Node child = childs.item(i);
                    if (child.getNodeType() != Node.TEXT_NODE)
                    {
                        nodes.add(child.getTextContent());
                    }
                }
                parameters.put(node.getNodeName(), nodes);
            }
            else if (RESULTS.equals(node.getNodeName()))
            {
                List<Map<String, Object>> nodes = new ArrayList<Map<String, Object>>();
                NodeList childs = node.getChildNodes();
                for (int i = 0; i < childs.getLength(); i++)
                {
                    Node child = childs.item(i);
                    if (child.getNodeType() != Node.TEXT_NODE)
                    {
                        Map<String, Object> map = new HashMap<>();
                        unmarshalMap(child.getFirstChild(), map);
                        nodes.add(map);
                    }
                }
                parameters.put(node.getNodeName(), nodes);
            }
            else
            {
                Map<String, Object> map = new HashMap<>();
                parameters.put(node.getNodeName(), map);
                unmarshalMap(node.getFirstChild(), map);
            }

        }
        else if (node.getNodeType() == Node.ELEMENT_NODE
            && node.getFirstChild().getNodeType() == Node.TEXT_NODE)
        {
            parameters.put(node.getNodeName(), node.getFirstChild().getNodeValue());
        }

        if (node.getNextSibling() != null)
        {
            unmarshalMap(node.getNextSibling(), parameters);
        }
        return parameters;
    }

}
