/**
 * Licensed to Abiquo Holdings S.L. (Abiquo) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package com.abiquo.server.core.cloud;

import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.validation.constraints.NotNull;
import javax.xml.bind.annotation.XmlAnyElement;
import javax.xml.bind.annotation.XmlElement;
import javax.xml.bind.annotation.XmlRootElement;
import javax.xml.bind.annotation.XmlType;
import javax.xml.bind.annotation.adapters.XmlJavaTypeAdapter;

import com.abiquo.model.rest.RESTLink;
import com.abiquo.model.transport.SingleResourceTransportDto;
import com.abiquo.server.core.cloud.chef.RunlistElementsDto;
import com.abiquo.server.core.infrastructure.storage.DvdManagementDto;

/**
 * @restlink <link rel="template" href=
 *           "example.com/api/admin/enterprises/{id}/datacenterrepositories/{id}/virtualmachinetemplates/{id}"
 *           />
 */
@XmlRootElement(name = "virtualMachine")
@XmlType(propOrder = {"id", "uuid", "name", "label", "description", "cpu", "coresPerSocket", "ram",
"hdInBytes", "vdrpEnabled", "vdrpPort", "vdrpIP", "idState", "state", "idType", "type",
"highDisponibility", "password", "keymap", "dvd", "runlist", "metadata", "monitored"})
public class VirtualMachineDto extends SingleResourceTransportDto
{
    private static final long serialVersionUID = -8877350185009627544L;

    private static final String TYPE = "application/vnd.abiquo.virtualmachine";

    public static final String MEDIA_TYPE = TYPE + "+" + JSON;

    public static final String SHORT_MEDIA_TYPE_JSON = MEDIA_TYPE;

    public static final String SHORT_MEDIA_TYPE_XML = TYPE + "+" + XML;

    public static final String MEDIA_TYPE_XML = TYPE + "+" + XML + VERSION_PARAM + API_VERSION;

    public static final String MEDIA_TYPE_JSON = MEDIA_TYPE + VERSION_PARAM + API_VERSION;

    public static final String MEDIA_TYPE_22 = TYPE + "+" + XML + VERSION_PARAM + API_VERSION_22;

    public static final String MEDIA_TYPE_23 = TYPE + "+" + XML + VERSION_PARAM + API_VERSION_23;

    public static final String MEDIA_TYPE_24 = TYPE + "+" + XML + VERSION_PARAM + API_VERSION_24;

    public static final String MEDIA_TYPE_XML_26 = TYPE + "+" + XML + VERSION_PARAM
        + API_VERSION_26;

    public static final String MEDIA_TYPE_JSON_26 = TYPE + "+" + XML + VERSION_PARAM
        + API_VERSION_26;

    private Integer id;

    private String name;

    private String description;

    private Integer ram;

    private Integer cpu;

    private Integer coresPerSocket;

    private Long hdInBytes;

    private Integer vdrpPort;

    private String vdrpIP;

    private Boolean vdrpEnabled;

    private int idState;

    private VirtualMachineState state;

    private int highDisponibility;

    private int idType;

    private VirtualMachineType type;

    private String password;

    private String keymap;

    private String uuid;

    private RunlistElementsDto runlistElements;

    private DvdManagementDto dvd;

    private Map<String, Object> metadata;

    private String label;

    private boolean monitored;

    @Override
    public String getBaseMediaType()
    {
        return MEDIA_TYPE;
    }

    /**
     * Number of CPUs of the virtual machine.
     */
    @NotNull
    public Integer getCpu()
    {
        return cpu;
    }

    /**
     * Optional ratio to distribute cores in sockets (sockets = cpu / coresPerSocket).
     * <p>
     * This attribute MUST be divisor of CPU.
     */
    public Integer getCoresPerSocket()
    {
        return coresPerSocket;
    }

    public String getDescription()
    {
        return description;
    }

    /**
     * Whether the virtual machine must mount a CD or DVD drive.
     */
    @XmlElement(name = "dvd")
    public DvdManagementDto getDvd()
    {
        return dvd;
    }

    /**
     * Hard disk size in Bytes.
     */
    @NotNull
    public Long getHdInBytes()
    {
        return hdInBytes;
    }

    /**
     * Deprecated
     */
    @NotNull
    public int getHighDisponibility()
    {
        return highDisponibility;
    }

    /**
     * Opaque ID of the virtual machine.
     * 
     * @output
     */
    public Integer getId()
    {
        return id;
    }

    /**
     * Opaque ID of the state.
     */
    public int getIdState()
    {
        return idState;
    }

    /**
     * Opaque ID of the virtual machine type.
     */
    @Deprecated
    public int getIdType()
    {
        return idType;
    }

    /**
     * The type of the virtual machine
     * 
     * @output
     */
    public VirtualMachineType getType()
    {
        return type;
    }

    /**
     * Keyboard keymap to be setted on the virtual machine.
     */
    public String getKeymap()
    {
        return keymap;
    }

    @Override
    public String getMediaType()
    {
        return MEDIA_TYPE_JSON;
    }

    /**
     * Name of the virtual machine.
     */
    @NotNull
    public String getName()
    {
        return name;
    }

    /**
     * VNC password.
     */
    public String getPassword()
    {
        return password;
    }

    /**
     * Ram size of the virtual machine.
     */
    @NotNull
    public Integer getRam()
    {
        return ram;
    }

    /**
     * Chef runlist to bootstrap.
     */
    public RunlistElementsDto getRunlist()
    {
        return runlistElements;
    }

    /**
     * Current virtual machine state.
     */
    public VirtualMachineState getState()
    {
        return state;
    }

    /**
     * UUID of the virtual machine.
     * 
     * @output
     */
    public String getUuid()
    {
        return uuid;
    }

    /**
     * Virtual remote desktop IP.
     * 
     * @output
     */
    public String getVdrpIP()
    {
        return vdrpIP;
    }

    /**
     * Virtual remote desktop port.
     * 
     * @output
     */
    public Integer getVdrpPort()
    {
        return vdrpPort;
    }

    public String getLabel()
    {
        return label;
    }

    public Boolean getVdrpEnabled()
    {
        return vdrpEnabled;
    }

    public void setVdrpEnabled(final Boolean vdrpEnabled)
    {
        this.vdrpEnabled = vdrpEnabled;
    }

    public void setCpu(final Integer cpu)
    {
        this.cpu = cpu;
    }

    public void setCoresPerSocket(final Integer coresPerSocket)
    {
        this.coresPerSocket = coresPerSocket;
    }

    public void setDescription(final String description)
    {
        this.description = description;
    }

    public void setDvd(final DvdManagementDto dvd)
    {
        this.dvd = dvd;
    }

    public void setHdInBytes(final Long hdInBytes)
    {
        this.hdInBytes = hdInBytes;
    }

    public void setHighDisponibility(final int highDisponibility)
    {
        this.highDisponibility = highDisponibility;
    }

    public void setId(final Integer id)
    {
        this.id = id;
    }

    public void setIdState(final int idState)
    {
        this.idState = idState;
    }

    @Deprecated
    public void setIdType(final int idType)
    {
        this.idType = idType;
    }

    public void setType(final VirtualMachineType type)
    {
        this.type = type;
    }

    public void setKeymap(final String keymap)
    {
        this.keymap = keymap;
    }

    public void setName(final String name)
    {
        this.name = name;
    }

    public void setPassword(final String password)
    {
        this.password = password;
    }

    public void setRam(final Integer ram)
    {
        this.ram = ram;
    }

    public void setRunlist(final RunlistElementsDto runlistElements)
    {
        this.runlistElements = runlistElements;
    }

    public void setState(final VirtualMachineState state)
    {
        this.state = state;
    }

    public void setUuid(final String uuid)
    {
        this.uuid = uuid;
    }

    public void setVdrpIP(final String vdrpIP)
    {
        this.vdrpIP = vdrpIP;
    }

    public void setVdrpPort(final Integer vdrpPort)
    {
        this.vdrpPort = vdrpPort;
    }

    @XmlAnyElement
    @XmlJavaTypeAdapter(ParameterAdapter.class)
    public Map<String, Object> getMetadata()
    {
        return metadata;
    }

    public void setMetadata(final Map<String, Object> metadata)
    {
        this.metadata = metadata;
    }

    public void setLabel(final String label)
    {
        this.label = label;
    }

    /**
     * Modify the NIC links.
     * 
     * @param nics new NIC links
     */
    public void setNicLinks(final List<RESTLink> nics)
    {
        // First delete current Nic links. Convert to iterator
        // because it is impossible to delete objects in a list.
        Iterator<RESTLink> currentNics = getLinks().iterator();
        while (currentNics.hasNext())
        {
            if (currentNics.next().getRel().toLowerCase().startsWith("nic"))
            {
                currentNics.remove();
            }
        }

        for (RESTLink nicLink : nics)
        {
            // add nic link to list of links
            if (nicLink.getRel().toLowerCase().startsWith("nic"))
            {
                // avoid problems with another kind of rels
                getLinks().add(nicLink);
            }
        }
    }

    public boolean isMonitored()
    {
        return monitored;
    }

    public void setMonitored(final boolean monitored)
    {
        this.monitored = monitored;
    }
}
