/**
 * Licensed to Abiquo Holdings S.L. (Abiquo) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package com.abiquo.server.core.cloud;

import javax.xml.bind.annotation.XmlEnum;
import javax.xml.bind.annotation.XmlType;

/**
 * The virtual machine state model. The transitions between states are in virtual machine state
 * transition.
 * 
 * @author ssedano
 */
@XmlType(name = "VirtualMachineState")
@XmlEnum
public enum VirtualMachineState
{

    /**
     * The virtual machine only exists in Abiquo and has not yet a physical machine assigned.
     */
    NOT_ALLOCATED,
    
    /**
     * The virtual machine does not exists in the hypervisor but has physical machine assigned.
     */
    ALLOCATED,
    
    /**
     * The virtual machine exists in the hypervisor.
     */
    CONFIGURED,
    
    /**
     * The virtual machine exists in the hypervisor and is ON.
     */
    ON,
    
    /**
     * The virtual machine exists in the hypervisor and is SUSPENDED.
     */
    PAUSED,
    
    /**
     * The virtual machine exists in the hypervisor and is OFF.
     */
    OFF,
    
    /**
     * Some operation is being performed on the virtual machine.
     */
    LOCKED,
    
    /**
     * Abiquo does know the actual state of the virtual machine. But it exists in the hypervisor.
     */
    UNKNOWN;

    public static VirtualMachineState fromValue(final String value)
    {
        return VirtualMachineState.valueOf(value.toUpperCase());
    }

    public VirtualMachineState travel(final VirtualMachineStateTransition transaction)
    {
        if (!transaction.isValidOrigin(this))
        {
            throw new RuntimeException("Invalid origin " + this + " for transaction " + transaction);
        }

        return transaction.getEndState();
    }

    public int id()
    {
        return ordinal() + 1;
    }

    public static VirtualMachineState fromId(final int id)
    {
        return VirtualMachineState.values()[id - 1];
    }

    public String toOVF()
    {
        switch (this)
        {
            case ON:
                return "POWERUP_ACTION";
            case PAUSED:
                return "PAUSE_ACTION";
            case OFF:
                return "POWERDOWN_ACTION";
        }
        return null;
    }

    public String toResourceState()
    {
        switch (this)
        {
            case ON:
                return "PowerUp";
            case PAUSED:
                return "Pause";
            case OFF:
                return "PowerOff";
                // case REBOOTED:
                // return "Resume";
        }
        return null;
    }

    public boolean isDeployed()
    {
        switch (this)
        {
            case ON:
            case OFF:
            case PAUSED:
                return true;
                // Configured state. The VM is in the hypervisor but it
                // has never been powered on. The Chef agent has not run
                // yet and the node does not exist in the Chef server.
            case CONFIGURED:
            default:
                return false;
        }
    }

    public boolean existsInHypervisor()
    {
        switch (this)
        {
            case ON:
            case OFF:
            case PAUSED:
            case CONFIGURED:
                return true;
            default:
                return false;
        }
    }

    public boolean reconfigureAllowed()
    {
        switch (this)
        {
            case ALLOCATED:
            case NOT_ALLOCATED:
            case OFF:
                return true;
            default:
                return false;
        }
    }

    public boolean instanceAllowed()
    {
        switch (this)
        {
            case OFF:
                return true;

            default:
                return false;
        }
    }
}
