/**
 * Licensed to Abiquo Holdings S.L. (Abiquo) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package com.abiquo.server.core.cloud;

import static com.abiquo.server.core.cloud.VirtualMachineState.ALLOCATED;
import static com.abiquo.server.core.cloud.VirtualMachineState.CONFIGURED;
import static com.abiquo.server.core.cloud.VirtualMachineState.NOT_ALLOCATED;
import static com.abiquo.server.core.cloud.VirtualMachineState.OFF;
import static com.abiquo.server.core.cloud.VirtualMachineState.ON;
import static com.abiquo.server.core.cloud.VirtualMachineState.PAUSED;
import static java.util.Collections.singleton;

import java.util.Arrays;
import java.util.HashSet;
import java.util.Set;

/**
 * Valid transactions between virtual machine states.
 */
public enum VirtualMachineStateTransition
{
    // Configure transition
    CONFIGURE(singleton(ALLOCATED), CONFIGURED),

    // Reconfigure transition
    RECONFIGURE(singleton(OFF), CONFIGURED),

    // Deconfigure transition
    DECONFIGURE(singleton(CONFIGURED), ALLOCATED),

    // PowerOn transition
    POWERON(new HashSet<VirtualMachineState>(Arrays.asList(CONFIGURED, OFF)), ON),

    // PowerOff transition
    POWEROFF(singleton(ON), OFF),

    // Reset transition
    RESET(singleton(ON), ON),

    // Pause transition
    PAUSE(singleton(ON), PAUSED),

    // Resume transition
    RESUME(singleton(PAUSED), ON),

    // Snapshot transition
    SNAPSHOT(singleton(OFF), OFF),

    // Not allocated yet
    ALLOCATE(singleton(NOT_ALLOCATED), ALLOCATED),

    // Exists the machine in Abiquo, and has hypervisor assigned, but does not exists in hypervisor
    DEALLOCATE(singleton(ALLOCATED), NOT_ALLOCATED);

    private Set<VirtualMachineState> origins;

    private VirtualMachineState end;

    private VirtualMachineStateTransition(final Set<VirtualMachineState> origins,
        final VirtualMachineState end)
    {
        this.origins = origins;
        this.end = end;
    }

    public VirtualMachineState getEndState()
    {
        return this.end;
    }

    public boolean isValidOrigin(final VirtualMachineState origin)
    {
        return this.origins.contains(origin);
    }

    public static VirtualMachineStateTransition fromValue(final String value)
    {
        return VirtualMachineStateTransition.valueOf(value.toUpperCase());
    }

    public static VirtualMachineStateTransition rollback(final VirtualMachineStateTransition s)
    {
        switch (s)
        {
            case CONFIGURE:
                return DECONFIGURE;

            case DECONFIGURE:
                return CONFIGURE;

            case POWEROFF:
                return POWERON;
            case POWERON:
                return POWEROFF;

            case PAUSE:
                return RESUME;

            case RESUME:
                return PAUSE;

            default: // reset reconfigure snapshot
                return s;
        }
    }

    public static boolean isValidTransition(final VirtualMachineState origin,
        final VirtualMachineState end)
    {
        for (VirtualMachineStateTransition l : VirtualMachineStateTransition.values())
        {
            if (l.isValidOrigin(origin) && l.getEndState().equals(end))
            {
                return true;
            }
        }
        return false;
    }

    public static VirtualMachineStateTransition getValidTransition(
        final VirtualMachineState origin, final VirtualMachineState end)
    {
        for (VirtualMachineStateTransition l : VirtualMachineStateTransition.values())
        {
            if (l.isValidOrigin(origin) && l.getEndState().equals(end))
            {
                return l;
            }
        }
        return null;
    }

    public static VirtualMachineStateTransition getValidVmStateChangeTransition(
        final VirtualMachineState origin, final VirtualMachineState end)
    {
        VirtualMachineStateTransition validTransition = getValidTransition(origin, end);
        if (validTransition != null)
        {
            switch (validTransition)
            {
                case POWEROFF:
                case PAUSE:
                case POWERON:
                case RESUME:
                    return validTransition;
            }
        }
        return null;
    }
}
