/**
 * Licensed to Abiquo Holdings S.L. (Abiquo) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package com.abiquo.server.core.config;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;

import javax.validation.constraints.NotNull;
import javax.xml.bind.annotation.XmlRootElement;
import javax.xml.bind.annotation.XmlTransient;
import javax.xml.bind.annotation.XmlType;

import com.abiquo.model.transport.SingleResourceTransportDto;
import com.fasterxml.jackson.annotation.JsonIgnore;

/**
 * Representation of a License.
 * 
 * @author Ignasi Barrera
 */
@XmlRootElement(name = "license")
// enabledPLugins not used yet. Do not serialize to xml.
@XmlType(propOrder = {"id", "numcores", "expiration", /* "enabledPLugins", */"code"})
public class LicenseDto extends SingleResourceTransportDto
{
    private static final long serialVersionUID = 1L;

    private static final String TYPE = "application/vnd.abiquo.license";

    public static final String MEDIA_TYPE = TYPE + "+" + JSON;

    public static final String SHORT_MEDIA_TYPE_JSON = MEDIA_TYPE;

    public static final String SHORT_MEDIA_TYPE_XML = TYPE + "+" + XML;

    public static final String MEDIA_TYPE_XML = TYPE + "+" + XML + VERSION_PARAM + API_VERSION;

    public static final String MEDIA_TYPE_JSON = MEDIA_TYPE + VERSION_PARAM + API_VERSION;

    /** The date format used to store license dates. */
    public static final String DATE_FORMAT = "yyyy-MM-dd HH:mm:ss";

    /** The license id **/
    private Integer id;

    /** The license code. */
    private String code;

    /** The number of cores. */
    private Integer numcores;

    /** The expiration date. */
    private String expiration;

    /** The enabled plugins. */
    private String enabledPlugins;

    /**
     * The identifier of the entity
     * 
     * @output
     */
    public Integer getId()
    {
        return id;
    }

    public void setId(final Integer id)
    {
        this.id = id;
    }

    /**
     * The license code
     */
    @NotNull
    public String getCode()
    {
        return code;
    }

    public void setCode(final String code)
    {
        this.code = code;
    }

    /**
     * The number of cores that the license enables
     * 
     * @output
     */
    public Integer getNumcores()
    {
        return numcores;
    }

    public void setNumcores(final Integer numcores)
    {
        this.numcores = numcores;
    }

    /**
     * The expiration date of the license
     * 
     * @output
     */
    public String getExpiration()
    {
        return expiration;
    }

    public void setExpiration(final String expiration)
    {
        this.expiration = expiration;
    }

    /**
     * The enabled plugins
     * 
     * @output
     */
    // Not used yet. Do not serialize to xml.
    @XmlTransient
    public String getEnabledPlugins()
    {
        return enabledPlugins;
    }

    public void setEnabledPlugins(final String enabledPlugins)
    {
        this.enabledPlugins = enabledPlugins;
    }

    @Override
    public String getMediaType()
    {
        return MEDIA_TYPE_JSON;
    }

    @Override
    public String getBaseMediaType()
    {
        return MEDIA_TYPE;
    }

    // *************************** Helper methods ***********************

    @XmlTransient
    @JsonIgnore
    public boolean isActive()
    {
        SimpleDateFormat format = new SimpleDateFormat(DATE_FORMAT);
        try
        {
            Date expiration = format.parse(getExpiration());
            return expiration.getTime() >= System.currentTimeMillis();
        }
        catch (ParseException e)
        {
            throw new RuntimeException("The license expiration date cannot be parsed");
        }
    }
}
