/**
 * Licensed to Abiquo Holdings S.L. (Abiquo) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package com.abiquo.server.core.enterprise;

import javax.validation.constraints.NotNull;
import javax.xml.bind.annotation.XmlElement;
import javax.xml.bind.annotation.XmlRootElement;
import javax.xml.bind.annotation.XmlType;

import com.abiquo.model.transport.Expandable;
import com.abiquo.model.transport.SingleResourceTransportDto;
import com.fasterxml.jackson.annotation.JsonProperty;

/**
 * The User entity offers the functionality of managing the users of an enterprise in a logical way
 * 
 * @author destevezg
 */
@XmlRootElement(name = "user")
@XmlType(propOrder = {"id", "nick", "password", "name", "surname", "description", "email",
"locale", "authType", "availableVirtualDatacenters", "active", "expands", "publicSshKey"})
public class UserDto extends SingleResourceTransportDto implements Expandable<PrivilegeDto>
{
    private static final long serialVersionUID = -3233121860012587479L;

    private static final String TYPE = "application/vnd.abiquo.user";

    public static final String MEDIA_TYPE = TYPE + "+" + JSON;

    public static final String SHORT_MEDIA_TYPE_JSON = MEDIA_TYPE;

    public static final String SHORT_MEDIA_TYPE_XML = TYPE + "+" + XML;

    public static final String MEDIA_TYPE_XML = TYPE + "+" + XML + VERSION_PARAM + API_VERSION;

    public static final String MEDIA_TYPE_JSON = MEDIA_TYPE + VERSION_PARAM + API_VERSION;

    private Integer id;

    private String name;

    private String nick;

    private String locale;

    private String password;

    private String surname;

    private boolean active;

    private String email;

    private String description;

    // comma separated values
    private String availableVirtualDatacenters;

    /** Expanded nodes (privileges flat). */
    private PrivilegesDto expands;

    /**
     * String representation for {@link com.abiquo.model.enumerator.AuthType} value.
     */
    private String authType;

    public UserDto()
    {

    }

    /**
     * Constructor.
     * 
     * @param name Name.
     * @param surname Surname.
     * @param email Email Address.
     * @param nick Login.
     * @param password Password.
     * @param locale Language.
     * @param description Desc.
     * @deprecated use
     *             {@link #UserDto(String, String, String, String, String, String, String, String)}
     *             instead.
     */
    @Deprecated
    public UserDto(final String name, final String surname, final String email, final String nick,
        final String password, final String locale, final String description)
    {
        this.name = name;
        this.surname = surname;
        this.email = email;
        this.nick = nick;
        this.password = password;
        this.locale = locale;
        this.description = description;
    }

    /**
     * Constructor.
     * 
     * @param name Name.
     * @param surname Surname.
     * @param email Email Address.
     * @param nick Login.
     * @param password Password.
     * @param locale Language.
     * @param description Desc.
     * @param authType AuthType value.
     */
    public UserDto(final String name, final String surname, final String email, final String nick,
        final String password, final String locale, final String description, final String authType)
    {
        this.name = name;
        this.surname = surname;
        this.email = email;
        this.nick = nick;
        this.password = password;
        this.locale = locale;
        this.description = description;
        this.authType = authType;
    }

    /**
     * Auth type selected for this user: Abiquo managed or LDAP
     * {@link com.abiquo.model.enumerator.AuthType} value.
     * 
     * @output
     */
    @NotNull
    public String getAuthType()
    {
        return authType;
    }

    /**
     * List with restricted VDCs for this user separated by commas
     * 
     * @output
     */
    public String getAvailableVirtualDatacenters()
    {
        return availableVirtualDatacenters;
    }

    /**
     * A short description of the user
     * 
     * @output
     */
    public String getDescription()
    {
        return description;
    }

    /**
     * The contact e-mail address of the user for Abiquo messages
     * 
     * @output
     */
    public String getEmail()
    {
        return email;
    }

    /**
     * Identifier for this user
     * 
     * @output
     */
    public Integer getId()
    {
        return id;
    }

    /**
     * Locale assigned to this user
     * 
     * @output
     */
    public String getLocale()
    {
        return locale;
    }

    /**
     * User's real first name
     * 
     * @output
     */
    @NotNull
    public String getName()
    {
        return name;
    }

    /**
     * User name / nickname. The username for login
     * 
     * @output
     */
    @NotNull
    public String getNick()
    {
        return nick;
    }

    /**
     * Password set for this user
     * 
     * @output
     */
    public String getPassword()
    {
        return password;
    }

    /**
     * User's real last name
     * 
     * @output
     */
    public String getSurname()
    {
        return surname;
    }

    /**
     * Flag indicating if the user is activated or not
     * 
     * @output
     */
    @NotNull
    @JsonProperty
    public boolean isActive()
    {
        return active;
    }

    @JsonProperty
    public void setActive(final boolean active)
    {
        this.active = active;
    }

    /**
     * {@link com.abiquo.model.enumerator.AuthType} value.
     * 
     * @param authType {@link com.abiquo.model.enumerator.AuthType} value.
     */
    public void setAuthType(final String authType)
    {
        this.authType = authType;
    }

    public void setAvailableVirtualDatacenters(final String availableVirtualDatacenters)
    {
        this.availableVirtualDatacenters = availableVirtualDatacenters;
    }

    public void setDescription(final String description)
    {
        this.description = description;
    }

    public void setEmail(final String email)
    {
        this.email = email;
    }

    public void setId(final Integer id)
    {
        this.id = id;
    }

    public void setLocale(final String locale)
    {
        this.locale = locale;
    }

    public void setName(final String name)
    {
        this.name = name;
    }

    public void setNick(final String nick)
    {
        this.nick = nick;
    }

    public void setPassword(final String password)
    {
        this.password = password;
    }

    public void setSurname(final String surname)
    {
        this.surname = surname;
    }

    /**
     * This representation might include privileges.
     * 
     * @output
     * @return flat privileges.
     */
    @Override
    @XmlElement(name = "privileges")
    public PrivilegesDto getExpands()
    {
        return expands;
    }

    /**
     * This representation might include privileges.
     * 
     * @output
     * @return flat privileges.
     */
    public void setExpands(final PrivilegesDto expands)
    {
        this.expands = expands;
    }

    /**
     * The SSH public key of the user.
     */
    private String publicSshKey;

    public String getPublicSshKey()
    {
        return publicSshKey;
    }

    public void setPublicSshKey(final String publicSshKey)
    {
        this.publicSshKey = publicSshKey;
    }

    @Override
    public String getMediaType()
    {
        return MEDIA_TYPE_JSON;
    }

    @Override
    public String getBaseMediaType()
    {
        return MEDIA_TYPE;
    }
}
