/**
 * Licensed to Abiquo Holdings S.L. (Abiquo) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package com.abiquo.server.core.infrastructure;

import java.io.Serializable;

import javax.validation.constraints.NotNull;
import javax.xml.bind.annotation.XmlElement;
import javax.xml.bind.annotation.XmlRootElement;
import javax.xml.bind.annotation.XmlType;

import com.abiquo.model.enumerator.MachineState;
import com.abiquo.model.transport.SingleResourceTransportDto;
import com.abiquo.server.core.infrastructure.network.NetworkInterfacesDto;

@XmlRootElement(name = "machine")
@XmlType(propOrder = {"id", "name", "description", "state", "type", "ip", "ipService", "port",
"virtualRamInMb", "virtualRamUsedInMb", "virtualCpuCores", "virtualCpusUsed", "user", "password",
"initiatorIQN", "ipmiIP", "ipmiPort", "ipmiUser", "ipmiPassword", "datastores",
"networkInterfaces", "managerIp", "managerPort", "managerUser", "managerPassword", "agentIp",
"agentPort", "agentUser", "agentPassword"})
public class MachineDto extends SingleResourceTransportDto implements Serializable
{
    private static final long serialVersionUID = -4971248626582072165L;

    private static final String TYPE = "application/vnd.abiquo.machine";

    public static final String MEDIA_TYPE = TYPE + "+" + JSON;

    public static final String SHORT_MEDIA_TYPE_JSON = MEDIA_TYPE;

    public static final String SHORT_MEDIA_TYPE_XML = TYPE + "+" + XML;

    public static final String MEDIA_TYPE_XML = TYPE + "+" + XML + VERSION_PARAM + API_VERSION;

    public static final String MEDIA_TYPE_JSON = MEDIA_TYPE + VERSION_PARAM + API_VERSION;

    public static final String MEDIA_TYPE_JSON_29 = TYPE + "+" + JSON + VERSION_PARAM
        + API_VERSION_29;

    public static final String MEDIA_TYPE_JSON_28 = TYPE + "+" + JSON + VERSION_PARAM
        + API_VERSION_28;

    public static final String MEDIA_TYPE_JSON_27 = TYPE + "+" + JSON + VERSION_PARAM
        + API_VERSION_27;

    public static final String MEDIA_TYPE_JSON_26 = TYPE + "+" + JSON + VERSION_PARAM
        + API_VERSION_26;

    public static final String MEDIA_TYPE_29 = TYPE + "+" + XML + VERSION_PARAM + API_VERSION_29;

    public static final String MEDIA_TYPE_28 = TYPE + "+" + XML + VERSION_PARAM + API_VERSION_28;

    public static final String MEDIA_TYPE_27 = TYPE + "+" + XML + VERSION_PARAM + API_VERSION_27;

    public static final String MEDIA_TYPE_26 = TYPE + "+" + XML + VERSION_PARAM + API_VERSION_26;

    public static final String MEDIA_TYPE_24 = TYPE + "+" + XML + VERSION_PARAM + API_VERSION_24;

    public static final String MEDIA_TYPE_23 = TYPE + "+" + XML + VERSION_PARAM + API_VERSION_23;

    private Integer id;

    private String name, description;

    private Integer virtualRamInMb, virtualRamUsedInMb;

    private Integer virtualCpuCores, virtualCpusUsed;

    private MachineState state;

    private Integer port;

    private String ip;

    private String ipService;

    private String type;

    private String user;

    private String password;

    private DatastoresDto datastores;

    private NetworkInterfacesDto networkInterfaces;

    private String ipmiIP;

    private Integer ipmiPort;

    private String ipmiUser;

    private String ipmiPassword;

    private String initiatorIQN;

    private String managerIp;

    private Integer managerPort;

    private String managerUser;

    private String managerPassword;

    private String agentIp;

    private Integer agentPort;

    private String agentUser;

    private String agentPassword;

    /**
     * Connection port of the machine
     */
    public Integer getPort()
    {
        return port;
    }

    /**
     * @param port the port to set
     */
    public void setPort(final Integer port)
    {
        this.port = port;
    }

    /**
     * IP of the machine
     */
    public String getIp()
    {
        return ip;
    }

    /**
     * @param ip the ip to set
     */
    public void setIp(final String ip)
    {
        this.ip = ip;
    }

    /**
     * IP of the hypervisor service of machine
     */
    public String getIpService()
    {
        return ipService;
    }

    /**
     * @param ipService the ipService to set
     */
    public void setIpService(final String ipService)
    {
        this.ipService = ipService;
    }

    /**
     * Value of the hypervisor type supported by abiquo
     */
    @NotNull
    public String getType()
    {
        return type;
    }

    /**
     * @param type the type to set
     */
    public void setType(final String type)
    {
        this.type = type;
    }

    /**
     * Name of the user for loggon into the hypervisor service
     */
    public String getUser()
    {
        return user;
    }

    /**
     * @param user the user to set
     */
    public void setUser(final String user)
    {
        this.user = user;
    }

    /**
     * Password of the user for loggon into the hypervisor service
     */
    public String getPassword()
    {
        return password;
    }

    /**
     * @param password the password to set
     */
    public void setPassword(final String password)
    {
        this.password = password;
    }

    /**
     * Identifier of the machine
     */
    public Integer getId()
    {
        return id;
    }

    public void setId(final Integer id)
    {
        this.id = id;
    }

    /**
     * State of the machine, can be one of the following {STOPPED, PROVISIONED, NOT_MANAGED,
     * MANAGED, HALTED, UNLICENSED, HA_IN_PROGRESS, DISABLED_FOR_HA, HALTED_FOR_SAVE}
     * 
     * @output
     */
    public MachineState getState()
    {
        return state;
    }

    public void setState(final MachineState state)
    {
        this.state = state;
    }

    /**
     * Name of the machine
     */
    @NotNull
    public String getName()
    {
        return name;
    }

    public void setName(final String name)
    {
        this.name = name;
    }

    /**
     * Description of the machine
     */
    public String getDescription()
    {
        return description;
    }

    public void setDescription(final String description)
    {
        this.description = description;
    }

    /**
     * RAM in MB of the machine
     */
    @XmlElement(name = "ram")
    @NotNull
    public Integer getVirtualRamInMb()
    {
        return getDefaultMb(virtualRamInMb).intValue();
    }

    public void setVirtualRamInMb(final Integer virtualRamInMb)
    {
        this.virtualRamInMb = virtualRamInMb;
    }

    /**
     * Used RAM in MB of the machine
     */
    @XmlElement(name = "ramUsed")
    @NotNull
    public Integer getVirtualRamUsedInMb()
    {
        return getDefaultMb(virtualRamUsedInMb).intValue();
    }

    public void setVirtualRamUsedInMb(final Integer virtualRamUsedInMb)
    {
        this.virtualRamUsedInMb = virtualRamUsedInMb;
    }

    /**
     * Numbre of cpus of the machine
     */
    @XmlElement(name = "cpu")
    @NotNull
    public Integer getVirtualCpuCores()
    {
        return (Integer) getDefaultMb(virtualCpuCores);
    }

    public void setVirtualCpuCores(final Integer virtualCpuCores)
    {
        this.virtualCpuCores = virtualCpuCores;
    }

    /**
     * Numbre of used cpus of the machine
     */
    @XmlElement(name = "cpuUsed")
    @NotNull
    public Integer getVirtualCpusUsed()
    {
        return (Integer) getDefaultMb(virtualCpusUsed);
    }

    public void setVirtualCpusUsed(final Integer virtualCpusUsed)
    {
        this.virtualCpusUsed = virtualCpusUsed;
    }

    private Number getDefaultMb(final Number mb)
    {
        return mb == null ? 1 : mb;
    }

    /**
     * @param datastores the datastores to set
     */
    public void setDatastores(final DatastoresDto datastores)
    {
        this.datastores = datastores;
    }

    /**
     * List of datastore of the machine
     */
    @NotNull
    public DatastoresDto getDatastores()
    {
        if (datastores == null)
        {
            datastores = new DatastoresDto();
        }
        return datastores;
    }

    /**
     * IP of the impi service of the machine
     */
    public String getIpmiIP()
    {
        return ipmiIP;
    }

    public void setIpmiIP(final String ipmiIP)
    {
        this.ipmiIP = ipmiIP;
    }

    /**
     * Connection port of the impi service of the machine
     */
    public Integer getIpmiPort()
    {
        return ipmiPort;
    }

    public void setIpmiPort(final Integer ipmiPort)
    {
        this.ipmiPort = ipmiPort;
    }

    /**
     * Name of the user for loggon into the impi service of the machine
     */
    public String getIpmiUser()
    {
        return ipmiUser;
    }

    public void setIpmiUser(final String ipmiUser)
    {
        this.ipmiUser = ipmiUser;
    }

    /**
     * Password of the user for loggon into the impi service of the machine
     */
    public String getIpmiPassword()
    {
        return ipmiPassword;
    }

    public void setIpmiPassword(final String ipmiPassword)
    {
        this.ipmiPassword = ipmiPassword;
    }

    /**
     * Initiator IQN of the machine used to use storage devices
     */
    public String getInitiatorIQN()
    {
        return initiatorIQN;
    }

    public void setInitiatorIQN(final String initiatorIQN)
    {
        this.initiatorIQN = initiatorIQN;
    }

    @Override
    public String getMediaType()
    {
        return MEDIA_TYPE_JSON;
    }

    @Override
    public String getBaseMediaType()
    {
        return MEDIA_TYPE;
    }

    /**
     * The networkInterfaces
     */
    public NetworkInterfacesDto getNetworkInterfaces()
    {
        if (networkInterfaces == null)
        {
            networkInterfaces = new NetworkInterfacesDto();
        }
        return networkInterfaces;
    }

    /**
     * @param networkInterfaces the networkInterfaces to set
     */
    public void setNetworkInterfaces(final NetworkInterfacesDto networkInterfaces)
    {
        this.networkInterfaces = networkInterfaces;
    }

    public String getManagerIp()
    {
        return managerIp;
    }

    public void setManagerIp(final String managerIp)
    {
        this.managerIp = managerIp;
    }

    public Integer getManagerPort()
    {
        return managerPort;
    }

    public void setManagerPort(final Integer managerPort)
    {
        this.managerPort = managerPort;
    }

    public String getManagerUser()
    {
        return managerUser;
    }

    public void setManagerUser(final String managerUser)
    {
        this.managerUser = managerUser;
    }

    public String getManagerPassword()
    {
        return managerPassword;
    }

    public void setManagerPassword(final String managerPassword)
    {
        this.managerPassword = managerPassword;
    }

    public String getAgentIp()
    {
        return agentIp;
    }

    public void setAgentIp(final String agentIp)
    {
        this.agentIp = agentIp;
    }

    public Integer getAgentPort()
    {
        return agentPort;
    }

    public void setAgentPort(final Integer agentPort)
    {
        this.agentPort = agentPort;
    }

    public String getAgentUser()
    {
        return agentUser;
    }

    public void setAgentUser(final String agentUser)
    {
        this.agentUser = agentUser;
    }

    public String getAgentPassword()
    {
        return agentPassword;
    }

    public void setAgentPassword(final String agentPassword)
    {
        this.agentPassword = agentPassword;
    }
}
