/**
 * Licensed to Abiquo Holdings S.L. (Abiquo) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package com.abiquo.server.core.infrastructure.network;

import javax.validation.constraints.NotNull;
import javax.xml.bind.annotation.XmlElement;
import javax.xml.bind.annotation.XmlRootElement;
import javax.xml.bind.annotation.XmlType;

import com.abiquo.model.enumerator.NetworkType;
import com.abiquo.model.transport.SingleResourceTransportDto;

@XmlRootElement(name = "network")
@XmlType(propOrder = {"id", "name", "address", "mask", "gateway", "primaryDNS", "secondaryDNS",
"sufixDNS", "tag", "defaultNetwork", "unmanaged", "type", "dhcpOptions", "ipv6", "strict",
"providerId"})
public class VLANNetworkDto extends SingleResourceTransportDto
{

    private static final long serialVersionUID = 7128354911707490770L;

    private static final String TYPE = "application/vnd.abiquo.vlan";

    public static final String MEDIA_TYPE = TYPE + "+" + JSON;

    public static final String SHORT_MEDIA_TYPE_JSON = MEDIA_TYPE;

    public static final String SHORT_MEDIA_TYPE_XML = TYPE + "+" + XML;

    public static final String MEDIA_TYPE_XML = TYPE + "+" + XML + VERSION_PARAM + API_VERSION;

    public static final String MEDIA_TYPE_JSON = MEDIA_TYPE + VERSION_PARAM + API_VERSION;

    public static final String MEDIA_TYPE_XML_22 = TYPE + "+" + XML + VERSION_PARAM
        + API_VERSION_22;

    private Integer id;

    private String name;

    private Integer tag;

    private String gateway;

    private String address;

    private Integer mask;

    private String primaryDNS;

    private String secondaryDNS;

    private String sufixDNS;

    private Boolean defaultNetwork;

    private Boolean unmanaged;

    private NetworkType type;

    private DhcpOptionsDto dhcpOptions;

    /** The identifier used by public cloud provider. */
    private String providerId;

    // ************** Primitive attributes because default value is false **************

    private boolean ipv6;

    private boolean strict;

    /**
     * @return Network Address of the VLAN
     */
    @NotNull
    public String getAddress()
    {
        return address;
    }

    @Override
    public String getBaseMediaType()
    {
        return MEDIA_TYPE;
    }

    /**
     * @return Return if it is used as Default Network.
     */
    public Boolean getDefaultNetwork()
    {
        return defaultNetwork;
    }

    public DhcpOptionsDto getDhcpOptions()
    {
        return dhcpOptions;
    }

    /**
     * @return Gateway of the VLAN
     */
    @NotNull
    public String getGateway()
    {
        return gateway;
    }

    /**
     * @return Identifier of the entity
     */
    public Integer getId()
    {
        return id;
    }

    /**
     * @return Numerical value of the VLAN mask
     */
    @NotNull
    public Integer getMask()
    {
        return mask;
    }

    @Override
    public String getMediaType()
    {
        return MEDIA_TYPE_JSON;
    }

    /**
     * @return Name of the VLAN
     */
    @NotNull
    @XmlElement(defaultValue = "")
    public String getName()
    {
        return name;
    }

    /**
     * @return Primary DNS address
     */
    public String getPrimaryDNS()
    {
        return primaryDNS;
    }

    /**
     * @return Secondary DNS value
     */
    public String getSecondaryDNS()
    {
        return secondaryDNS;
    }

    /**
     * @return Suffix DNS value
     */
    public String getSufixDNS()
    {
        return sufixDNS;
    }

    /**
     * @return Tag of the VLAN. It will need an input value for PUBLIC, EXTERNAL and UNMANAGED
     *         networks
     */
    public Integer getTag()
    {
        return tag;
    }

    /**
     * @return Kind of network: PUBLIC, INTERNAL, EXTERNAL and UNMANAGED
     */
    public NetworkType getType()
    {
        return type;
    }

    /**
     * @return Return if the VLAN is unmanaged
     */
    public Boolean getUnmanaged()
    {
        return unmanaged;
    }

    /**
     * @return Is this network IPv6?
     */
    public boolean isIpv6()
    {
        return ipv6;
    }

    /**
     * @return Is this network strict?
     */
    public boolean isStrict()
    {
        return strict;
    }

    public void setAddress(final String address)
    {
        this.address = address;
    }

    public void setDefaultNetwork(final Boolean defaultNetwork)
    {
        this.defaultNetwork = defaultNetwork;
    }

    public void setDhcpOptions(final DhcpOptionsDto dhcpOptions)
    {
        this.dhcpOptions = dhcpOptions;
    }

    public void setGateway(final String gateway)
    {
        this.gateway = gateway;
    }

    public void setId(final Integer id)
    {
        this.id = id;
    }

    public void setMask(final Integer mask)
    {
        this.mask = mask;
    }

    public void setName(final String name)
    {
        this.name = name;
    }

    public void setPrimaryDNS(final String primaryDNS)
    {
        this.primaryDNS = primaryDNS;
    }

    public void setSecondaryDNS(final String secondaryDNS)
    {
        this.secondaryDNS = secondaryDNS;
    }

    public void setSufixDNS(final String sufixDNS)
    {
        this.sufixDNS = sufixDNS;
    }

    public void setTag(final Integer tag)
    {
        this.tag = tag;
    }

    public void setType(final NetworkType type)
    {
        this.type = type;
    }

    public void setUnmanaged(final Boolean unmanaged)
    {
        this.unmanaged = unmanaged;
    }

    public void setIpv6(final boolean ipv6)
    {
        this.ipv6 = ipv6;
    }

    public void setStrict(final boolean strict)
    {
        this.strict = strict;
    }

    /**
     * Sets the identifier used by public cloud provider
     */
    public String getProviderId()
    {
        return providerId;
    }

    /**
     * Sets the identifier used by public cloud provider
     * 
     * @param providerId
     */
    public void setProviderId(final String providerId)
    {
        this.providerId = providerId;
    }

}
