package com.addc.commons.jmx;

import java.io.IOException;
import java.rmi.NoSuchObjectException;
import java.rmi.RemoteException;
import java.rmi.registry.LocateRegistry;
import java.rmi.registry.Registry;
import java.rmi.server.UnicastRemoteObject;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import javax.management.remote.JMXAuthenticator;
import javax.management.remote.JMXConnectorServer;
import javax.management.remote.JMXConnectorServerFactory;
import javax.management.remote.JMXServiceURL;
import javax.management.remote.MBeanServerForwarder;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.addc.commons.alerts.Notifier;
import com.addc.commons.configuration.ConfigurationException;
import com.addc.commons.jmx.configuration.JMXConfig;
import com.addc.commons.shutdown.Stoppable;
import com.addc.commons.slp.NetworkManager;
import com.addc.commons.slp.NetworkManagerImpl;
import com.addc.commons.slp.SLPRegistrarException;
import com.addc.commons.slp.SLPRegistrarThread;
import com.addc.commons.slp.ServiceLocationException;
import com.addc.commons.slp.ServiceLocationManager;
import com.addc.commons.slp.ServiceURL;

/**
 * The JMXAdaptorFactory creates and starts an RMI ConnectorServer to a new
 * MBean Server (NOT the platform MBean Server) on a single port using a
 * predefined JMXAutheticator and JMXAccessController and optionally registers
 * the URL with a Service Location Protocol Daemon.
 */
public class JMXAdaptorFactory implements Stoppable {
    /**
     * The default configuration file name
     */
    public static final String DEFAULT_CFG_FILE= "classpath:jmx.properties";
    private static final Logger LOGGER= LoggerFactory.getLogger(JMXAdaptorFactory.class);
    private final JMXConfig config;
    private JMXAuthenticator authenticator;
    private MBeanServerForwarder accessController;
    private Notifier notifier;
    private JMXConnectorServer connectorServer;
    private JMXServiceURL serviceUrl;
    private SLPRegistrarThread slpRegistrar;
    private Registry registry;

    /**
     * Create a new JMXAdaptorFactory using the default configuration file name
     * 
     * @throws IOException
     *             If there is an I/O error
     * @throws ConfigurationException
     *             If there is a problem with the configuration
     */
    public JMXAdaptorFactory() throws IOException, ConfigurationException {
        this(DEFAULT_CFG_FILE);
    }

    /**
     * Create a new JMXAdaptorFactory
     * 
     * @param configFile
     *            The name of the configuration file
     * @throws IOException
     *             If there is an I/O error
     * @throws ConfigurationException
     *             If there is a problem with the configuration
     */
    public JMXAdaptorFactory(String configFile) throws IOException, ConfigurationException {
        this(new JMXConfig(configFile));
    }

    /**
     * Create a new JMXAdaptorFactory with no notifier
     * 
     * @param config
     *            The JMX configuration to use
     */
    public JMXAdaptorFactory(JMXConfig config) {
        this.config= config;
    }

    /**
     * Initialize the adaptor, create and wrap the MBeanServer with the access
     * controller, attach the adaptor and start the adaptor.
     *
     * @throws MBeanHelperException
     *             If there is a problem accessing the MBean Server
     */
    public void initialize() throws MBeanHelperException {
        initAccessCtl();

        initRegistry();
        Map<String, Object> env= new ConcurrentHashMap<>();
        env.put(JMXConnectorServer.AUTHENTICATOR, authenticator);
        LOGGER.info("Added the authenticator {} to the environment.", authenticator.getClass().getSimpleName());
        serviceUrl= config.getServiceUrl();
        LOGGER.info("Publish MBean Server on {}.", serviceUrl);

        initConnectorServer(env);

        initSlpRegistrar();
    }

    @Override
    public void shutdown() {
        if ((connectorServer != null) && connectorServer.isActive()) {
            if (slpRegistrar != null) {
                slpRegistrar.shutdown();
            }
            try {
                LOGGER.info("Stopping the JMX Connector Server..");
                connectorServer.stop();
                LOGGER.info("JMX Connector Server has terminated.");
            } catch (IOException e) {
                LOGGER.error("Error stopping JMX Connection Servcer", e);
            }
            try {
                UnicastRemoteObject.unexportObject(registry, true);
            } catch (NoSuchObjectException e) {
                LOGGER.error("Error closing down registry", e);
            }
        }
    }

    /**
     * Get the connectorServer
     * 
     * @return the connectorServer
     */
    public JMXConnectorServer getConnectorServer() {
        return connectorServer;
    }

    /**
     * Get the serviceUrl
     * 
     * @return the serviceUrl
     */
    public JMXServiceURL getServiceUrl() {
        return serviceUrl;
    }

    /**
     * Get the authenticator
     * 
     * @return the authenticator
     */
    public JMXAuthenticator getAuthenticator() {
        return authenticator;
    }

    /**
     * Set the authenticator
     * 
     * @param authenticator
     *            the authenticator to set
     */
    public void setAuthenticator(JMXAuthenticator authenticator) {
        this.authenticator= authenticator;
    }

    /**
     * Get the accessController
     * 
     * @return the accessController
     */
    public MBeanServerForwarder getAccessController() {
        return accessController;
    }

    /**
     * Set the accessController
     * 
     * @param accessController
     *            the accessController to set
     */
    public void setAccessController(MBeanServerForwarder accessController) {
        this.accessController= accessController;
    }

    /**
     * Get the notifier
     * 
     * @return the notifier
     */
    public Notifier getNotifier() {
        return notifier;
    }

    /**
     * Set the notifier
     * 
     * @param notifier
     *            the notifier to set
     */
    public void setNotifier(Notifier notifier) {
        this.notifier= notifier;
        if (slpRegistrar != null) {
            slpRegistrar.setNotifier(notifier);
        }
    }

    /**
     * Get the config
     * 
     * @return the config
     */
    public JMXConfig getConfig() {
        return config;
    }

    /**
     * Get the slpRegistrar (may be null)
     *
     * @return the slpRegistrar
     */
    public SLPRegistrarThread getSlpRegistrar() {
        return slpRegistrar;
    }

    private void initAccessCtl() throws MBeanHelperException {
        if (authenticator == null) {
            if (config.getJmxAuthenticator() == null) {
                throw new MBeanHelperException(
                        "Either the user/password file must be set in the configuration or a JMXAutheticator must be set");
            }
            authenticator= config.getJmxAuthenticator();
        }
        if (accessController == null) {
            if (config.getAccessController() == null) {
                throw new MBeanHelperException(
                        "Either the access file must be set in the configuration or a JMXAccessController must be set");
            }
            accessController= config.getAccessController();
        }
        accessController.setMBeanServer(config.getMBeanServer());
        LOGGER.info("Wrapped the MBeanServer with the access controller {}.",
                accessController.getClass().getSimpleName());
    }

    private void initSlpRegistrar() {
        if (config.isSlpEnabled()) {
            try {
                NetworkManager netManager= new NetworkManagerImpl(config.getSlpConfig());
                ServiceLocationManager slm= new ServiceLocationManager(config.getSlpConfig(), netManager);
                ServiceURL surl= new ServiceURL(config.getSlpConfig(), serviceUrl.toString(),
                        config.getSlpLeaseTimeSecs());
                slpRegistrar= new SLPRegistrarThread(slm, surl, config.getSlpAgentName(), notifier);
                slpRegistrar.start();
            } catch (SLPRegistrarException | ServiceLocationException e) {
                LOGGER.error("Failed to start the SLP Registrar", e);
            }
        }
    }

    private void initConnectorServer(Map<String, Object> env) throws MBeanHelperException {
        try {
            connectorServer= JMXConnectorServerFactory.newJMXConnectorServer(serviceUrl, env, accessController);
            LOGGER.info("Created JMX Connector Server {}", connectorServer.getClass().getSimpleName());
        } catch (IOException e) {
            LOGGER.error("Error creating JMX Connector Server", e);
            throw new MBeanHelperException("Error creating JMX Connector Server", e);
        }
        try {
            connectorServer.start();
            LOGGER.info("Started Connector {}", connectorServer.getClass().getSimpleName());
        } catch (IOException e) {
            LOGGER.error("Error starting JMX Connector Server", e);
            throw new MBeanHelperException("Error starting JMX Connector Server", e);
        }
    }

    private void initRegistry() throws MBeanHelperException {
        System.setProperty("java.rmi.server.randomIDs", "true");
        System.setProperty("java.rmi.server.hostname", config.getRmiHostname());
        int port= config.getRegistryPort();
        try {
            registry= LocateRegistry.createRegistry(port);
        } catch (RemoteException e) {
            LOGGER.error("Failed to create the registry", e);
            throw new MBeanHelperException("Failed to create the registry", e);
        }

        LOGGER.info("Created registry on port {}", config.getRegistryPort());
    }

}
