package com.addc.commons.jmx;

import java.lang.management.ManagementFactory;

import javax.management.MBeanServer;
import javax.management.MBeanServerFactory;
import javax.management.MalformedObjectNameException;
import javax.management.ObjectName;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * The MBeanServerHelper supplies helper methods for accessing either the
 * platform MBeanServer (default) or a new MBeanServer if the System Property
 * <code>jmx.helper.platform.mbs</code> is set to false;
 * 
 */
public final class MBeanServerHelper {
    public static final String KEY_PLATFORM_MBS= "jmx.helper.platform.mbs";
    private static final Logger LOGGER= LoggerFactory.getLogger(MBeanServerHelper.class);
    private static MBeanServerHelper instance;
    private MBeanServer mbeanServer;

    @SuppressWarnings("PMD.AvoidSynchronizedAtMethodLevel")
    static synchronized void clear() {
        instance= null;
    }

    /**
     * Get the singleton instance of this class.
     * 
     * @return the singleton instance of this class.
     */
    @SuppressWarnings("PMD.AvoidSynchronizedAtMethodLevel")
    public static synchronized MBeanServerHelper getInstance() {
        if (instance == null) {
            instance= new MBeanServerHelper();
        }
        return instance;
    }

    /**
     * Get a reference to the MBeanServer.
     * 
     * @return the MBeanServer.
     */
    public MBeanServer getMBeanServer() {
        return mbeanServer;
    }

    /**
     * Register an Object with the MBeanServer
     * 
     * @param mbean
     *            A Standard MBean
     * @param name
     *            The object name
     * @return <code>true</code> if successful
     */
    public boolean registerStandardMBean(Object mbean, String name) {
        try {
            ObjectName oname= new ObjectName(name);
            return registerStandardMBean(mbean, oname);
        } catch (MalformedObjectNameException e) {
            LOGGER.error("Failed to register MBean {}", name, e);
        }
        return false;
    }

    /**
     * Register an Object with the MBeanServer
     * 
     * @param mbean
     *            A Standard MBean
     * @param name
     *            The object name
     * @return <code>true</code> if successful
     */
    public boolean registerStandardMBean(Object mbean, ObjectName name) {
        try {
            mbeanServer.registerMBean(mbean, name);
        } catch (Exception e) {
            LOGGER.error("Failed to register MBean {}", name, e);
            return false;
        }
        return true;
    }

    /**
     * Register a JmxBean with the MBeanServer
     * 
     * @param mbean
     *            An object implementing JmxBean
     * @param name
     *            The object name
     * @return <code>true</code> if successful
     */
    public boolean registerMBean(JmxBean mbean, String name) {
        try {
            ObjectName oname= new ObjectName(name);
            return registerMBean(mbean, oname);
        } catch (MalformedObjectNameException e) {
            LOGGER.error("Failed to register MBean {}", name, e);
        }
        return false;
    }

    /**
     * Register a JmxBean with the MBeanServer
     * 
     * @param mbean
     *            An object implementing JmxBean
     * @param name
     *            The object name
     * @return <code>true</code> if successful
     */
    public boolean registerMBean(JmxBean mbean, ObjectName name) {
        try {
            mbeanServer.registerMBean(mbean, name);
        } catch (Exception e) {
            LOGGER.error("Failed to register MBean {}", name, e);
            return false;
        }
        return true;
    }

    /**
     * Un-register a JmxBean with the MBean Server
     * 
     * @param mbean
     *            An object implementing JmxBean that has been previously
     *            registered.
     * @return <code>true</code> if successful
     */
    public boolean unregisterMBean(JmxBean mbean) {
        return unregisterMBean(mbean.getObjectName());
    }

    /**
     * Un-register an MBean with the MBean Server
     * 
     * @param name
     *            The object name.
     * @return <code>true</code> if successful
     */
    public boolean unregisterMBean(String name) {
        try {
            ObjectName objectname= new ObjectName(name);
            return unregisterMBean(objectname);
        } catch (MalformedObjectNameException e) {
            LOGGER.error("Failed to create ObjectName {}", name, e);
        }
        return false;
    }

    /**
     * Un-register an MBean with the MBean Server
     * 
     * @param name
     *            The object name
     * @return <code>true</code> if successful
     */
    public boolean unregisterMBean(ObjectName name) {
        try {
            mbeanServer.unregisterMBean(name);
        } catch (Exception e) {
            LOGGER.error("Failed to unregister MBean {}", name, e);
            return false;
        }
        return true;
    }

    /**
     * Create the MBeanServerHelper and get the current platform MBeanServer if
     * available.
     */
    private MBeanServerHelper() {
        if (Boolean.valueOf(System.getProperty(KEY_PLATFORM_MBS, "true"))) {
            LOGGER.info("Using the Platform MBean Server");
            mbeanServer= ManagementFactory.getPlatformMBeanServer();
        } else {
            LOGGER.info("Creating a new MBean Server");
            mbeanServer= MBeanServerFactory.createMBeanServer();
        }
        LOGGER.info("Using MBean Server of type {}", mbeanServer.getClass().getName());
    }

}
