package com.addc.commons.jmx.auth;

import java.io.ObjectInputStream;
import java.security.AccessController;
import java.util.Set;

import javax.management.Attribute;
import javax.management.AttributeList;
import javax.management.AttributeNotFoundException;
import javax.management.InstanceAlreadyExistsException;
import javax.management.InstanceNotFoundException;
import javax.management.IntrospectionException;
import javax.management.InvalidAttributeValueException;
import javax.management.ListenerNotFoundException;
import javax.management.MBeanException;
import javax.management.MBeanInfo;
import javax.management.MBeanRegistrationException;
import javax.management.MBeanServer;
import javax.management.NotCompliantMBeanException;
import javax.management.NotificationFilter;
import javax.management.NotificationListener;
import javax.management.ObjectInstance;
import javax.management.ObjectName;
import javax.management.OperationsException;
import javax.management.QueryExp;
import javax.management.ReflectionException;
import javax.management.loading.ClassLoaderRepository;
import javax.management.remote.MBeanServerForwarder;

/**
 * The JMXAccessController supplies a wrapper for an MBEanServer that will check
 * the permissions for any call to the server
 */
@SuppressWarnings("PMD")
public abstract class JMXAccessController implements MBeanServerForwarder {
    private MBeanServer mbs;

    /**
     * Enumeration defining the access type
     */
    protected enum AccessType {
        READ, WRITE;
    }

    @Override
    public MBeanServer getMBeanServer() {
        return mbs;
    }

    @Override
    public void setMBeanServer(MBeanServer mbs) {
        if (mbs == null) {
            throw new IllegalArgumentException("Null MBeanServer");
        }
        if (this.mbs != null) {
            throw new IllegalArgumentException("MBeanServer object already initialized");
        }
        this.mbs= mbs;
    }

    /**
     * Actually check the access rights using the specific mechanism
     *
     * @param requiredAccess
     *            The required access
     * @throws SecurityException
     *             If the current user does not have the reuired access
     *             permission
     */
    protected abstract void checkAccess(AccessType requiredAccess) throws SecurityException;

    /**
     * Check if the caller can do read operations. This method does nothing if
     * so, otherwise throws SecurityException.
     */
    protected void checkRead() {
        checkAccess(AccessType.READ);
    }

    /**
     * Check if the caller can do write operations. This method does nothing if
     * so, otherwise throws SecurityException.
     */
    protected void checkWrite() {
        checkAccess(AccessType.WRITE);
    }

    /**
     * Check if the caller can create the named class. The default
     * implementation of this method calls {@link #checkWrite()}.
     * 
     * @param className
     *            Unused
     */
    protected void checkCreate(@SuppressWarnings("unused") String className) {
        checkAccess(AccessType.WRITE);
    }

    /**
     * Check if the caller can unregister the named MBean. The default
     * implementation of this method calls {@link #checkWrite()}.
     *
     * @param name
     *            The name of the object
     */
    protected void checkUnregister(@SuppressWarnings("unused") ObjectName name) {
        checkAccess(AccessType.WRITE);
    }

    /**
     * Call <code>checkRead()</code>, then forward this method to the wrapped
     * object.
     */
    @Override
    public void addNotificationListener(ObjectName name, NotificationListener listener, NotificationFilter filter,
            Object handback) throws InstanceNotFoundException {
        checkRead();
        getMBeanServer().addNotificationListener(name, listener, filter, handback);
    }

    /**
     * Call <code>checkRead()</code>, then forward this method to the wrapped
     * object.
     */
    @Override
    public void addNotificationListener(ObjectName name, ObjectName listener, NotificationFilter filter,
            Object handback) throws InstanceNotFoundException {
        checkRead();
        getMBeanServer().addNotificationListener(name, listener, filter, handback);
    }

    /**
     * Call <code>checkCreate(className)</code>, then forward this method to the
     * wrapped object.
     */
    @Override
    public ObjectInstance createMBean(String className, ObjectName name) throws ReflectionException,
            InstanceAlreadyExistsException, MBeanRegistrationException, MBeanException, NotCompliantMBeanException {
        checkCreate(className);
        SecurityManager sm= System.getSecurityManager();
        if (sm == null) {
            Object object= getMBeanServer().instantiate(className);
            checkClassLoader(object);
            return getMBeanServer().registerMBean(object, name);
        }
        return getMBeanServer().createMBean(className, name);
    }

    /**
     * Call <code>checkCreate(className)</code>, then forward this method to the
     * wrapped object.
     */
    @Override
    public ObjectInstance createMBean(String className, ObjectName name, Object params[], String signature[])
            throws ReflectionException, InstanceAlreadyExistsException, MBeanRegistrationException, MBeanException,
            NotCompliantMBeanException {
        checkCreate(className);
        SecurityManager sm= System.getSecurityManager();
        if (sm == null) {
            Object object= getMBeanServer().instantiate(className, params, signature);
            checkClassLoader(object);
            return getMBeanServer().registerMBean(object, name);
        }
        return getMBeanServer().createMBean(className, name, params, signature);
    }

    /**
     * Call <code>checkCreate(className)</code>, then forward this method to the
     * wrapped object.
     */
    @Override
    public ObjectInstance createMBean(String className, ObjectName name, ObjectName loaderName)
            throws ReflectionException, InstanceAlreadyExistsException, MBeanRegistrationException, MBeanException,
            NotCompliantMBeanException, InstanceNotFoundException {
        checkCreate(className);
        SecurityManager sm= System.getSecurityManager();
        if (sm == null) {
            Object object= getMBeanServer().instantiate(className, loaderName);
            checkClassLoader(object);
            return getMBeanServer().registerMBean(object, name);
        }
        return getMBeanServer().createMBean(className, name, loaderName);
    }

    /**
     * Call <code>checkCreate(className)</code>, then forward this method to the
     * wrapped object.
     */
    @Override
    public ObjectInstance createMBean(String className, ObjectName name, ObjectName loaderName, Object params[],
            String signature[]) throws ReflectionException, InstanceAlreadyExistsException, MBeanRegistrationException,
            MBeanException, NotCompliantMBeanException, InstanceNotFoundException {
        checkCreate(className);
        SecurityManager sm= System.getSecurityManager();
        if (sm == null) {
            Object object= getMBeanServer().instantiate(className, loaderName, params, signature);
            checkClassLoader(object);
            return getMBeanServer().registerMBean(object, name);
        }
        return getMBeanServer().createMBean(className, name, loaderName, params, signature);
    }

    /**
     * Call <code>checkRead()</code>, then forward this method to the wrapped
     * object.
     */
    @Override
    @Deprecated
    public ObjectInputStream deserialize(ObjectName name, byte[] data)
            throws InstanceNotFoundException, OperationsException {
        checkRead();
        return getMBeanServer().deserialize(name, data);
    }

    /**
     * Call <code>checkRead()</code>, then forward this method to the wrapped
     * object.
     */
    @Override
    @Deprecated
    public ObjectInputStream deserialize(String className, byte[] data)
            throws OperationsException, ReflectionException {
        checkRead();
        return getMBeanServer().deserialize(className, data);
    }

    /**
     * Call <code>checkRead()</code>, then forward this method to the wrapped
     * object.
     */
    @Override
    @Deprecated
    public ObjectInputStream deserialize(String className, ObjectName loaderName, byte[] data)
            throws InstanceNotFoundException, OperationsException, ReflectionException {
        checkRead();
        return getMBeanServer().deserialize(className, loaderName, data);
    }

    /**
     * Call <code>checkRead()</code>, then forward this method to the wrapped
     * object.
     */
    @Override
    public Object getAttribute(ObjectName name, String attribute)
            throws MBeanException, AttributeNotFoundException, InstanceNotFoundException, ReflectionException {
        checkRead();
        return getMBeanServer().getAttribute(name, attribute);
    }

    /**
     * Call <code>checkRead()</code>, then forward this method to the wrapped
     * object.
     */
    @Override
    public AttributeList getAttributes(ObjectName name, String[] attributes)
            throws InstanceNotFoundException, ReflectionException {
        checkRead();
        return getMBeanServer().getAttributes(name, attributes);
    }

    /**
     * Call <code>checkRead()</code>, then forward this method to the wrapped
     * object.
     */
    @Override
    @SuppressWarnings("PMD.UseProperClassLoader")
    public ClassLoader getClassLoader(ObjectName loaderName) throws InstanceNotFoundException {
        checkRead();
        return getMBeanServer().getClassLoader(loaderName);
    }

    /**
     * Call <code>checkRead()</code>, then forward this method to the wrapped
     * object.
     */
    @Override
    public ClassLoader getClassLoaderFor(ObjectName mbeanName) throws InstanceNotFoundException {
        checkRead();
        return getMBeanServer().getClassLoaderFor(mbeanName);
    }

    /**
     * Call <code>checkRead()</code>, then forward this method to the wrapped
     * object.
     */
    @Override
    public ClassLoaderRepository getClassLoaderRepository() {
        checkRead();
        return getMBeanServer().getClassLoaderRepository();
    }

    /**
     * Call <code>checkRead()</code>, then forward this method to the wrapped
     * object.
     */
    @Override
    public String getDefaultDomain() {
        checkRead();
        return getMBeanServer().getDefaultDomain();
    }

    /**
     * Call <code>checkRead()</code>, then forward this method to the wrapped
     * object.
     */
    @Override
    public String[] getDomains() {
        checkRead();
        return getMBeanServer().getDomains();
    }

    /**
     * Call <code>checkRead()</code>, then forward this method to the wrapped
     * object.
     */
    @Override
    public Integer getMBeanCount() {
        checkRead();
        return getMBeanServer().getMBeanCount();
    }

    /**
     * Call <code>checkRead()</code>, then forward this method to the wrapped
     * object.
     */
    @Override
    public MBeanInfo getMBeanInfo(ObjectName name)
            throws InstanceNotFoundException, IntrospectionException, ReflectionException {
        checkRead();
        return getMBeanServer().getMBeanInfo(name);
    }

    /**
     * Call <code>checkRead()</code>, then forward this method to the wrapped
     * object.
     */
    @Override
    public ObjectInstance getObjectInstance(ObjectName name) throws InstanceNotFoundException {
        checkRead();
        return getMBeanServer().getObjectInstance(name);
    }

    /**
     * Call <code>checkCreate(className)</code>, then forward this method to the
     * wrapped object.
     */
    @Override
    public Object instantiate(String className) throws ReflectionException, MBeanException {
        checkCreate(className);
        return getMBeanServer().instantiate(className);
    }

    /**
     * Call <code>checkCreate(className)</code>, then forward this method to the
     * wrapped object.
     */
    @Override
    public Object instantiate(String className, Object params[], String signature[])
            throws ReflectionException, MBeanException {
        checkCreate(className);
        return getMBeanServer().instantiate(className, params, signature);
    }

    /**
     * Call <code>checkCreate(className)</code>, then forward this method to the
     * wrapped object.
     */
    @Override
    public Object instantiate(String className, ObjectName loaderName)
            throws ReflectionException, MBeanException, InstanceNotFoundException {
        checkCreate(className);
        return getMBeanServer().instantiate(className, loaderName);
    }

    /**
     * Call <code>checkCreate(className)</code>, then forward this method to the
     * wrapped object.
     */
    @Override
    public Object instantiate(String className, ObjectName loaderName, Object params[], String signature[])
            throws ReflectionException, MBeanException, InstanceNotFoundException {
        checkCreate(className);
        return getMBeanServer().instantiate(className, loaderName, params, signature);
    }

    /**
     * Call <code>checkWrite()</code>, then forward this method to the wrapped
     * object.
     */
    @Override
    public Object invoke(ObjectName name, String operationName, Object params[], String signature[])
            throws InstanceNotFoundException, MBeanException, ReflectionException {
        checkWrite();
        checkMLetMethods(name, operationName);
        return getMBeanServer().invoke(name, operationName, params, signature);
    }

    /**
     * Call <code>checkRead()</code>, then forward this method to the wrapped
     * object.
     */
    @Override
    public boolean isInstanceOf(ObjectName name, String className) throws InstanceNotFoundException {
        checkRead();
        return getMBeanServer().isInstanceOf(name, className);
    }

    /**
     * Call <code>checkRead()</code>, then forward this method to the wrapped
     * object.
     */
    @Override
    public boolean isRegistered(ObjectName name) {
        checkRead();
        return getMBeanServer().isRegistered(name);
    }

    /**
     * Call <code>checkRead()</code>, then forward this method to the wrapped
     * object.
     */
    @Override
    public Set<ObjectInstance> queryMBeans(ObjectName name, QueryExp query) {
        checkRead();
        return getMBeanServer().queryMBeans(name, query);
    }

    /**
     * Call <code>checkRead()</code>, then forward this method to the wrapped
     * object.
     */
    @Override
    public Set<ObjectName> queryNames(ObjectName name, QueryExp query) {
        checkRead();
        return getMBeanServer().queryNames(name, query);
    }

    /**
     * Call <code>checkWrite()</code>, then forward this method to the wrapped
     * object.
     */
    @Override
    public ObjectInstance registerMBean(Object object, ObjectName name)
            throws InstanceAlreadyExistsException, MBeanRegistrationException, NotCompliantMBeanException {
        checkWrite();
        return getMBeanServer().registerMBean(object, name);
    }

    /**
     * Call <code>checkRead()</code>, then forward this method to the wrapped
     * object.
     */
    @Override
    public void removeNotificationListener(ObjectName name, NotificationListener listener)
            throws InstanceNotFoundException, ListenerNotFoundException {
        checkRead();
        getMBeanServer().removeNotificationListener(name, listener);
    }

    /**
     * Call <code>checkRead()</code>, then forward this method to the wrapped
     * object.
     */
    @Override
    public void removeNotificationListener(ObjectName name, NotificationListener listener, NotificationFilter filter,
            Object handback) throws InstanceNotFoundException, ListenerNotFoundException {
        checkRead();
        getMBeanServer().removeNotificationListener(name, listener, filter, handback);
    }

    /**
     * Call <code>checkRead()</code>, then forward this method to the wrapped
     * object.
     */
    @Override
    public void removeNotificationListener(ObjectName name, ObjectName listener)
            throws InstanceNotFoundException, ListenerNotFoundException {
        checkRead();
        getMBeanServer().removeNotificationListener(name, listener);
    }

    /**
     * Call <code>checkRead()</code>, then forward this method to the wrapped
     * object.
     */
    @Override
    public void removeNotificationListener(ObjectName name, ObjectName listener, NotificationFilter filter,
            Object handback) throws InstanceNotFoundException, ListenerNotFoundException {
        checkRead();
        getMBeanServer().removeNotificationListener(name, listener, filter, handback);
    }

    /**
     * Call <code>checkWrite()</code>, then forward this method to the wrapped
     * object.
     */
    @Override
    public void setAttribute(ObjectName name, Attribute attribute) throws InstanceNotFoundException,
            AttributeNotFoundException, InvalidAttributeValueException, MBeanException, ReflectionException {
        checkWrite();
        getMBeanServer().setAttribute(name, attribute);
    }

    /**
     * Call <code>checkWrite()</code>, then forward this method to the wrapped
     * object.
     */
    @Override
    public AttributeList setAttributes(ObjectName name, AttributeList attributes)
            throws InstanceNotFoundException, ReflectionException {
        checkWrite();
        return getMBeanServer().setAttributes(name, attributes);
    }

    /**
     * Call <code>checkUnregister()</code>, then forward this method to the
     * wrapped object.
     */
    @Override
    public void unregisterMBean(ObjectName name) throws InstanceNotFoundException, MBeanRegistrationException {
        checkUnregister(name);
        getMBeanServer().unregisterMBean(name);
    }

    // ----------------
    // PRIVATE METHODS
    // ----------------

    private void checkClassLoader(Object object) {
        if (object instanceof ClassLoader) {
            throw new SecurityException("Access denied! Creating an " + "MBean that is a ClassLoader "
                    + "is forbidden unless a security " + "manager is installed.");
        }
    }

    private void checkMLetMethods(ObjectName name, String operation) throws InstanceNotFoundException {
        // Check if security manager installed
        SecurityManager sm= System.getSecurityManager();
        if (sm != null) {
            return;
        }
        // Check for addURL and getMBeansFromURL methods
        if (!"addURL".equals(operation) && !"getMBeansFromURL".equals(operation)) {
            return;
        }
        // Check if MBean is instance of MLet
        if (!getMBeanServer().isInstanceOf(name, "javax.management.loading.MLet")) {
            return;
        }
        // Throw security exception
        if ("addURL".equals(operation)) { // addURL
            throw new SecurityException(
                    "Access denied! MLet method addURL cannot be invoked unless a security manager is installed.");
        }
        // getMBeansFromURL
        // Whether or not calling getMBeansFromURL is allowed is controlled
        // by the value of the "jmx.remote.x.mlet.allow.getMBeansFromURL"
        // system property. If the value of this property is true, calling
        // the MLet's getMBeansFromURL method is allowed. The default value
        // for this property is false.
        final String propName= "jmx.remote.x.mlet.allow.getMBeansFromURL";
        GetPropertyAction propAction= new GetPropertyAction(propName);
        String propValue= AccessController.doPrivileged(propAction);
        boolean allowGetMBeansFromURL= "true".equalsIgnoreCase(propValue);
        if (!allowGetMBeansFromURL) {
            throw new SecurityException("Access denied! MLet method getMBeansFromURL cannot be invoked unless a "
                    + "security manager is installed or the system property "
                    + "-Djmx.remote.x.mlet.allow.getMBeansFromURL=true is specified.");
        }
    }

}
