package com.addc.commons.jmx.auth;

import java.security.AccessControlContext;
import java.security.AccessController;
import java.security.PrivilegedAction;
import java.util.HashSet;
import java.util.Set;

import javax.security.auth.Subject;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.addc.commons.Constants;
import com.addc.commons.i18n.I18nTextFactory;
import com.addc.commons.i18n.Translator;

/**
 * The JMXFileAccessController supplies a simple access controller that only
 * understands JmxAdmin and JmxMonitor access permissions. JmxAdmin can invoke
 * operations and setters on the target MBeanServer and JmxMonitor has simple
 * read only access.
 */
public class JMXFileAccessController extends JMXAccessController {
    private static final Logger LOGGER= LoggerFactory.getLogger(JMXFileAccessController.class);
    private final Translator translator= I18nTextFactory.getTranslator(Constants.BASENAME);

    private final String adminRole;
    private final String monitorRole;

    /**
     * Create a new JMXFileAccessController
     * @param adminRole The admin role name
     * @param monitorRole The monitor role name
     */
    public JMXFileAccessController(String adminRole, String monitorRole) {
        this.adminRole= adminRole;
        this.monitorRole= monitorRole;
    }

    @Override
    protected void checkAccess(AccessType requiredAccess) {
        final AccessControlContext acc= AccessController.getContext();
        final Subject subject= AccessController.doPrivileged(new PrivilegedAction<Subject>() {
            @Override
            public Subject run() {
                return Subject.getSubject(acc);
            }
        });
        if (subject == null) {
            // We will always come through here at logon
            LOGGER.debug("No current subject");
            return;
        }
        LOGGER.debug("Check subject {}", subject.getPrincipals());
        Set<String> roles= new HashSet<>();
        Set<Object> pubCreds= subject.getPublicCredentials();
        for (Object object : pubCreds) {
            roles.add(object.toString());
        }
        if ((requiredAccess == AccessType.READ) && (roles.contains(adminRole) || roles.contains(monitorRole))) {
            return;
        } else if (roles.contains(adminRole)) {
            return;
        }
        LOGGER.warn("{} is denied access for the current {} operation", subject.getPrincipals(), requiredAccess);
        throw new SecurityException(translator.translate(Constants.ERROR_ACCESS_DENIED));
    }

}
