package com.addc.commons.jmx.auth;

import java.io.IOException;
import java.security.GeneralSecurityException;
import java.util.Collections;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Properties;
import java.util.concurrent.ConcurrentHashMap;

import javax.management.remote.JMXAuthenticator;
import javax.management.remote.JMXPrincipal;
import javax.security.auth.Subject;

import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.addc.commons.Constants;
import com.addc.commons.i18n.I18nTextFactory;
import com.addc.commons.i18n.Translator;
import com.addc.commons.passwd.PasswordEncryptor;
import com.addc.commons.properties.PropertiesLoader;

/**
 * The JMXPropertiesAuthenticator supplies a JMXAuthenticator for users.properties files
 * with same format as Spring security: user=password[,role[,role[,...]]
 * The passwords can be encrypted with the PasswordEncryptor class. When authenticated the
 * resulting Subject has the list of roles set in the public credentials set. if should be
 * used in conjunction with the {@link JMXFileAccessController}
 */
public class JMXPropertiesAuthenticator implements JMXAuthenticator {
    private static final Logger LOGGER= LoggerFactory.getLogger(JMXPropertiesAuthenticator.class);
    static final String LOGIN_CONFIG_NAME= "JMXPropertiesAuthenticator";
    private final Map<String, UserDetails> userDetails= new ConcurrentHashMap<>();
    private final PasswordEncryptor cipher= new PasswordEncryptor();
    private final Translator translator= I18nTextFactory.getTranslator(Constants.BASENAME);

    /**
     * Create a new JMXPropertiesAuthenticator
     * 
     * @param usersFileName
     *            The name of the file with user/password/roles info
     * @throws IOException
     *             If the file cannot be loaded
     */
    public JMXPropertiesAuthenticator(String usersFileName) throws IOException {
        initialize(PropertiesLoader.getInstance().load(usersFileName));
    }

    private void initialize(Properties props) {
        for (Entry<Object, Object> entry : props.entrySet()) {
            String user= (String) entry.getKey();
            UserDetails details= new UserDetails(user, (String) entry.getValue());
            userDetails.put(user, details);
        }
    }

    @Override
    public Subject authenticate(Object credentials) {
        String[] creds= getCredentialsStrings(credentials);
        validateCredentialsStrings(creds);

        if (!userDetails.containsKey(creds[0])) {
            String msg= translator.translate(Constants.ERROR_CREDS_UNKNOWN_USER, creds[0]);
            LOGGER.warn(msg);
            throw new SecurityException(msg);
        }
        UserDetails details= userDetails.get(creds[0]);
        
        checkPasswordMatch(creds, details);

        LOGGER.info("Authentication {}: success", creds[0]);
        return new Subject(true, Collections.singleton(new JMXPrincipal(creds[0])), details.getRoles(),
                Collections.EMPTY_SET);

    }

    private void checkPasswordMatch(String[] creds, UserDetails details) throws SecurityException {
        boolean passwdMatch= false;
        try {
            passwdMatch= cipher.decrypt(details.getPasswd()).equals(creds[1]);
        } catch (GeneralSecurityException e) {
            LOGGER.error("Failed to decrypt password", e);
        }

        if (!passwdMatch) {
            String msg= translator.translate(Constants.ERROR_CREDS_INVALID_PWD, creds[0]);
            LOGGER.warn(msg);
            throw new SecurityException(msg);
        }
    }

    private void validateCredentialsStrings(String[] creds) throws SecurityException {
        if (creds.length != 2) {
            LOGGER.error("Credentials must be array of 2 strings not {}", creds.length);
            throw new SecurityException(translator.translate(Constants.ERROR_INVALID_CREDS));
        }
        if (StringUtils.isBlank(creds[0])) {
            LOGGER.error("Credentials must contains a user user name");
            throw new SecurityException(translator.translate(Constants.ERROR_INVALID_CREDS));
        }
        if (StringUtils.isBlank(creds[1])) {
            LOGGER.error("Credentials must contain a password");
            throw new SecurityException(translator.translate(Constants.ERROR_INVALID_CREDS));
        }
    }

    private String[] getCredentialsStrings(Object credentials) throws SecurityException {
        if (credentials == null) {
            LOGGER.error("Credentials required cannot be null");
            throw new SecurityException(translator.translate(Constants.ERROR_INVALID_CREDS));
        }
        if (!(credentials instanceof String[])) {
            LOGGER.error("Invalid credential type must be String[]");
            throw new SecurityException(translator.translate(Constants.ERROR_INVALID_CREDS));
        }

        return (String[]) credentials;
    }

}
