package com.addc.commons.jmx.configuration;

import java.io.IOException;
import java.net.MalformedURLException;
import java.util.HashSet;
import java.util.Properties;
import java.util.Set;

import javax.management.MBeanAttributeInfo;
import javax.management.MBeanInfo;
import javax.management.MBeanServer;
import javax.management.StandardMBean;
import javax.management.remote.JMXAuthenticator;
import javax.management.remote.JMXServiceURL;

import com.addc.commons.Constants;
import com.addc.commons.configuration.ConfigurationException;
import com.addc.commons.jmx.MBeanServerHelper;
import com.addc.commons.jmx.auth.JMXAccessController;
import com.addc.commons.jmx.auth.JMXFileAccessController;
import com.addc.commons.jmx.auth.JMXPropertiesAuthenticator;
import com.addc.commons.properties.BoundsFactory;
import com.addc.commons.properties.PropertiesLoader;
import com.addc.commons.properties.PropertiesParser;
import com.addc.commons.slp.configuration.SLPConfig;

/**
 * 
 * The JMXConfig class supplies configuration to run a MBeanServer with an RMI
 * connection on a single port. This configuration will also define the
 * parameters to open a lease for the BeanServer with an SLP daemon and will
 * support RMI over SSL using the {@link SSLConfig} class for configuration
 * using property keys with the prefix <b>jmx.</b>.
 */
public class JMXConfig extends StandardMBean implements IMbJMXConfig {
    private static final String MBEAN_NAME= "addc:Type=JMX,Id=JMXConfig";
    private Integer registryPort;
    private boolean authFileBased;
    private String rmiHostname;
    private String usersFileName;
    private SLPConfig slpConfig;
    private int slpLeaseTimeSecs;
    private String slpAgentName;
    private JMXAuthenticator jmxAuthenticator;
    private JMXAccessController accessController;
    private JMXServiceURL serviceUrl;
    private String adminRole;
    private String monitorRole;
    private final MBeanServer mbeanServer;

    /**
     * Create a new JMXConfig from a file
     * 
     * @param propsFileName
     *            The properties file name (may start with classpath:
     * @throws IOException
     *             If the file cannot be loaded
     * @throws ConfigurationException
     *             If there are any configuration errors
     */
    public JMXConfig(String propsFileName) throws IOException, ConfigurationException {
        this(PropertiesLoader.getInstance().load(propsFileName));
    }

    /**
     * Create a new JMXConfig from a properties object
     * 
     * @param properties
     *            The properties object to use
     * @throws ConfigurationException
     *             If there are any configuration errors
     */
    public JMXConfig(Properties properties) throws ConfigurationException {
        super(IMbJMXConfig.class, false);
        Set<String> errors= new HashSet<>();
        initialize(new PropertiesParser(properties, errors));
        if (!errors.isEmpty()) {
            throw new ConfigurationException(errors);
        }
        mbeanServer= MBeanServerHelper.getInstance().getMBeanServer();
        MBeanServerHelper.getInstance().registerStandardMBean(this, MBEAN_NAME);
    }

    /**
     * Create a new JMXConfig for use within a higher level configuration
     * 
     * @param parser
     *            the properties parser
     */
    private void initialize(PropertiesParser parser) {
        registryPort= parser.parsePort(JMXConfigConstants.RMI_REGISTRY_PORT);
        rmiHostname= parser.parseString(JMXConfigConstants.RMI_HOSTNAME, Constants.localHostName);
        adminRole= parser.parseString(JMXConfigConstants.JMX_ADMIN_ROLE, JMXConfigConstants.DEF_JMX_ADMIN_ROLE);
        monitorRole= parser.parseString(JMXConfigConstants.JMX_MONITOR_ROLE, JMXConfigConstants.DEF_JMX_MONITOR_ROLE);
        authFileBased= parser.parseBoolean(JMXConfigConstants.JMX_FILE_AUTH_ENABLED, true);
        if (authFileBased) {
            usersFileName= parser.parseString(JMXConfigConstants.JMX_USERS_FILE, JMXConfigConstants.DEF_JMX_USERS_FILE);
            if (parser.getParserErrors().isEmpty()) {
                try {
                    jmxAuthenticator= new JMXPropertiesAuthenticator(usersFileName);
                } catch (IOException e) {
                    parser.getParserErrors().add("Error creating authenticator - " + e.getMessage());
                }
                accessController= new JMXFileAccessController(adminRole, monitorRole);
            }
        }
        slpConfig= new SLPConfig(parser, JMXConfigConstants.JMX_PREFIX);
        if (slpConfig.isEnabled()) {
            slpLeaseTimeSecs= parser.parseInteger(JMXConfigConstants.SLP_LEASE_TIME, BoundsFactory.getIntBoundsGte(10),
                    null);
            slpAgentName= parser.parseString(JMXConfigConstants.SLP_AGENT_NAME);
        }
        StringBuilder sb= new StringBuilder();
        sb.append("service:jmx:rmi://").append(rmiHostname).append(':').append(registryPort);
        sb.append("/jndi/rmi://").append(rmiHostname).append(':').append(registryPort);
        sb.append("/jmxrmi");
        try {
            serviceUrl= new JMXServiceURL(sb.toString());
        } catch (MalformedURLException e) {
            parser.getParserErrors().add(e.getMessage());
        }
    }

    @Override
    public Integer getRegistryPort() {
        return registryPort;
    }

    @Override
    public String getUsersFileName() {
        return usersFileName;
    }

    @Override
    public int getSlpLeaseTimeSecs() {
        return slpLeaseTimeSecs;
    }

    @Override
    public String getSlpAgentName() {
        return slpAgentName;
    }

    @Override
    public boolean isAuthFileBased() {
        return authFileBased;
    }

    @Override
    public JMXServiceURL getServiceUrl() {
        return serviceUrl;
    }

    @Override
    public String getRmiHostname() {
        return rmiHostname;
    }

    @Override
    public String getAdminRole() {
        return adminRole;
    }

    @Override
    public String getMonitorRole() {
        return monitorRole;
    }

    /**
     * Get the mbeanServer
     * 
     * @return the mbeanServer
     */
    public MBeanServer getMBeanServer() {
        return mbeanServer;
    }

    /**
     * Get the jmxAuthenticator for the user/password file, null is there is no
     * file
     * 
     * @return the jmxAuthenticator
     */
    public JMXAuthenticator getJmxAuthenticator() {
        return jmxAuthenticator;
    }

    /**
     * Get the accessController for the access file null if there is no file
     * 
     * @return the accessController
     */
    public JMXAccessController getAccessController() {
        return accessController;
    }

    /**
     * Get the slpConfig
     * 
     * @return the slpConfig
     */
    public SLPConfig getSlpConfig() {
        return slpConfig;
    }

    @Override
    public boolean isSlpEnabled() {
        return slpConfig.isEnabled();
    }

    @Override
    protected String getDescription(MBeanInfo info) {
        return "Configuration to run a MBeanServer with an RMI connection on a single port";
    }

    @Override
    @SuppressWarnings("PMD.NPathComplexity")
    protected String getDescription(MBeanAttributeInfo info) {
        if ("RegistryPort".equals(info.getName())) {
            return "The port for the RMI registry";
        }
        if ("RmiHostname".equals(info.getName())) {
            return "The host name for the RMI registry";
        }
        if ("UsersFileName".equals(info.getName())) {
            return "The name of the properties file with user definitions";
        }
        if ("SlpEnabled".equals(info.getName())) {
            return "Is SLP client enabled";
        }
        if ("SlpLeaseTimeSecs".equals(info.getName())) {
            return "The SLP lease time in seconds";
        }
        if ("SlpAgentName".equals(info.getName())) {
            return "The SLP agent name";
        }
        if ("AuthFileBased".equals(info.getName())) {
            return "Is authentication based on a local users properties file";
        }
        if ("ServiceUrl".equals(info.getName())) {
            return "The service URL";
        }
        if ("AdminRole".equals(info.getName())) {
            return "The admin rlole name";
        }
        if ("MonitorRole".equals(info.getName())) {
            return "The monitor role name";
        }
        return super.getDescription(info);
    }

}
