/*
 * Copyright Alibaba Group Holding Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.alibaba.lindorm.sql.ce.search;

import com.fasterxml.jackson.core.JsonFactory;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.collect.ImmutableList;
import io.airlift.json.ObjectMapperProvider;
import io.airlift.slice.DynamicSliceOutput;
import io.airlift.slice.Slice;
import io.airlift.slice.SliceOutput;
import io.prestosql.plugin.jdbc.BaseJdbcClient;
import io.prestosql.plugin.jdbc.BaseJdbcConfig;
import io.prestosql.plugin.jdbc.ColumnMapping;
import io.prestosql.plugin.jdbc.ConnectionFactory;
import io.prestosql.plugin.jdbc.JdbcColumnHandle;
import io.prestosql.plugin.jdbc.JdbcIdentity;
import io.prestosql.plugin.jdbc.JdbcTableHandle;
import io.prestosql.plugin.jdbc.JdbcTypeHandle;
import io.prestosql.plugin.jdbc.UnsupportedTypeHandling;
import io.prestosql.plugin.jdbc.WriteMapping;
import io.prestosql.spi.PrestoException;
import io.prestosql.spi.connector.ConnectorSession;
import io.prestosql.spi.connector.ConnectorTableMetadata;
import io.prestosql.spi.connector.SchemaTableName;
import io.prestosql.spi.connector.TableNotFoundException;
import io.prestosql.spi.type.Decimals;
import io.prestosql.spi.type.StandardTypes;
import io.prestosql.spi.type.Type;
import io.prestosql.spi.type.TypeManager;
import io.prestosql.spi.type.TypeSignature;
import io.prestosql.spi.type.VarcharType;

import javax.inject.Inject;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Types;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.function.BiFunction;

import static com.fasterxml.jackson.core.JsonFactory.Feature.CANONICALIZE_FIELD_NAMES;
import static com.fasterxml.jackson.databind.SerializationFeature.ORDER_MAP_ENTRIES_BY_KEYS;
import static com.google.common.base.Strings.emptyToNull;
import static com.google.common.base.Verify.verify;
import static io.airlift.slice.Slices.utf8Slice;
import static io.prestosql.plugin.jdbc.ColumnMapping.DISABLE_PUSHDOWN;
import static io.prestosql.plugin.jdbc.DecimalConfig.DecimalMapping.ALLOW_OVERFLOW;
import static io.prestosql.plugin.jdbc.DecimalSessionPropertiesProvider.getDecimalDefaultScale;
import static io.prestosql.plugin.jdbc.DecimalSessionPropertiesProvider.getDecimalRounding;
import static io.prestosql.plugin.jdbc.DecimalSessionPropertiesProvider.getDecimalRoundingMode;
import static io.prestosql.plugin.jdbc.JdbcErrorCode.JDBC_ERROR;
import static io.prestosql.plugin.jdbc.StandardColumnMappings.decimalColumnMapping;
import static io.prestosql.plugin.jdbc.StandardColumnMappings.realWriteFunction;
import static io.prestosql.plugin.jdbc.StandardColumnMappings.timestampWriteFunctionUsingSqlTimestamp;
import static io.prestosql.plugin.jdbc.StandardColumnMappings.varbinaryWriteFunction;
import static io.prestosql.plugin.jdbc.StandardColumnMappings.varcharWriteFunction;
import static io.prestosql.plugin.jdbc.TypeHandlingJdbcPropertiesProvider.getUnsupportedTypeHandling;
import static io.prestosql.plugin.jdbc.UnsupportedTypeHandling.IGNORE;
import static io.prestosql.spi.StandardErrorCode.INVALID_FUNCTION_ARGUMENT;
import static io.prestosql.spi.StandardErrorCode.NOT_SUPPORTED;
import static io.prestosql.spi.type.DecimalType.createDecimalType;
import static io.prestosql.spi.type.RealType.REAL;
import static io.prestosql.spi.type.TimeWithTimeZoneType.TIME_WITH_TIME_ZONE;
import static io.prestosql.spi.type.TimestampType.TIMESTAMP;
import static io.prestosql.spi.type.TimestampWithTimeZoneType.TIMESTAMP_WITH_TIME_ZONE;
import static io.prestosql.spi.type.VarbinaryType.VARBINARY;
import static io.prestosql.spi.type.Varchars.isVarcharType;
import static java.lang.Math.min;
import static java.lang.String.format;
import static java.nio.charset.StandardCharsets.UTF_8;
import static java.sql.DatabaseMetaData.columnNoNulls;

public class LindormSearchClient extends BaseJdbcClient {
  private final Type jsonType;

  @Inject
  public LindormSearchClient(BaseJdbcConfig config,
      ConnectionFactory connectionFactory, TypeManager typeManager) {
    super(config, "`", connectionFactory);
    this.jsonType = typeManager.getType(new TypeSignature(StandardTypes.JSON));
  }

  @Override
  public void abortReadConnection(Connection connection) throws SQLException {
    throw new RuntimeException("abortReadConnection is not supported");
  }

  @Override
  public PreparedStatement getPreparedStatement(Connection connection,
      String sql) throws SQLException {
    PreparedStatement statement = connection.prepareStatement(sql);
    return statement;
  }

  @Override
  protected ResultSet getTables(Connection connection,
      Optional<String> schemaName, Optional<String> tableName)
      throws SQLException {
    DatabaseMetaData metadata = connection.getMetaData();
    return metadata.getTables(connection.getCatalog(), schemaName.orElse(null),
        tableName.orElse(null), new String[] { "TABLE", "VIEW" });
  }

  @Override
  protected String getTableSchemaName(ResultSet resultSet) throws SQLException {
    return resultSet.getString("TABLE_SCHEM");
  }

  @Override
  public Optional<ColumnMapping> toPrestoType(ConnectorSession session,
      Connection connection, JdbcTypeHandle typeHandle) {
    String jdbcTypeName = typeHandle.getJdbcTypeName().orElseThrow(
        () -> new PrestoException(JDBC_ERROR,
            "Type name is missing: " + typeHandle));

    Optional<ColumnMapping> mapping = getForcedMappingToVarchar(typeHandle);
    if (mapping.isPresent()) {
      return mapping;
    }
    if (jdbcTypeName.equalsIgnoreCase("json")) {
      return Optional.of(jsonColumnMapping());
    }
    if (typeHandle.getJdbcType() == Types.DECIMAL
        && getDecimalRounding(session) == ALLOW_OVERFLOW) {
      int precision = typeHandle.getColumnSize();
      if (precision > Decimals.MAX_PRECISION) {
        int scale = min(typeHandle.getDecimalDigits(),
            getDecimalDefaultScale(session));
        return Optional.of(decimalColumnMapping(
            createDecimalType(Decimals.MAX_PRECISION, scale),
            getDecimalRoundingMode(session)));
      }
    }
    return super.toPrestoType(session, connection, typeHandle);
  }

  @Override
  public WriteMapping toWriteMapping(ConnectorSession session, Type type) {
    if (REAL.equals(type)) {
      return WriteMapping.longMapping("float", realWriteFunction());
    }
    if (TIME_WITH_TIME_ZONE.equals(type) || TIMESTAMP_WITH_TIME_ZONE.equals(
        type)) {
      throw new PrestoException(NOT_SUPPORTED,
          "Unsupported column type: " + type.getDisplayName());
    }
    if (TIMESTAMP.equals(type)) {
      // TODO use `timestampWriteFunction`
      return WriteMapping.longMapping("datetime",
          timestampWriteFunctionUsingSqlTimestamp(session));
    }
    if (VARBINARY.equals(type)) {
      return WriteMapping.sliceMapping("mediumblob", varbinaryWriteFunction());
    }
    if (isVarcharType(type)) {
      VarcharType varcharType = (VarcharType) type;
      String dataType;
      if (varcharType.isUnbounded()) {
        dataType = "longtext";
      } else if (varcharType.getBoundedLength() <= 255) {
        dataType = "tinytext";
      } else if (varcharType.getBoundedLength() <= 65535) {
        dataType = "text";
      } else if (varcharType.getBoundedLength() <= 16777215) {
        dataType = "mediumtext";
      } else {
        dataType = "longtext";
      }
      return WriteMapping.sliceMapping(dataType, varcharWriteFunction());
    }
    if (type.equals(jsonType)) {
      return WriteMapping.sliceMapping("json", varcharWriteFunction());
    }

    return super.toWriteMapping(session, type);
  }

  @Override
  public void createTable(ConnectorSession session,
      ConnectorTableMetadata tableMetadata) {
    throw new RuntimeException("createTable is not supported");
  }

  @Override
  public void renameColumn(JdbcIdentity identity, JdbcTableHandle handle,
      JdbcColumnHandle jdbcColumn, String newColumnName) {
    throw new RuntimeException("renameColumn is not supported");
  }

  @Override
  protected void copyTableSchema(Connection connection, String catalogName,
      String schemaName, String tableName, String newTableName,
      List<String> columnNames) {
    throw new RuntimeException("copyTableSchema is not supported");
  }

  @Override
  public void renameTable(JdbcIdentity identity, JdbcTableHandle handle,
      SchemaTableName newTableName) {
    throw new RuntimeException("renameTable is not supported");
  }

  @Override
  protected Optional<BiFunction<String, Long, String>> limitFunction() {
    return Optional.of((sql, limit) -> sql + " LIMIT " + limit);
  }

  @Override
  public boolean isLimitGuaranteed() {
    return true;
  }

  private ColumnMapping jsonColumnMapping() {
    return ColumnMapping.sliceMapping(jsonType,
        (resultSet, columnIndex) -> jsonParse(
            utf8Slice(resultSet.getString(columnIndex))),
        varcharWriteFunction(), DISABLE_PUSHDOWN);
  }

  private static final JsonFactory JSON_FACTORY = new JsonFactory().disable(
      CANONICALIZE_FIELD_NAMES);

  private static final ObjectMapper SORTED_MAPPER = new ObjectMapperProvider().get()
      .configure(ORDER_MAP_ENTRIES_BY_KEYS, true);

  private static Slice jsonParse(Slice slice) {
    try (JsonParser parser = createJsonParser(slice)) {
      byte[] in = slice.getBytes();
      SliceOutput dynamicSliceOutput = new DynamicSliceOutput(in.length);
      SORTED_MAPPER.writeValue((OutputStream) dynamicSliceOutput,
          SORTED_MAPPER.readValue(parser, Object.class));
      // nextToken() returns null if the input is parsed correctly,
      // but will throw an exception if there are trailing characters.
      parser.nextToken();
      return dynamicSliceOutput.slice();
    } catch (Exception e) {
      throw new PrestoException(INVALID_FUNCTION_ARGUMENT,
          format("Cannot convert '%s' to JSON", slice.toStringUtf8()));
    }
  }

  private static JsonParser createJsonParser(Slice json) throws IOException {
    // Jackson tries to detect the character encoding automatically when using InputStream
    // so we pass an InputStreamReader instead.
    return JSON_FACTORY.createParser(
        new InputStreamReader(json.getInput(), UTF_8));
  }

  @Override
  public List<JdbcColumnHandle> getColumns(ConnectorSession session, JdbcTableHandle tableHandle)
  {
    try (Connection connection = connectionFactory.openConnection(JdbcIdentity.from(session));
        ResultSet resultSet = getColumns(tableHandle, connection.getMetaData())) {
      int allColumns = 0;
      List<JdbcColumnHandle> columns = new ArrayList<>();
      while (resultSet.next()) {
        allColumns++;
        String columnName = resultSet.getString("COLUMN_NAME");
        int dataType = resultSet.getInt("DATA_TYPE");
        int columnSize = resultSet.getInt("COLUMN_SIZE");
        if(dataType == Types.VARCHAR){
          columnSize = Integer.MAX_VALUE;
        }
        JdbcTypeHandle typeHandle = new JdbcTypeHandle(
            dataType,
            Optional.ofNullable(resultSet.getString("TYPE_NAME")),
            columnSize,
            resultSet.getInt("DECIMAL_DIGITS"),
            Optional.empty());
        Optional<ColumnMapping> columnMapping = toPrestoType(session, connection, typeHandle);
//        log.debug("Mapping data type of '%s' column '%s': %s mapped to %s", tableHandle.getSchemaTableName(), columnName, typeHandle, columnMapping);
        // skip unsupported column types
        boolean nullable = (resultSet.getInt("NULLABLE") != columnNoNulls);
        // Note: some databases (e.g. SQL Server) do not return column remarks/comment here.
        Optional<String> comment = Optional.ofNullable(emptyToNull(resultSet.getString("REMARKS")));
        if (columnMapping.isPresent()) {
          columns.add(JdbcColumnHandle.builder()
              .setColumnName(columnName)
              .setJdbcTypeHandle(typeHandle)
              .setColumnType(columnMapping.get().getType())
              .setNullable(nullable)
              .setComment(comment)
              .build());
        }
        if (!columnMapping.isPresent()) {
          UnsupportedTypeHandling unsupportedTypeHandling = getUnsupportedTypeHandling(session);
          verify(unsupportedTypeHandling == IGNORE, "Unsupported type handling is set to %s, but toPrestoType() returned empty", unsupportedTypeHandling);
        }
      }
      if (columns.isEmpty()) {
        // A table may have no supported columns. In rare cases (e.g. PostgreSQL) a table might have no columns at all.
        throw new TableNotFoundException(
            tableHandle.getSchemaTableName(),
            format("Table '%s' has no supported columns (all %s columns are not supported)", tableHandle.getSchemaTableName(), allColumns));
      }
      return ImmutableList.copyOf(columns);
    }
    catch (SQLException e) {
      throw new PrestoException(JDBC_ERROR, e);
    }
  }
}
