/*
 * Copyright Alibaba Group Holding Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.alibaba.lindorm.sql.se.search;

import java.io.InputStream;
import java.io.Reader;
import java.math.BigDecimal;
import java.net.URL;
import java.sql.Array;
import java.sql.Blob;
import java.sql.Clob;
import java.sql.Connection;
import java.sql.Date;
import java.sql.NClob;
import java.sql.ParameterMetaData;
import java.sql.PreparedStatement;
import java.sql.Ref;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.RowId;
import java.sql.SQLException;
import java.sql.SQLWarning;
import java.sql.SQLXML;
import java.sql.Statement;
import java.sql.Time;
import java.sql.Timestamp;
import java.util.Calendar;

public class SearchPreparedStatement implements PreparedStatement {
  private Connection connection = null;
  private String sql = null;
  private Object[] params = null;
  private int paramSize = 0;
  private ResultSet rs = null;
  private Statement stmt = null;
  private boolean isClosed = false;

  public SearchPreparedStatement(Connection connection, String sql, int paramSize) {
    this.connection = connection;
    this.sql = sql;
    this.paramSize = paramSize;
    this.params = new Object[paramSize];
  }


  @Override
  public void close() throws SQLException {
    try {
      if (rs != null) {
        rs.close();
      }
      if (stmt != null) {
        stmt.close();
      }
      isClosed = true;
    } catch (Throwable e) {
      throw new SQLException(e);
    }
  }

  @Override
  public ResultSet getResultSet() throws SQLException {
    return rs;
  }

  @Override
  public boolean execute() throws SQLException {
    try {
      String finalSql = getFinalSql();
      if (stmt == null) {
        stmt = connection.createStatement();
      }
      if (finalSql.trim().toUpperCase().startsWith("SELECT")) {
        rs = stmt.executeQuery(finalSql);
        return true;
      } else {
        stmt.execute(finalSql);
        return false;
      }
    } catch (Throwable e) {
      throw new SQLException(e);
    }
  }

  private String getFinalSql() {
    String tmpStr = sql;
    if (paramSize == 0) {
      return tmpStr;
    }
    StringBuilder sb = new StringBuilder();
    for (int i = 0; i < paramSize; i++) {
      int pos = tmpStr.indexOf("?");
      sb.append(tmpStr.substring(0, pos));
      if (params[i] instanceof String) {
        sb.append("'").append(params[i]).append("'");
      } else {
        sb.append(params[i]);
      }
      if (i == paramSize - 1) {
        if (tmpStr.length() > pos + 1) {
          sb.append(tmpStr.substring(pos + 1));
        }
      } else {
        tmpStr = tmpStr.substring(pos + 1);
      }
    }
    return sb.toString();
  }

  @Override
  public ResultSet executeQuery() throws SQLException {
    try {
      String finalSql = getFinalSql();
      if (stmt == null) {
        stmt = connection.createStatement();
      }
      return stmt.executeQuery(finalSql);
    } catch (Throwable e) {
      throw new SQLException(e);
    }
  }

  @Override
  public int executeUpdate() throws SQLException {
    try {
      String finalSql = getFinalSql();
      if (stmt == null) {
        stmt = connection.createStatement();
      }
      return stmt.executeUpdate(finalSql);
    } catch (Throwable e) {
      throw new SQLException(e);
    }
  }

  private void checkIndex(int index) throws SQLException {
    if (index > paramSize) {
      throw new SQLException("index " + index + " out of bound");
    }
  }

  @Override
  public void setNull(int parameterIndex, int sqlType) throws SQLException {
    checkIndex(parameterIndex);
    params[parameterIndex - 1] = sqlType;
  }

  @Override
  public void setBoolean(int parameterIndex, boolean x) throws SQLException {
    checkIndex(parameterIndex);
    params[parameterIndex - 1] = x;
  }

  @Override
  public void setByte(int parameterIndex, byte x) throws SQLException {
    checkIndex(parameterIndex);
    params[parameterIndex - 1] = x;
  }

  @Override
  public void setShort(int parameterIndex, short x) throws SQLException {
    checkIndex(parameterIndex);
    params[parameterIndex - 1] = x;
  }

  @Override
  public void setInt(int parameterIndex, int x) throws SQLException {
    checkIndex(parameterIndex);
    params[parameterIndex - 1] = x;
  }

  @Override
  public void setLong(int parameterIndex, long x) throws SQLException {
    checkIndex(parameterIndex);
    params[parameterIndex - 1] = x;
  }

  @Override
  public void setFloat(int parameterIndex, float x) throws SQLException {
    checkIndex(parameterIndex);
    params[parameterIndex - 1] = x;
  }

  @Override
  public void setDouble(int parameterIndex, double x) throws SQLException {
    checkIndex(parameterIndex);
    params[parameterIndex - 1] = x;
  }

  @Override
  public void setBigDecimal(int parameterIndex, BigDecimal x) throws SQLException {
    checkIndex(parameterIndex);
    params[parameterIndex - 1] = x;
  }

  @Override
  public void setString(int parameterIndex, String x) throws SQLException {
    checkIndex(parameterIndex);
    params[parameterIndex - 1] = x;
  }

  @Override
  public void setBytes(int parameterIndex, byte[] x) throws SQLException {
    checkIndex(parameterIndex);
    params[parameterIndex - 1] = x;
  }

  @Override
  public void setDate(int parameterIndex, Date x) throws SQLException {
    checkIndex(parameterIndex);
    params[parameterIndex - 1] = x;
  }

  @Override
  public void setTime(int parameterIndex, Time x) throws SQLException {
    checkIndex(parameterIndex);
    params[parameterIndex - 1] = x;
  }

  @Override
  public void setTimestamp(int parameterIndex, Timestamp x) throws SQLException {
    checkIndex(parameterIndex);
    params[parameterIndex - 1] = x;
  }

  @Override
  public void setAsciiStream(int parameterIndex, InputStream x, int length) throws SQLException {
    throw new SQLException("setAsciiStream not supported");
  }

  @Override
  public void setUnicodeStream(int parameterIndex, InputStream x, int length) throws SQLException {
    throw new SQLException("setUnicodeStream not supported");
  }

  @Override
  public void setBinaryStream(int parameterIndex, InputStream x, int length) throws SQLException {
    throw new SQLException("setBinaryStream not supported");
  }

  @Override
  public void clearParameters() throws SQLException {
    params = new Object[paramSize];
  }

  @Override
  public void setObject(int parameterIndex, Object x, int targetSqlType) throws SQLException {
    throw new SQLException("setObject(int parameterIndex, Object x, int targetSqlType) not supported");
  }

  @Override
  public void setObject(int parameterIndex, Object x) throws SQLException {
    checkIndex(parameterIndex);
    params[parameterIndex - 1] = x;
  }

  @Override
  public void addBatch() throws SQLException {
    throw new SQLException("addBatch not supported");
  }

  @Override
  public void setCharacterStream(int parameterIndex, Reader reader, int length) throws SQLException {
    throw new SQLException("setCharacterStream(int parameterIndex, Reader reader, int length) not supported");
  }

  @Override
  public void setRef(int parameterIndex, Ref x) throws SQLException {
    throw new SQLException("setRef(int parameterIndex, Ref x) not supported");
  }

  @Override
  public void setBlob(int parameterIndex, Blob x) throws SQLException {
    throw new SQLException("setBlob(int parameterIndex, Blob x) not supported");
  }

  @Override
  public void setClob(int parameterIndex, Clob x) throws SQLException {
    throw new SQLException("setClob(int parameterIndex, Clob x) not supported");
  }

  @Override
  public void setArray(int parameterIndex, Array x) throws SQLException {
    throw new SQLException("setArray(int parameterIndex, Array x) not supported");
  }

  @Override
  public ResultSetMetaData getMetaData() throws SQLException {
    if (rs != null) {
      return rs.getMetaData();
    } else {
      return null;
    }
  }

  @Override
  public void setDate(int parameterIndex, Date x, Calendar cal) throws SQLException {
    throw new SQLException("setDate(int parameterIndex, Date x, Calendar cal) not supported");
  }

  @Override
  public void setTime(int parameterIndex, Time x, Calendar cal) throws SQLException {
    throw new SQLException("setTime(int parameterIndex, Time x, Calendar cal) not supported");
  }

  @Override
  public void setTimestamp(int parameterIndex, Timestamp x, Calendar cal) throws SQLException {
    throw new SQLException("setTimestamp(int parameterIndex, Timestamp x, Calendar cal) not supported");
  }

  @Override
  public void setNull(int parameterIndex, int sqlType, String typeName) throws SQLException {
    throw new SQLException("setNull(int parameterIndex, int sqlType, String typeName) not supported");
  }

  @Override
  public void setURL(int parameterIndex, URL x) throws SQLException {
    throw new SQLException("setURL(int parameterIndex, URL x) not supported");
  }

  @Override
  public ParameterMetaData getParameterMetaData() throws SQLException {
    return new SearchParameterMetaData(paramSize);
  }

  @Override
  public void setRowId(int parameterIndex, RowId x) throws SQLException {
    throw new SQLException("setRowId(int parameterIndex, RowId x) not supported");
  }

  @Override
  public void setNString(int parameterIndex, String value) throws SQLException {
    checkIndex(parameterIndex);
    params[parameterIndex - 1] = value;
  }

  @Override
  public void setNCharacterStream(int parameterIndex, Reader value, long length) throws SQLException {
    throw new SQLException("setNCharacterStream(int parameterIndex, Reader value, long length) not supported");
  }

  @Override
  public void setNClob(int parameterIndex, NClob value) throws SQLException {
    throw new SQLException("setNClob(int parameterIndex, NClob value) not supported");
  }

  @Override
  public void setClob(int parameterIndex, Reader reader, long length) throws SQLException {
    throw new SQLException("setClob(int parameterIndex, Reader reader, long length) not supported");
  }

  @Override
  public void setBlob(int parameterIndex, InputStream inputStream, long length) throws SQLException {
    throw new SQLException("setBlob(int parameterIndex, InputStream inputStream, long length) not supported");
  }

  @Override
  public void setNClob(int parameterIndex, Reader reader, long length) throws SQLException {
    throw new SQLException("setNClob(int parameterIndex, Reader reader, long length) not supported");
  }

  @Override
  public void setSQLXML(int parameterIndex, SQLXML xmlObject) throws SQLException {
    throw new SQLException("setSQLXML(int parameterIndex, SQLXML xmlObject) not supported");
  }

  @Override
  public void setObject(int parameterIndex, Object x, int targetSqlType, int scaleOrLength) throws SQLException {
    throw new SQLException("setObject(int parameterIndex, Object x, int targetSqlType, int scaleOrLength) not supported");
  }

  @Override
  public void setAsciiStream(int parameterIndex, InputStream x, long length) throws SQLException {
    throw new SQLException("setAsciiStream(int parameterIndex, InputStream x, long length) not supported");
  }

  @Override
  public void setBinaryStream(int parameterIndex, InputStream x, long length) throws SQLException {
    throw new SQLException("setBinaryStream(int parameterIndex, InputStream x, long length) not supported");
  }

  @Override
  public void setCharacterStream(int parameterIndex, Reader reader, long length) throws SQLException {
    throw new SQLException("setCharacterStream(int parameterIndex, Reader reader, long length) not supported");
  }

  @Override
  public void setAsciiStream(int parameterIndex, InputStream x) throws SQLException {
    throw new SQLException("setAsciiStream(int parameterIndex, InputStream x) not supported");
  }

  @Override
  public void setBinaryStream(int parameterIndex, InputStream x) throws SQLException {
    throw new SQLException("setBinaryStream(int parameterIndex, InputStream x) not supported");
  }

  @Override
  public void setCharacterStream(int parameterIndex, Reader reader) throws SQLException {
    throw new SQLException("setCharacterStream(int parameterIndex, Reader reader) not supported");
  }

  @Override
  public void setNCharacterStream(int parameterIndex, Reader value) throws SQLException {
    throw new SQLException("setNCharacterStream(int parameterIndex, Reader value) not supported");
  }

  @Override
  public void setClob(int parameterIndex, Reader reader) throws SQLException {
    throw new SQLException("setClob(int parameterIndex, Reader reader) not supported");
  }

  @Override
  public void setBlob(int parameterIndex, InputStream inputStream) throws SQLException {
    throw new SQLException("setBlob(int parameterIndex, InputStream inputStream) not supported");
  }

  @Override
  public void setNClob(int parameterIndex, Reader reader) throws SQLException {
    throw new SQLException("setNClob(int parameterIndex, Reader reader) not supported");
  }

  @Override
  public int getMaxFieldSize() throws SQLException {
    throw new SQLException("getMaxFieldSize() not supported");
  }

  @Override
  public void setMaxFieldSize(int max) throws SQLException {
    throw new SQLException("setMaxFieldSize(int max) not supported");
  }

  @Override
  public int getMaxRows() throws SQLException {
    throw new SQLException("getMaxRows() not supported");
  }

  @Override
  public void setMaxRows(int max) throws SQLException {
  }

  @Override
  public void setEscapeProcessing(boolean enable) throws SQLException {
    throw new SQLException("setEscapeProcessing(boolean enable) not supported");
  }

  @Override
  public int getQueryTimeout() throws SQLException {
    throw new SQLException("getQueryTimeout() not supported");
  }

  @Override
  public void setQueryTimeout(int seconds) throws SQLException {
    throw new SQLException("setQueryTimeout(int seconds) not supported");
  }

  @Override
  public void cancel() throws SQLException {
  }

  @Override
  public SQLWarning getWarnings() throws SQLException {
    return stmt.getWarnings();
  }

  @Override
  public void clearWarnings() throws SQLException {

  }

  @Override
  public void setCursorName(String name) throws SQLException {

  }

  @Override
  public int getUpdateCount() throws SQLException {
    return 1;
  }

  @Override
  public boolean getMoreResults() throws SQLException {
    throw new SQLException("getMoreResults() not supported");
  }

  @Override
  public void setFetchDirection(int direction) throws SQLException {
    throw new SQLException("setFetchDirection(int direction) not supported");
  }

  @Override
  public int getFetchDirection() throws SQLException {
    throw new SQLException("getFetchDirection() not supported");
  }

  @Override
  public void setFetchSize(int rows) throws SQLException {
    throw new SQLException("setFetchSize(int rows) not supported");
  }

  @Override
  public int getFetchSize() throws SQLException {
    throw new SQLException("getFetchSize() not supported");
  }

  @Override
  public int getResultSetConcurrency() throws SQLException {
    throw new SQLException("getResultSetConcurrency() not supported");
  }

  @Override
  public int getResultSetType() throws SQLException {
    return getResultSetType();
  }

  @Override
  public void addBatch(String sql) throws SQLException {
    throw new SQLException("addBatch(String sql) not supported");
  }

  @Override
  public void clearBatch() throws SQLException {
    throw new SQLException("clearBatch()  not supported");
  }

  @Override
  public int[] executeBatch() throws SQLException {
    throw new SQLException("executeBatch()  not supported");
  }

  @Override
  public Connection getConnection() throws SQLException {
    return connection;
  }

  @Override
  public boolean getMoreResults(int current) throws SQLException {
    throw new SQLException("getMoreResults(int current) not supported");
  }

  @Override
  public ResultSet getGeneratedKeys() throws SQLException {
    throw new SQLException("getGeneratedKeys()not supported");
  }

  @Override
  public int executeUpdate(String sql, int autoGeneratedKeys) throws SQLException {
    throw new SQLException("executeUpdate(String sql, int autoGeneratedKeys) not supported");
  }

  @Override
  public int executeUpdate(String sql, int[] columnIndexes) throws SQLException {
    throw new SQLException("executeUpdate(String sql, int[] columnIndexes) not supported");
  }

  @Override
  public int executeUpdate(String sql, String[] columnNames) throws SQLException {
    throw new SQLException("executeUpdate(String sql, String[] columnNames) not supported");
  }

  @Override
  public boolean execute(String sql, int autoGeneratedKeys) throws SQLException {
    throw new SQLException("execute(String sql, int autoGeneratedKeys) not supported");
  }

  @Override
  public boolean execute(String sql, int[] columnIndexes) throws SQLException {
    throw new SQLException("execute(String sql, int[] columnIndexes) not supported");
  }

  @Override
  public boolean execute(String sql, String[] columnNames) throws SQLException {
    throw new SQLException("execute(String sql, String[] columnNames) not supported");
  }

  @Override
  public int getResultSetHoldability() throws SQLException {
    throw new SQLException("getResultSetHoldability() not supported");
  }

  @Override
  public boolean isClosed() throws SQLException {
    return isClosed;
  }

  @Override
  public void setPoolable(boolean poolable) throws SQLException {
    throw new SQLException("setPoolable(boolean poolable) not supported");
  }

  @Override
  public boolean isPoolable() throws SQLException {
    throw new SQLException("isPoolable() not supported");
  }

  @Override
  public void closeOnCompletion() throws SQLException {
  }

  @Override
  public boolean isCloseOnCompletion() throws SQLException {
    throw new SQLException("isCloseOnCompletion() not supported");
  }

  @Override
  public <T> T unwrap(Class<T> iface) throws SQLException {
    throw new SQLException("unwrap(Class<T> iface) not supported");
  }

  @Override
  public boolean isWrapperFor(Class<?> iface) throws SQLException {
    return false;
  }

  @Override
  public boolean execute(String sql) throws SQLException {
    throw new SQLException("execute(String sql) not supported");
  }

  @Override
  public ResultSet executeQuery(String sql) throws SQLException {
    throw new SQLException("executeQuery(String sql) not supported");
  }

  @Override
  public int executeUpdate(String sql) throws SQLException {
    throw new SQLException("executeUpdate(String sql) not supported");
  }
}
