/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.alibaba.lindorm.sql.se.search;


import com.alibaba.lindorm.thirdparty.org.apache.calcite.plan.RelOptCluster;
import com.alibaba.lindorm.thirdparty.org.apache.calcite.plan.RelOptCost;
import com.alibaba.lindorm.thirdparty.org.apache.calcite.plan.RelOptPlanner;
import com.alibaba.lindorm.thirdparty.org.apache.calcite.plan.RelOptRule;
import com.alibaba.lindorm.thirdparty.org.apache.calcite.plan.RelOptTable;
import com.alibaba.lindorm.thirdparty.org.apache.calcite.plan.RelTraitSet;
import com.alibaba.lindorm.thirdparty.org.apache.calcite.rel.RelNode;
import com.alibaba.lindorm.thirdparty.org.apache.calcite.rel.core.TableScan;
import com.alibaba.lindorm.thirdparty.org.apache.calcite.rel.metadata.RelMetadataQuery;
import com.alibaba.lindorm.thirdparty.org.apache.calcite.rel.type.RelDataType;

import java.util.List;

/**
 * Relational expression representing a scan of a Solr collection.
 */
class SearchTableScan extends TableScan implements SearchRel {
  private final SearchTable searchTable;
  private final RelDataType projectRowType;

  /**
   * Creates a SolrTableScan.
   *
   * @param cluster        Cluster
   * @param traitSet       Traits
   * @param table          Table
   * @param searchTable      Solr table
   * @param projectRowType Fields and types to project; null to project raw row
   */
  SearchTableScan(RelOptCluster cluster, RelTraitSet traitSet, RelOptTable table, SearchTable searchTable,
                RelDataType projectRowType) {
    super(cluster, traitSet, table);
    this.searchTable = searchTable;
    this.projectRowType = projectRowType;

    assert searchTable != null;
    assert getConvention() == SearchRel.CONVENTION;
  }

  @Override public RelOptCost computeSelfCost(RelOptPlanner planner, RelMetadataQuery mq) {
    final float f = projectRowType == null ? 1f : (float) projectRowType.getFieldCount() / 100f;
    return super.computeSelfCost(planner, mq).multiplyBy(.1 * f);
  }

  @Override
  public RelNode copy(RelTraitSet traitSet, List<RelNode> inputs) {
    assert inputs.isEmpty();
    return this;
  }

  @Override
  public RelDataType deriveRowType() {
    return projectRowType != null ? projectRowType : super.deriveRowType();
  }

  @Override
  public void register(RelOptPlanner planner) {
    planner.addRule(SearchToEnumerableConverterRule.INSTANCE);
    for (RelOptRule rule : SearchRules.RULES) {
      planner.addRule(rule);
    }

    for (RelOptRule rule : SearchRules.CONSTANT_REDUCTION_RULES) {
      planner.addRule(rule);
    }
  }

  public void implement(Implementor implementor) {
    implementor.searchTable = searchTable;
    implementor.table = table;
  }
}
