/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.alibaba.lindorm.sql.se.search;

import com.alibaba.lindorm.thirdparty.org.apache.calcite.jdbc.CalciteConnection;
import com.alibaba.lindorm.thirdparty.org.apache.calcite.jdbc.Driver;
import com.alibaba.lindorm.thirdparty.org.apache.calcite.schema.SchemaPlus;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.conn.PoolingHttpClientConnectionManager;
import org.apache.http.protocol.HttpRequestExecutor;
import org.apache.solr.client.solrj.impl.HttpClientUtil;
import org.apache.solr.client.solrj.io.SolrClientCache;
import org.apache.solr.common.params.ModifiableSolrParams;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.sql.Connection;
import java.sql.SQLException;
import java.util.Properties;

/**
 * JDBC driver for Calcite Solr.
 *
 * <p>It accepts connect strings that start with "jdbc:calcitesolr:".</p>
 */
public class SimpleSearchDriver extends Driver {
  private static final Logger LOG = LoggerFactory.getLogger(
      SimpleSearchDriver.class);

  public final static String CONNECT_STRING_PREFIX = "jdbc:lindormsearchse:";

  public static SimpleSearchDriver INSTANCE = new SimpleSearchDriver();
  public final static String DEFAULT_SCHEMA_NAME = "default";

  private static SolrClientCache solrClientCache;
  private static CloseableHttpClient defaultClient;

  public SimpleSearchDriver() {
    super();
  }

  static {
    INSTANCE.register();
    ModifiableSolrParams clientParams = new ModifiableSolrParams();
    clientParams.set(HttpClientUtil.PROP_SO_TIMEOUT, 340000);
    clientParams.set(HttpClientUtil.PROP_CONNECTION_TIMEOUT, 45000);
    // following is done only for logging complete configuration.
    // The maxConnections and maxConnectionsPerHost have already been specified on the connection manager
    clientParams.set(HttpClientUtil.PROP_MAX_CONNECTIONS, 100000);
    clientParams.set(HttpClientUtil.PROP_MAX_CONNECTIONS_PER_HOST, 100000);
    PoolingHttpClientConnectionManager defaultConnectionManager = new PoolingHttpClientConnectionManager(
        HttpClientUtil.getSocketFactoryRegistryProvider()
            .getSocketFactoryRegistry());
    defaultConnectionManager.setMaxTotal(100000);
    defaultConnectionManager.setDefaultMaxPerRoute(100000);
    HttpRequestExecutor httpRequestExecutor = new HttpRequestExecutor();
    defaultClient = HttpClientUtil.createClient(
        clientParams, defaultConnectionManager, false, httpRequestExecutor);
    SolrClientCache solrClientCache = new SolrClientCache(defaultClient);
    INSTANCE.setSolrClientCache(solrClientCache);
  }

  @Override
  protected String getConnectStringPrefix() {
    return CONNECT_STRING_PREFIX;
  }

  @Override
  public Connection connect(String url, Properties info) throws SQLException {
    if (!this.acceptsURL(url)) {
      return null;
    }

    String zk = info.getProperty("zk");
    if (zk == null) {
      zk = url.substring(CONNECT_STRING_PREFIX.length());
      if (zk == null || zk.equals("")) {
        throw new SQLException("url is error:" + url);
      } else {
        info.put("zk", zk);
      }
    }
    if (info.getProperty("lex") == null) {
      info.put("lex", "MYSQL");
    }

    Connection connection = super.connect(url, info);
    CalciteConnection calciteConnection = (CalciteConnection) connection;
    final SchemaPlus rootSchema = calciteConnection.getRootSchema();
    final SearchSchema searchSchema = new SearchSchema(info, solrClientCache);
    rootSchema.add(DEFAULT_SCHEMA_NAME, searchSchema);

    // Set the default schema
    calciteConnection.setSchema(DEFAULT_SCHEMA_NAME);
    return new SearchConnection(connection);
  }

  public void setSolrClientCache(SolrClientCache solrClientCache) {
    if (this.solrClientCache != null) {
      this.solrClientCache.close();
      HttpClientUtil.close(defaultClient);
    }
    this.solrClientCache = solrClientCache;
  }

  public static void close(){
    if (solrClientCache != null) {
      solrClientCache.close();
    }
    if (defaultClient != null) {
      HttpClientUtil.close(defaultClient);
    }
  }

  public static void main(String[] args) {

  }
}
