/*
 * Decompiled with CFR 0.152.
 */
package com.amadeus.session.repository.redis;

import com.amadeus.session.SessionData;
import com.amadeus.session.SessionManager;
import com.amadeus.session.WrappedException;
import com.amadeus.session.repository.redis.ExpirationListener;
import com.amadeus.session.repository.redis.RedisExpirationStrategy;
import com.amadeus.session.repository.redis.RedisFacade;
import com.amadeus.session.repository.redis.RedisSessionRepository;
import com.amadeus.session.repository.redis.SafeEncoder;
import com.amadeus.session.shaded.org.slf4j.Logger;
import com.amadeus.session.shaded.org.slf4j.LoggerFactory;
import java.util.Calendar;
import java.util.Set;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;

class NotificationExpirationManagement
implements RedisExpirationStrategy {
    static Logger logger = LoggerFactory.getLogger(NotificationExpirationManagement.class);
    private static final long ONE_MINUTE = TimeUnit.MINUTES.toSeconds(1L);
    private static final int SPOP_BULK_SIZE = 1000;
    static final int SESSION_PERSISTENCE_SAFETY_MARGIN = 300;
    private static final byte[] EMPTY_STRING = SafeEncoder.encode("");
    static final String DEFAULT_SESSION_EXPIRE_PREFIX = "com.amadeus.session:expire:";
    static final byte[] DEFAULT_SESSION_EXPIRE_PREFIX_BUF = SafeEncoder.encode("com.amadeus.session:expire:");
    private static final long RESET_RETRY_THRESHOLD = TimeUnit.SECONDS.toMillis(377L);
    private static final int[] FIBONACCI_DELAY_PATTERN = new int[]{0, 1, 1, 2, 3, 5, 8, 13, 21, 34, 55, 89, 144, 233};
    private static final int MAX_CONNECTION_ERRORS = FIBONACCI_DELAY_PATTERN.length;
    private final RedisFacade redis;
    private final RedisSessionRepository repository;
    private final String keyExpirePrefix;
    private final String expirationsPrefix;
    private final String forcedExpirationsPrefix;
    private final String namespace;
    private final String owner;
    private final boolean sticky;
    private ExpirationListener expirationListener;
    private ScheduledFuture<?> cleanupFuture;
    private ScheduledFuture<?> forceCleanupFuture;

    NotificationExpirationManagement(RedisFacade redis, RedisSessionRepository redisSession, String namespace, String owner, String keyPrefix, boolean sticky) {
        this.redis = redis;
        this.repository = redisSession;
        this.sticky = sticky;
        this.namespace = namespace;
        this.owner = owner;
        if (sticky) {
            this.forcedExpirationsPrefix = keyPrefix + "forced-expirations:";
            this.keyExpirePrefix = this.constructKeyExpirePrefix(owner);
        } else {
            this.forcedExpirationsPrefix = null;
            this.keyExpirePrefix = "com.amadeus.session:expire::" + namespace + ":";
        }
        this.expirationsPrefix = keyPrefix + "expirations:";
    }

    private String constructKeyExpirePrefix(String sessionOwner) {
        return "com.amadeus.session:expire::" + sessionOwner + ":" + this.namespace + ":";
    }

    @Override
    public void sessionDeleted(SessionData session) {
        long expireCleanupInstant = NotificationExpirationManagement.roundUpToNextMinute(session.expiresAt());
        byte[] expireKey = this.getExpirationsKey(expireCleanupInstant);
        this.redis.srem(expireKey, new byte[][]{this.repository.sessionKey(session.getId())});
        byte[] sessionExpireKey = this.getSessionExpireKey(session.getId());
        this.redis.del(new byte[][]{sessionExpireKey});
    }

    @Override
    public void sessionTouched(SessionData session) {
        new ExpirationManagement().manageExpiration(session);
    }

    private static long roundUpToNextMinute(long timeInMs) {
        Calendar date = Calendar.getInstance();
        date.setTimeInMillis(timeInMs);
        date.add(12, 1);
        date.clear(13);
        date.clear(14);
        return date.getTimeInMillis();
    }

    static long roundDownMinute(long timeInMs) {
        Calendar date = Calendar.getInstance();
        date.setTimeInMillis(timeInMs);
        date.clear(13);
        date.clear(14);
        return date.getTimeInMillis();
    }

    @Override
    public void startExpiredSessionsTask(SessionManager sessionManager) {
        sessionManager.submit(null, new SubscriptionRunner(sessionManager));
        TriggerExpiredSessionsTask taskTriggerExpiration = new TriggerExpiredSessionsTask();
        this.cleanupFuture = sessionManager.schedule("redis.expiration-cleanup", taskTriggerExpiration, ONE_MINUTE);
        if (this.sticky) {
            CleanHangingSessionsTask taskForceExpiration = new CleanHangingSessionsTask(sessionManager);
            this.forceCleanupFuture = sessionManager.schedule("redis.force-cleanup", taskForceExpiration, ONE_MINUTE);
        }
    }

    Set<byte[]> getKeysToExpire(byte[] key) {
        if (!this.redis.supportsMultiSpop()) {
            return this.redis.transaction(key, NotificationExpirationManagement.smembersAndDel(key)).get();
        }
        Set<byte[]> res = this.redis.spop(key, 1000L);
        if (res == null || res.isEmpty() || res.size() < 1000) {
            this.redis.del(new byte[][]{key});
        }
        return res;
    }

    static RedisFacade.TransactionRunner<Set<byte[]>> smembersAndDel(final byte[] key) {
        return new RedisFacade.TransactionRunner<Set<byte[]>>(){

            @Override
            public RedisFacade.ResponseFacade<Set<byte[]>> run(RedisFacade.TransactionFacade transaction) {
                RedisFacade.ResponseFacade<Set<byte[]>> result = transaction.smembers(key);
                transaction.del(new byte[][]{key});
                return result;
            }
        };
    }

    byte[] getSessionExpireKey(String id) {
        return SafeEncoder.encode(new StringBuilder(this.keyExpirePrefix.length() + id.length() + 1).append(this.keyExpirePrefix).append('{').append(id).append('}').toString());
    }

    byte[] getSessionExpireKey(String owner, String id) {
        String ownerBasedPrefix = this.constructKeyExpirePrefix(owner);
        return SafeEncoder.encode(new StringBuilder(ownerBasedPrefix.length() + id.length() + 1).append(ownerBasedPrefix).append('{').append(id).append('}').toString());
    }

    private byte[] getExpirationsKey(long instant) {
        String exp = Long.toString(instant);
        return SafeEncoder.encode(new StringBuilder(this.expirationsPrefix.length() + exp.length()).append(this.expirationsPrefix).append(exp).toString());
    }

    private byte[] getForcedExpirationsKey(long instant) {
        String exp = Long.toString(instant);
        return SafeEncoder.encode(new StringBuilder(this.forcedExpirationsPrefix.length() + exp.length()).append(this.forcedExpirationsPrefix).append(exp).toString());
    }

    @Override
    public void close() {
        if (this.expirationListener != null) {
            this.expirationListener.close(this.redis);
            this.expirationListener = null;
        }
        if (this.cleanupFuture != null) {
            this.cleanupFuture.cancel(true);
            this.cleanupFuture = null;
        }
        if (this.forceCleanupFuture != null) {
            this.forceCleanupFuture.cancel(true);
            this.forceCleanupFuture = null;
        }
    }

    @Override
    public void sessionIdChange(SessionData session) {
        this.redis.rename(this.getSessionExpireKey(session.getOldSessionId()), this.getSessionExpireKey(session.getId()));
        long expireCleanupInstant = NotificationExpirationManagement.roundUpToNextMinute(session.expiresAt());
        byte[] expirationsKey = this.getExpirationsKey(expireCleanupInstant);
        byte[] sessionKey = this.repository.sessionKey(session.getId());
        byte[] oldSessionKey = this.repository.sessionKey(session.getOldSessionId());
        this.redis.srem(expirationsKey, new byte[][]{oldSessionKey});
        this.redis.sadd(expirationsKey, new byte[][]{sessionKey});
        if (this.sticky) {
            long forceCleanupInstant = NotificationExpirationManagement.roundUpToNextMinute(expireCleanupInstant);
            byte[] forceExpirationsKey = this.getForcedExpirationsKey(forceCleanupInstant);
            this.redis.srem(forceExpirationsKey, new byte[][]{oldSessionKey});
            this.redis.sadd(forceExpirationsKey, new byte[][]{sessionKey});
        }
    }

    class SubscriptionRunner
    implements Runnable {
        private final SessionManager sessionManager;
        int attempt;
        long lastConnect;

        SubscriptionRunner(SessionManager sessionManager) {
            this.sessionManager = sessionManager;
        }

        @Override
        public void run() {
            logger.info("Registering subscriber for expiration events.");
            this.lastConnect = System.currentTimeMillis();
            this.attempt = 0;
            while (true) {
                try {
                    NotificationExpirationManagement.this.expirationListener = new ExpirationListener(this.sessionManager, NotificationExpirationManagement.this.keyExpirePrefix);
                    NotificationExpirationManagement.this.expirationListener.start(NotificationExpirationManagement.this.redis);
                    logger.info("Stopped subscribing for expiration events.");
                    return;
                }
                catch (Exception e) {
                    if (Thread.interrupted()) {
                        return;
                    }
                    if (NotificationExpirationManagement.this.redis.isRedisException(e) && e.getCause() instanceof InterruptedException) {
                        logger.warn("Interrupted subscribtion for expiration events.");
                        return;
                    }
                    this.retryOnException(e);
                    if (!Thread.interrupted()) continue;
                    return;
                }
                break;
            }
        }

        void retryOnException(Exception e) {
            logger.error("Failure during subscribing to redis events. Will be retrying...", e);
            long instant = System.currentTimeMillis();
            long delta = instant - this.lastConnect;
            if (delta > RESET_RETRY_THRESHOLD) {
                this.attempt = 0;
            } else {
                ++this.attempt;
                if (this.attempt >= MAX_CONNECTION_ERRORS) {
                    logger.error("Unable to connect to redis servers after trying {} times. Stopped listening to expiration events.", (Object)this.attempt);
                    throw new IllegalStateException("Stopped listening to expiration events.", e);
                }
                this.doWait();
                this.lastConnect = instant;
            }
        }

        void doWait() {
            try {
                Thread.sleep(this.getDelay());
            }
            catch (InterruptedException e) {
                throw new WrappedException(e);
            }
        }

        long getDelay() {
            return TimeUnit.SECONDS.toMillis(FIBONACCI_DELAY_PATTERN[this.attempt]);
        }
    }

    final class ExpirationManagement {
        private long expireCleanupInstant;
        private byte[] sessionKey;
        private int sessionExpireInSeconds;
        private byte[] expirationsKey;
        long forceCleanupInstant;
        byte[] forceExpirationsKey;

        ExpirationManagement() {
        }

        void manageExpiration(SessionData session) {
            this.prepareKeys(session);
            this.manageCleanupKeys(session);
            this.manageSessionFailover(session);
            byte[] sessionExpireKey = NotificationExpirationManagement.this.getSessionExpireKey(session.getId());
            if (this.sessionExpireInSeconds <= 0) {
                NotificationExpirationManagement.this.redis.del(new byte[][]{sessionExpireKey});
                NotificationExpirationManagement.this.redis.persist(this.sessionKey);
            } else {
                NotificationExpirationManagement.this.redis.sadd(this.expirationsKey, new byte[][]{this.sessionKey});
                NotificationExpirationManagement.this.redis.expireAt(this.expirationsKey, TimeUnit.MILLISECONDS.toSeconds(this.expireCleanupInstant) + 300L);
                if (NotificationExpirationManagement.this.sticky) {
                    NotificationExpirationManagement.this.redis.sadd(this.forceExpirationsKey, new byte[][]{this.sessionKey});
                    NotificationExpirationManagement.this.redis.expireAt(this.forceExpirationsKey, TimeUnit.MILLISECONDS.toSeconds(this.forceCleanupInstant) + 300L);
                }
                NotificationExpirationManagement.this.redis.setex(sessionExpireKey, this.sessionExpireInSeconds, EMPTY_STRING);
                NotificationExpirationManagement.this.redis.expire(this.sessionKey, this.sessionExpireInSeconds + 300);
            }
        }

        private void manageSessionFailover(SessionData session) {
            if (NotificationExpirationManagement.this.sticky && !NotificationExpirationManagement.this.owner.equals(session.getPreviousOwner())) {
                NotificationExpirationManagement.this.redis.del(new byte[][]{NotificationExpirationManagement.this.getSessionExpireKey(session.getPreviousOwner(), session.getId())});
            }
        }

        private void prepareKeys(SessionData session) {
            this.sessionKey = NotificationExpirationManagement.this.repository.sessionKey(session.getId());
            this.sessionExpireInSeconds = session.getMaxInactiveInterval();
            this.expireCleanupInstant = NotificationExpirationManagement.roundUpToNextMinute(session.expiresAt());
            this.expirationsKey = NotificationExpirationManagement.this.getExpirationsKey(this.expireCleanupInstant);
            if (NotificationExpirationManagement.this.sticky) {
                this.forceCleanupInstant = NotificationExpirationManagement.roundUpToNextMinute(this.expireCleanupInstant);
                this.forceExpirationsKey = NotificationExpirationManagement.this.getForcedExpirationsKey(this.forceCleanupInstant);
            } else {
                this.forceCleanupInstant = 0L;
                this.forceExpirationsKey = null;
            }
        }

        private void manageCleanupKeys(SessionData session) {
            if (!session.isNew()) {
                long originalCleanupInstant = NotificationExpirationManagement.roundUpToNextMinute(session.getOriginalLastAccessed());
                if (this.expireCleanupInstant != originalCleanupInstant) {
                    byte[] originalExpirationsKey = NotificationExpirationManagement.this.getExpirationsKey(originalCleanupInstant);
                    NotificationExpirationManagement.this.redis.srem(originalExpirationsKey, new byte[][]{this.sessionKey});
                    if (NotificationExpirationManagement.this.sticky) {
                        long originalForceCleanupInstant = NotificationExpirationManagement.roundUpToNextMinute(this.expireCleanupInstant);
                        byte[] originalForcedExpirationsKey = NotificationExpirationManagement.this.getForcedExpirationsKey(originalForceCleanupInstant);
                        NotificationExpirationManagement.this.redis.srem(originalForcedExpirationsKey, new byte[][]{this.sessionKey});
                    }
                } else if (this.sessionExpireInSeconds <= 0) {
                    NotificationExpirationManagement.this.redis.srem(this.expirationsKey, new byte[][]{this.sessionKey});
                    if (NotificationExpirationManagement.this.sticky) {
                        NotificationExpirationManagement.this.redis.srem(this.forceExpirationsKey, new byte[][]{this.sessionKey});
                    }
                }
            }
        }
    }

    final class TriggerExpiredSessionsTask
    implements Runnable {
        TriggerExpiredSessionsTask() {
        }

        @Override
        public void run() {
            long prevMin = NotificationExpirationManagement.roundDownMinute(System.currentTimeMillis());
            logger.debug("Triggering up sessions expiring at {}", (Object)prevMin);
            byte[] key = NotificationExpirationManagement.this.getExpirationsKey(prevMin);
            Set<byte[]> sessionsToExpire = NotificationExpirationManagement.this.getKeysToExpire(key);
            if (sessionsToExpire == null || sessionsToExpire.isEmpty()) {
                return;
            }
            for (byte[] session : sessionsToExpire) {
                if (logger.isDebugEnabled()) {
                    logger.debug("Expiring session {}", (Object)new String(session));
                }
                byte[] sessionExpireKey = NotificationExpirationManagement.this.getSessionExpireKey(SafeEncoder.encode(session));
                NotificationExpirationManagement.this.redis.exists(sessionExpireKey);
            }
        }
    }

    final class CleanHangingSessionsTask
    implements Runnable {
        private final SessionManager sessionManager;

        CleanHangingSessionsTask(SessionManager sessionManager) {
            this.sessionManager = sessionManager;
        }

        @Override
        public void run() {
            long prevMin = NotificationExpirationManagement.roundDownMinute(System.currentTimeMillis());
            logger.debug("Cleaning up sessions expiring at {}", (Object)prevMin);
            byte[] key = NotificationExpirationManagement.this.getForcedExpirationsKey(prevMin);
            Set<byte[]> sessionsToExpire = NotificationExpirationManagement.this.getKeysToExpire(key);
            if (sessionsToExpire == null || sessionsToExpire.isEmpty()) {
                return;
            }
            for (byte[] session : sessionsToExpire) {
                if (logger.isDebugEnabled()) {
                    logger.debug("Cleaning-up session {}", (Object)new String(session));
                }
                if (!NotificationExpirationManagement.this.redis.exists(NotificationExpirationManagement.this.repository.getSessionKey(session)).booleanValue()) continue;
                this.sessionManager.deleteAsync(SafeEncoder.encode(session), true);
            }
        }
    }
}

