/*
 * Copyright 2010-2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License").
 * You may not use this file except in compliance with the License.
 * A copy of the License is located at
 *
 *  http://aws.amazon.com/apache2.0
 *
 * or in the "license" file accompanying this file. This file is distributed
 * on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 * express or implied. See the License for the specific language governing
 * permissions and limitations under the License.
 */

package com.amazonaws.services.iot.model;

import java.io.Serializable;

import com.amazonaws.AmazonWebServiceRequest;

/**
 * <p>
 * Creates a thing record in the registry. If this call is made multiple times
 * using the same thing name and configuration, the call will succeed. If this
 * call is made with the same thing name but different configuration a
 * <code>ResourceAlreadyExistsException</code> is thrown.
 * </p>
 * <note>
 * <p>
 * This is a control plane operation. See <a href=
 * "https://docs.aws.amazon.com/iot/latest/developerguide/iot-authorization.html"
 * >Authorization</a> for information about authorizing control plane actions.
 * </p>
 * </note>
 */
public class CreateThingRequest extends AmazonWebServiceRequest implements Serializable {
    /**
     * <p>
     * The name of the thing to create.
     * </p>
     * <p>
     * You can't change a thing's name after you create it. To change a thing's
     * name, you must create a new thing, give it the new name, and then delete
     * the old thing.
     * </p>
     * <p>
     * <b>Constraints:</b><br/>
     * <b>Length: </b>1 - 128<br/>
     * <b>Pattern: </b>[a-zA-Z0-9:_-]+<br/>
     */
    private String thingName;

    /**
     * <p>
     * The name of the thing type associated with the new thing.
     * </p>
     * <p>
     * <b>Constraints:</b><br/>
     * <b>Length: </b>1 - 128<br/>
     * <b>Pattern: </b>[a-zA-Z0-9:_-]+<br/>
     */
    private String thingTypeName;

    /**
     * <p>
     * The attribute payload, which consists of up to three name/value pairs in
     * a JSON document. For example:
     * </p>
     * <p>
     * <code>{\"attributes\":{\"string1\":\"string2\"}}</code>
     * </p>
     */
    private AttributePayload attributePayload;

    /**
     * <p>
     * The name of the billing group the thing will be added to.
     * </p>
     * <p>
     * <b>Constraints:</b><br/>
     * <b>Length: </b>1 - 128<br/>
     * <b>Pattern: </b>[a-zA-Z0-9:_-]+<br/>
     */
    private String billingGroupName;

    /**
     * <p>
     * The name of the thing to create.
     * </p>
     * <p>
     * You can't change a thing's name after you create it. To change a thing's
     * name, you must create a new thing, give it the new name, and then delete
     * the old thing.
     * </p>
     * <p>
     * <b>Constraints:</b><br/>
     * <b>Length: </b>1 - 128<br/>
     * <b>Pattern: </b>[a-zA-Z0-9:_-]+<br/>
     *
     * @return <p>
     *         The name of the thing to create.
     *         </p>
     *         <p>
     *         You can't change a thing's name after you create it. To change a
     *         thing's name, you must create a new thing, give it the new name,
     *         and then delete the old thing.
     *         </p>
     */
    public String getThingName() {
        return thingName;
    }

    /**
     * <p>
     * The name of the thing to create.
     * </p>
     * <p>
     * You can't change a thing's name after you create it. To change a thing's
     * name, you must create a new thing, give it the new name, and then delete
     * the old thing.
     * </p>
     * <p>
     * <b>Constraints:</b><br/>
     * <b>Length: </b>1 - 128<br/>
     * <b>Pattern: </b>[a-zA-Z0-9:_-]+<br/>
     *
     * @param thingName <p>
     *            The name of the thing to create.
     *            </p>
     *            <p>
     *            You can't change a thing's name after you create it. To change
     *            a thing's name, you must create a new thing, give it the new
     *            name, and then delete the old thing.
     *            </p>
     */
    public void setThingName(String thingName) {
        this.thingName = thingName;
    }

    /**
     * <p>
     * The name of the thing to create.
     * </p>
     * <p>
     * You can't change a thing's name after you create it. To change a thing's
     * name, you must create a new thing, give it the new name, and then delete
     * the old thing.
     * </p>
     * <p>
     * Returns a reference to this object so that method calls can be chained
     * together.
     * <p>
     * <b>Constraints:</b><br/>
     * <b>Length: </b>1 - 128<br/>
     * <b>Pattern: </b>[a-zA-Z0-9:_-]+<br/>
     *
     * @param thingName <p>
     *            The name of the thing to create.
     *            </p>
     *            <p>
     *            You can't change a thing's name after you create it. To change
     *            a thing's name, you must create a new thing, give it the new
     *            name, and then delete the old thing.
     *            </p>
     * @return A reference to this updated object so that method calls can be
     *         chained together.
     */
    public CreateThingRequest withThingName(String thingName) {
        this.thingName = thingName;
        return this;
    }

    /**
     * <p>
     * The name of the thing type associated with the new thing.
     * </p>
     * <p>
     * <b>Constraints:</b><br/>
     * <b>Length: </b>1 - 128<br/>
     * <b>Pattern: </b>[a-zA-Z0-9:_-]+<br/>
     *
     * @return <p>
     *         The name of the thing type associated with the new thing.
     *         </p>
     */
    public String getThingTypeName() {
        return thingTypeName;
    }

    /**
     * <p>
     * The name of the thing type associated with the new thing.
     * </p>
     * <p>
     * <b>Constraints:</b><br/>
     * <b>Length: </b>1 - 128<br/>
     * <b>Pattern: </b>[a-zA-Z0-9:_-]+<br/>
     *
     * @param thingTypeName <p>
     *            The name of the thing type associated with the new thing.
     *            </p>
     */
    public void setThingTypeName(String thingTypeName) {
        this.thingTypeName = thingTypeName;
    }

    /**
     * <p>
     * The name of the thing type associated with the new thing.
     * </p>
     * <p>
     * Returns a reference to this object so that method calls can be chained
     * together.
     * <p>
     * <b>Constraints:</b><br/>
     * <b>Length: </b>1 - 128<br/>
     * <b>Pattern: </b>[a-zA-Z0-9:_-]+<br/>
     *
     * @param thingTypeName <p>
     *            The name of the thing type associated with the new thing.
     *            </p>
     * @return A reference to this updated object so that method calls can be
     *         chained together.
     */
    public CreateThingRequest withThingTypeName(String thingTypeName) {
        this.thingTypeName = thingTypeName;
        return this;
    }

    /**
     * <p>
     * The attribute payload, which consists of up to three name/value pairs in
     * a JSON document. For example:
     * </p>
     * <p>
     * <code>{\"attributes\":{\"string1\":\"string2\"}}</code>
     * </p>
     *
     * @return <p>
     *         The attribute payload, which consists of up to three name/value
     *         pairs in a JSON document. For example:
     *         </p>
     *         <p>
     *         <code>{\"attributes\":{\"string1\":\"string2\"}}</code>
     *         </p>
     */
    public AttributePayload getAttributePayload() {
        return attributePayload;
    }

    /**
     * <p>
     * The attribute payload, which consists of up to three name/value pairs in
     * a JSON document. For example:
     * </p>
     * <p>
     * <code>{\"attributes\":{\"string1\":\"string2\"}}</code>
     * </p>
     *
     * @param attributePayload <p>
     *            The attribute payload, which consists of up to three
     *            name/value pairs in a JSON document. For example:
     *            </p>
     *            <p>
     *            <code>{\"attributes\":{\"string1\":\"string2\"}}</code>
     *            </p>
     */
    public void setAttributePayload(AttributePayload attributePayload) {
        this.attributePayload = attributePayload;
    }

    /**
     * <p>
     * The attribute payload, which consists of up to three name/value pairs in
     * a JSON document. For example:
     * </p>
     * <p>
     * <code>{\"attributes\":{\"string1\":\"string2\"}}</code>
     * </p>
     * <p>
     * Returns a reference to this object so that method calls can be chained
     * together.
     *
     * @param attributePayload <p>
     *            The attribute payload, which consists of up to three
     *            name/value pairs in a JSON document. For example:
     *            </p>
     *            <p>
     *            <code>{\"attributes\":{\"string1\":\"string2\"}}</code>
     *            </p>
     * @return A reference to this updated object so that method calls can be
     *         chained together.
     */
    public CreateThingRequest withAttributePayload(AttributePayload attributePayload) {
        this.attributePayload = attributePayload;
        return this;
    }

    /**
     * <p>
     * The name of the billing group the thing will be added to.
     * </p>
     * <p>
     * <b>Constraints:</b><br/>
     * <b>Length: </b>1 - 128<br/>
     * <b>Pattern: </b>[a-zA-Z0-9:_-]+<br/>
     *
     * @return <p>
     *         The name of the billing group the thing will be added to.
     *         </p>
     */
    public String getBillingGroupName() {
        return billingGroupName;
    }

    /**
     * <p>
     * The name of the billing group the thing will be added to.
     * </p>
     * <p>
     * <b>Constraints:</b><br/>
     * <b>Length: </b>1 - 128<br/>
     * <b>Pattern: </b>[a-zA-Z0-9:_-]+<br/>
     *
     * @param billingGroupName <p>
     *            The name of the billing group the thing will be added to.
     *            </p>
     */
    public void setBillingGroupName(String billingGroupName) {
        this.billingGroupName = billingGroupName;
    }

    /**
     * <p>
     * The name of the billing group the thing will be added to.
     * </p>
     * <p>
     * Returns a reference to this object so that method calls can be chained
     * together.
     * <p>
     * <b>Constraints:</b><br/>
     * <b>Length: </b>1 - 128<br/>
     * <b>Pattern: </b>[a-zA-Z0-9:_-]+<br/>
     *
     * @param billingGroupName <p>
     *            The name of the billing group the thing will be added to.
     *            </p>
     * @return A reference to this updated object so that method calls can be
     *         chained together.
     */
    public CreateThingRequest withBillingGroupName(String billingGroupName) {
        this.billingGroupName = billingGroupName;
        return this;
    }

    /**
     * Returns a string representation of this object; useful for testing and
     * debugging.
     *
     * @return A string representation of this object.
     * @see java.lang.Object#toString()
     */
    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("{");
        if (getThingName() != null)
            sb.append("thingName: " + getThingName() + ",");
        if (getThingTypeName() != null)
            sb.append("thingTypeName: " + getThingTypeName() + ",");
        if (getAttributePayload() != null)
            sb.append("attributePayload: " + getAttributePayload() + ",");
        if (getBillingGroupName() != null)
            sb.append("billingGroupName: " + getBillingGroupName());
        sb.append("}");
        return sb.toString();
    }

    @Override
    public int hashCode() {
        final int prime = 31;
        int hashCode = 1;

        hashCode = prime * hashCode + ((getThingName() == null) ? 0 : getThingName().hashCode());
        hashCode = prime * hashCode
                + ((getThingTypeName() == null) ? 0 : getThingTypeName().hashCode());
        hashCode = prime * hashCode
                + ((getAttributePayload() == null) ? 0 : getAttributePayload().hashCode());
        hashCode = prime * hashCode
                + ((getBillingGroupName() == null) ? 0 : getBillingGroupName().hashCode());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj)
            return true;
        if (obj == null)
            return false;

        if (obj instanceof CreateThingRequest == false)
            return false;
        CreateThingRequest other = (CreateThingRequest) obj;

        if (other.getThingName() == null ^ this.getThingName() == null)
            return false;
        if (other.getThingName() != null
                && other.getThingName().equals(this.getThingName()) == false)
            return false;
        if (other.getThingTypeName() == null ^ this.getThingTypeName() == null)
            return false;
        if (other.getThingTypeName() != null
                && other.getThingTypeName().equals(this.getThingTypeName()) == false)
            return false;
        if (other.getAttributePayload() == null ^ this.getAttributePayload() == null)
            return false;
        if (other.getAttributePayload() != null
                && other.getAttributePayload().equals(this.getAttributePayload()) == false)
            return false;
        if (other.getBillingGroupName() == null ^ this.getBillingGroupName() == null)
            return false;
        if (other.getBillingGroupName() != null
                && other.getBillingGroupName().equals(this.getBillingGroupName()) == false)
            return false;
        return true;
    }
}
