/*
 * Copyright 2010-2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License").
 * You may not use this file except in compliance with the License.
 * A copy of the License is located at
 *
 *  http://aws.amazon.com/apache2.0
 *
 * or in the "license" file accompanying this file. This file is distributed
 * on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 * express or implied. See the License for the specific language governing
 * permissions and limitations under the License.
 */

package com.amazonaws.services.transcribe.model;

import java.io.Serializable;

/**
 * <p>
 * Allows additional optional settings in your request, including channel
 * identification, alternative transcriptions, and speaker labeling; allows you
 * to apply custom vocabularies to your medical transcription job.
 * </p>
 */
public class MedicalTranscriptionSetting implements Serializable {
    /**
     * <p>
     * Enables speaker identification (diarization) in your transcription
     * output. Speaker identification labels the speech from individual speakers
     * in your media file.
     * </p>
     * <p>
     * If you enable <code>ShowSpeakerLabels</code> in your request, you must
     * also include <code>MaxSpeakerLabels</code>.
     * </p>
     * <p>
     * You can't include both <code>ShowSpeakerLabels</code> and
     * <code>ChannelIdentification</code> in the same request. Including both
     * parameters returns a <code>BadRequestException</code>.
     * </p>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/transcribe/latest/dg/diarization.html"
     * >Identifying speakers (diarization)</a>.
     * </p>
     */
    private Boolean showSpeakerLabels;

    /**
     * <p>
     * Specify the maximum number of speakers you want to identify in your
     * media.
     * </p>
     * <p>
     * Note that if your media contains more speakers than the specified number,
     * multiple speakers will be identified as a single speaker.
     * </p>
     * <p>
     * If you specify the <code>MaxSpeakerLabels</code> field, you must set the
     * <code>ShowSpeakerLabels</code> field to true.
     * </p>
     * <p>
     * <b>Constraints:</b><br/>
     * <b>Range: </b>2 - 10<br/>
     */
    private Integer maxSpeakerLabels;

    /**
     * <p>
     * Enables channel identification in multi-channel audio.
     * </p>
     * <p>
     * Channel identification transcribes the audio on each channel
     * independently, then appends the output for each channel into one
     * transcript.
     * </p>
     * <p>
     * If you have multi-channel audio and do not enable channel identification,
     * your audio is transcribed in a continuous manner and your transcript does
     * not separate the speech by channel.
     * </p>
     * <p>
     * You can't include both <code>ShowSpeakerLabels</code> and
     * <code>ChannelIdentification</code> in the same request. Including both
     * parameters returns a <code>BadRequestException</code>.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/transcribe/latest/dg/channel-id.html"
     * >Transcribing multi-channel audio</a>.
     * </p>
     */
    private Boolean channelIdentification;

    /**
     * <p>
     * To include alternative transcriptions within your transcription output,
     * include <code>ShowAlternatives</code> in your transcription request.
     * </p>
     * <p>
     * If you include <code>ShowAlternatives</code>, you must also include
     * <code>MaxAlternatives</code>, which is the maximum number of alternative
     * transcriptions you want Amazon Transcribe Medical to generate.
     * </p>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/transcribe/latest/dg/how-alternatives.html"
     * >Alternative transcriptions</a>.
     * </p>
     */
    private Boolean showAlternatives;

    /**
     * <p>
     * Indicate the maximum number of alternative transcriptions you want Amazon
     * Transcribe Medical to include in your transcript.
     * </p>
     * <p>
     * If you select a number greater than the number of alternative
     * transcriptions generated by Amazon Transcribe Medical, only the actual
     * number of alternative transcriptions are included.
     * </p>
     * <p>
     * If you include <code>MaxAlternatives</code> in your request, you must
     * also include <code>ShowAlternatives</code> with a value of
     * <code>true</code>.
     * </p>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/transcribe/latest/dg/how-alternatives.html"
     * >Alternative transcriptions</a>.
     * </p>
     * <p>
     * <b>Constraints:</b><br/>
     * <b>Range: </b>2 - 10<br/>
     */
    private Integer maxAlternatives;

    /**
     * <p>
     * The name of the custom vocabulary you want to use when processing your
     * medical transcription job. Vocabulary names are case sensitive.
     * </p>
     * <p>
     * The language of the specified vocabulary must match the language code you
     * specify in your transcription request. If the languages don't match, the
     * vocabulary isn't applied. There are no errors or warnings associated with
     * a language mismatch. US English (<code>en-US</code>) is the only valid
     * language for Amazon Transcribe Medical.
     * </p>
     * <p>
     * <b>Constraints:</b><br/>
     * <b>Length: </b>1 - 200<br/>
     * <b>Pattern: </b>^[0-9a-zA-Z._-]+<br/>
     */
    private String vocabularyName;

    /**
     * <p>
     * Enables speaker identification (diarization) in your transcription
     * output. Speaker identification labels the speech from individual speakers
     * in your media file.
     * </p>
     * <p>
     * If you enable <code>ShowSpeakerLabels</code> in your request, you must
     * also include <code>MaxSpeakerLabels</code>.
     * </p>
     * <p>
     * You can't include both <code>ShowSpeakerLabels</code> and
     * <code>ChannelIdentification</code> in the same request. Including both
     * parameters returns a <code>BadRequestException</code>.
     * </p>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/transcribe/latest/dg/diarization.html"
     * >Identifying speakers (diarization)</a>.
     * </p>
     *
     * @return <p>
     *         Enables speaker identification (diarization) in your
     *         transcription output. Speaker identification labels the speech
     *         from individual speakers in your media file.
     *         </p>
     *         <p>
     *         If you enable <code>ShowSpeakerLabels</code> in your request, you
     *         must also include <code>MaxSpeakerLabels</code>.
     *         </p>
     *         <p>
     *         You can't include both <code>ShowSpeakerLabels</code> and
     *         <code>ChannelIdentification</code> in the same request. Including
     *         both parameters returns a <code>BadRequestException</code>.
     *         </p>
     *         <p>
     *         For more information, see <a href=
     *         "https://docs.aws.amazon.com/transcribe/latest/dg/diarization.html"
     *         >Identifying speakers (diarization)</a>.
     *         </p>
     */
    public Boolean isShowSpeakerLabels() {
        return showSpeakerLabels;
    }

    /**
     * <p>
     * Enables speaker identification (diarization) in your transcription
     * output. Speaker identification labels the speech from individual speakers
     * in your media file.
     * </p>
     * <p>
     * If you enable <code>ShowSpeakerLabels</code> in your request, you must
     * also include <code>MaxSpeakerLabels</code>.
     * </p>
     * <p>
     * You can't include both <code>ShowSpeakerLabels</code> and
     * <code>ChannelIdentification</code> in the same request. Including both
     * parameters returns a <code>BadRequestException</code>.
     * </p>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/transcribe/latest/dg/diarization.html"
     * >Identifying speakers (diarization)</a>.
     * </p>
     *
     * @return <p>
     *         Enables speaker identification (diarization) in your
     *         transcription output. Speaker identification labels the speech
     *         from individual speakers in your media file.
     *         </p>
     *         <p>
     *         If you enable <code>ShowSpeakerLabels</code> in your request, you
     *         must also include <code>MaxSpeakerLabels</code>.
     *         </p>
     *         <p>
     *         You can't include both <code>ShowSpeakerLabels</code> and
     *         <code>ChannelIdentification</code> in the same request. Including
     *         both parameters returns a <code>BadRequestException</code>.
     *         </p>
     *         <p>
     *         For more information, see <a href=
     *         "https://docs.aws.amazon.com/transcribe/latest/dg/diarization.html"
     *         >Identifying speakers (diarization)</a>.
     *         </p>
     */
    public Boolean getShowSpeakerLabels() {
        return showSpeakerLabels;
    }

    /**
     * <p>
     * Enables speaker identification (diarization) in your transcription
     * output. Speaker identification labels the speech from individual speakers
     * in your media file.
     * </p>
     * <p>
     * If you enable <code>ShowSpeakerLabels</code> in your request, you must
     * also include <code>MaxSpeakerLabels</code>.
     * </p>
     * <p>
     * You can't include both <code>ShowSpeakerLabels</code> and
     * <code>ChannelIdentification</code> in the same request. Including both
     * parameters returns a <code>BadRequestException</code>.
     * </p>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/transcribe/latest/dg/diarization.html"
     * >Identifying speakers (diarization)</a>.
     * </p>
     *
     * @param showSpeakerLabels <p>
     *            Enables speaker identification (diarization) in your
     *            transcription output. Speaker identification labels the speech
     *            from individual speakers in your media file.
     *            </p>
     *            <p>
     *            If you enable <code>ShowSpeakerLabels</code> in your request,
     *            you must also include <code>MaxSpeakerLabels</code>.
     *            </p>
     *            <p>
     *            You can't include both <code>ShowSpeakerLabels</code> and
     *            <code>ChannelIdentification</code> in the same request.
     *            Including both parameters returns a
     *            <code>BadRequestException</code>.
     *            </p>
     *            <p>
     *            For more information, see <a href=
     *            "https://docs.aws.amazon.com/transcribe/latest/dg/diarization.html"
     *            >Identifying speakers (diarization)</a>.
     *            </p>
     */
    public void setShowSpeakerLabels(Boolean showSpeakerLabels) {
        this.showSpeakerLabels = showSpeakerLabels;
    }

    /**
     * <p>
     * Enables speaker identification (diarization) in your transcription
     * output. Speaker identification labels the speech from individual speakers
     * in your media file.
     * </p>
     * <p>
     * If you enable <code>ShowSpeakerLabels</code> in your request, you must
     * also include <code>MaxSpeakerLabels</code>.
     * </p>
     * <p>
     * You can't include both <code>ShowSpeakerLabels</code> and
     * <code>ChannelIdentification</code> in the same request. Including both
     * parameters returns a <code>BadRequestException</code>.
     * </p>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/transcribe/latest/dg/diarization.html"
     * >Identifying speakers (diarization)</a>.
     * </p>
     * <p>
     * Returns a reference to this object so that method calls can be chained
     * together.
     *
     * @param showSpeakerLabels <p>
     *            Enables speaker identification (diarization) in your
     *            transcription output. Speaker identification labels the speech
     *            from individual speakers in your media file.
     *            </p>
     *            <p>
     *            If you enable <code>ShowSpeakerLabels</code> in your request,
     *            you must also include <code>MaxSpeakerLabels</code>.
     *            </p>
     *            <p>
     *            You can't include both <code>ShowSpeakerLabels</code> and
     *            <code>ChannelIdentification</code> in the same request.
     *            Including both parameters returns a
     *            <code>BadRequestException</code>.
     *            </p>
     *            <p>
     *            For more information, see <a href=
     *            "https://docs.aws.amazon.com/transcribe/latest/dg/diarization.html"
     *            >Identifying speakers (diarization)</a>.
     *            </p>
     * @return A reference to this updated object so that method calls can be
     *         chained together.
     */
    public MedicalTranscriptionSetting withShowSpeakerLabels(Boolean showSpeakerLabels) {
        this.showSpeakerLabels = showSpeakerLabels;
        return this;
    }

    /**
     * <p>
     * Specify the maximum number of speakers you want to identify in your
     * media.
     * </p>
     * <p>
     * Note that if your media contains more speakers than the specified number,
     * multiple speakers will be identified as a single speaker.
     * </p>
     * <p>
     * If you specify the <code>MaxSpeakerLabels</code> field, you must set the
     * <code>ShowSpeakerLabels</code> field to true.
     * </p>
     * <p>
     * <b>Constraints:</b><br/>
     * <b>Range: </b>2 - 10<br/>
     *
     * @return <p>
     *         Specify the maximum number of speakers you want to identify in
     *         your media.
     *         </p>
     *         <p>
     *         Note that if your media contains more speakers than the specified
     *         number, multiple speakers will be identified as a single speaker.
     *         </p>
     *         <p>
     *         If you specify the <code>MaxSpeakerLabels</code> field, you must
     *         set the <code>ShowSpeakerLabels</code> field to true.
     *         </p>
     */
    public Integer getMaxSpeakerLabels() {
        return maxSpeakerLabels;
    }

    /**
     * <p>
     * Specify the maximum number of speakers you want to identify in your
     * media.
     * </p>
     * <p>
     * Note that if your media contains more speakers than the specified number,
     * multiple speakers will be identified as a single speaker.
     * </p>
     * <p>
     * If you specify the <code>MaxSpeakerLabels</code> field, you must set the
     * <code>ShowSpeakerLabels</code> field to true.
     * </p>
     * <p>
     * <b>Constraints:</b><br/>
     * <b>Range: </b>2 - 10<br/>
     *
     * @param maxSpeakerLabels <p>
     *            Specify the maximum number of speakers you want to identify in
     *            your media.
     *            </p>
     *            <p>
     *            Note that if your media contains more speakers than the
     *            specified number, multiple speakers will be identified as a
     *            single speaker.
     *            </p>
     *            <p>
     *            If you specify the <code>MaxSpeakerLabels</code> field, you
     *            must set the <code>ShowSpeakerLabels</code> field to true.
     *            </p>
     */
    public void setMaxSpeakerLabels(Integer maxSpeakerLabels) {
        this.maxSpeakerLabels = maxSpeakerLabels;
    }

    /**
     * <p>
     * Specify the maximum number of speakers you want to identify in your
     * media.
     * </p>
     * <p>
     * Note that if your media contains more speakers than the specified number,
     * multiple speakers will be identified as a single speaker.
     * </p>
     * <p>
     * If you specify the <code>MaxSpeakerLabels</code> field, you must set the
     * <code>ShowSpeakerLabels</code> field to true.
     * </p>
     * <p>
     * Returns a reference to this object so that method calls can be chained
     * together.
     * <p>
     * <b>Constraints:</b><br/>
     * <b>Range: </b>2 - 10<br/>
     *
     * @param maxSpeakerLabels <p>
     *            Specify the maximum number of speakers you want to identify in
     *            your media.
     *            </p>
     *            <p>
     *            Note that if your media contains more speakers than the
     *            specified number, multiple speakers will be identified as a
     *            single speaker.
     *            </p>
     *            <p>
     *            If you specify the <code>MaxSpeakerLabels</code> field, you
     *            must set the <code>ShowSpeakerLabels</code> field to true.
     *            </p>
     * @return A reference to this updated object so that method calls can be
     *         chained together.
     */
    public MedicalTranscriptionSetting withMaxSpeakerLabels(Integer maxSpeakerLabels) {
        this.maxSpeakerLabels = maxSpeakerLabels;
        return this;
    }

    /**
     * <p>
     * Enables channel identification in multi-channel audio.
     * </p>
     * <p>
     * Channel identification transcribes the audio on each channel
     * independently, then appends the output for each channel into one
     * transcript.
     * </p>
     * <p>
     * If you have multi-channel audio and do not enable channel identification,
     * your audio is transcribed in a continuous manner and your transcript does
     * not separate the speech by channel.
     * </p>
     * <p>
     * You can't include both <code>ShowSpeakerLabels</code> and
     * <code>ChannelIdentification</code> in the same request. Including both
     * parameters returns a <code>BadRequestException</code>.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/transcribe/latest/dg/channel-id.html"
     * >Transcribing multi-channel audio</a>.
     * </p>
     *
     * @return <p>
     *         Enables channel identification in multi-channel audio.
     *         </p>
     *         <p>
     *         Channel identification transcribes the audio on each channel
     *         independently, then appends the output for each channel into one
     *         transcript.
     *         </p>
     *         <p>
     *         If you have multi-channel audio and do not enable channel
     *         identification, your audio is transcribed in a continuous manner
     *         and your transcript does not separate the speech by channel.
     *         </p>
     *         <p>
     *         You can't include both <code>ShowSpeakerLabels</code> and
     *         <code>ChannelIdentification</code> in the same request. Including
     *         both parameters returns a <code>BadRequestException</code>.
     *         </p>
     *         <p>
     *         For more information, see <a href=
     *         "https://docs.aws.amazon.com/transcribe/latest/dg/channel-id.html"
     *         >Transcribing multi-channel audio</a>.
     *         </p>
     */
    public Boolean isChannelIdentification() {
        return channelIdentification;
    }

    /**
     * <p>
     * Enables channel identification in multi-channel audio.
     * </p>
     * <p>
     * Channel identification transcribes the audio on each channel
     * independently, then appends the output for each channel into one
     * transcript.
     * </p>
     * <p>
     * If you have multi-channel audio and do not enable channel identification,
     * your audio is transcribed in a continuous manner and your transcript does
     * not separate the speech by channel.
     * </p>
     * <p>
     * You can't include both <code>ShowSpeakerLabels</code> and
     * <code>ChannelIdentification</code> in the same request. Including both
     * parameters returns a <code>BadRequestException</code>.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/transcribe/latest/dg/channel-id.html"
     * >Transcribing multi-channel audio</a>.
     * </p>
     *
     * @return <p>
     *         Enables channel identification in multi-channel audio.
     *         </p>
     *         <p>
     *         Channel identification transcribes the audio on each channel
     *         independently, then appends the output for each channel into one
     *         transcript.
     *         </p>
     *         <p>
     *         If you have multi-channel audio and do not enable channel
     *         identification, your audio is transcribed in a continuous manner
     *         and your transcript does not separate the speech by channel.
     *         </p>
     *         <p>
     *         You can't include both <code>ShowSpeakerLabels</code> and
     *         <code>ChannelIdentification</code> in the same request. Including
     *         both parameters returns a <code>BadRequestException</code>.
     *         </p>
     *         <p>
     *         For more information, see <a href=
     *         "https://docs.aws.amazon.com/transcribe/latest/dg/channel-id.html"
     *         >Transcribing multi-channel audio</a>.
     *         </p>
     */
    public Boolean getChannelIdentification() {
        return channelIdentification;
    }

    /**
     * <p>
     * Enables channel identification in multi-channel audio.
     * </p>
     * <p>
     * Channel identification transcribes the audio on each channel
     * independently, then appends the output for each channel into one
     * transcript.
     * </p>
     * <p>
     * If you have multi-channel audio and do not enable channel identification,
     * your audio is transcribed in a continuous manner and your transcript does
     * not separate the speech by channel.
     * </p>
     * <p>
     * You can't include both <code>ShowSpeakerLabels</code> and
     * <code>ChannelIdentification</code> in the same request. Including both
     * parameters returns a <code>BadRequestException</code>.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/transcribe/latest/dg/channel-id.html"
     * >Transcribing multi-channel audio</a>.
     * </p>
     *
     * @param channelIdentification <p>
     *            Enables channel identification in multi-channel audio.
     *            </p>
     *            <p>
     *            Channel identification transcribes the audio on each channel
     *            independently, then appends the output for each channel into
     *            one transcript.
     *            </p>
     *            <p>
     *            If you have multi-channel audio and do not enable channel
     *            identification, your audio is transcribed in a continuous
     *            manner and your transcript does not separate the speech by
     *            channel.
     *            </p>
     *            <p>
     *            You can't include both <code>ShowSpeakerLabels</code> and
     *            <code>ChannelIdentification</code> in the same request.
     *            Including both parameters returns a
     *            <code>BadRequestException</code>.
     *            </p>
     *            <p>
     *            For more information, see <a href=
     *            "https://docs.aws.amazon.com/transcribe/latest/dg/channel-id.html"
     *            >Transcribing multi-channel audio</a>.
     *            </p>
     */
    public void setChannelIdentification(Boolean channelIdentification) {
        this.channelIdentification = channelIdentification;
    }

    /**
     * <p>
     * Enables channel identification in multi-channel audio.
     * </p>
     * <p>
     * Channel identification transcribes the audio on each channel
     * independently, then appends the output for each channel into one
     * transcript.
     * </p>
     * <p>
     * If you have multi-channel audio and do not enable channel identification,
     * your audio is transcribed in a continuous manner and your transcript does
     * not separate the speech by channel.
     * </p>
     * <p>
     * You can't include both <code>ShowSpeakerLabels</code> and
     * <code>ChannelIdentification</code> in the same request. Including both
     * parameters returns a <code>BadRequestException</code>.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/transcribe/latest/dg/channel-id.html"
     * >Transcribing multi-channel audio</a>.
     * </p>
     * <p>
     * Returns a reference to this object so that method calls can be chained
     * together.
     *
     * @param channelIdentification <p>
     *            Enables channel identification in multi-channel audio.
     *            </p>
     *            <p>
     *            Channel identification transcribes the audio on each channel
     *            independently, then appends the output for each channel into
     *            one transcript.
     *            </p>
     *            <p>
     *            If you have multi-channel audio and do not enable channel
     *            identification, your audio is transcribed in a continuous
     *            manner and your transcript does not separate the speech by
     *            channel.
     *            </p>
     *            <p>
     *            You can't include both <code>ShowSpeakerLabels</code> and
     *            <code>ChannelIdentification</code> in the same request.
     *            Including both parameters returns a
     *            <code>BadRequestException</code>.
     *            </p>
     *            <p>
     *            For more information, see <a href=
     *            "https://docs.aws.amazon.com/transcribe/latest/dg/channel-id.html"
     *            >Transcribing multi-channel audio</a>.
     *            </p>
     * @return A reference to this updated object so that method calls can be
     *         chained together.
     */
    public MedicalTranscriptionSetting withChannelIdentification(Boolean channelIdentification) {
        this.channelIdentification = channelIdentification;
        return this;
    }

    /**
     * <p>
     * To include alternative transcriptions within your transcription output,
     * include <code>ShowAlternatives</code> in your transcription request.
     * </p>
     * <p>
     * If you include <code>ShowAlternatives</code>, you must also include
     * <code>MaxAlternatives</code>, which is the maximum number of alternative
     * transcriptions you want Amazon Transcribe Medical to generate.
     * </p>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/transcribe/latest/dg/how-alternatives.html"
     * >Alternative transcriptions</a>.
     * </p>
     *
     * @return <p>
     *         To include alternative transcriptions within your transcription
     *         output, include <code>ShowAlternatives</code> in your
     *         transcription request.
     *         </p>
     *         <p>
     *         If you include <code>ShowAlternatives</code>, you must also
     *         include <code>MaxAlternatives</code>, which is the maximum number
     *         of alternative transcriptions you want Amazon Transcribe Medical
     *         to generate.
     *         </p>
     *         <p>
     *         For more information, see <a href=
     *         "https://docs.aws.amazon.com/transcribe/latest/dg/how-alternatives.html"
     *         >Alternative transcriptions</a>.
     *         </p>
     */
    public Boolean isShowAlternatives() {
        return showAlternatives;
    }

    /**
     * <p>
     * To include alternative transcriptions within your transcription output,
     * include <code>ShowAlternatives</code> in your transcription request.
     * </p>
     * <p>
     * If you include <code>ShowAlternatives</code>, you must also include
     * <code>MaxAlternatives</code>, which is the maximum number of alternative
     * transcriptions you want Amazon Transcribe Medical to generate.
     * </p>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/transcribe/latest/dg/how-alternatives.html"
     * >Alternative transcriptions</a>.
     * </p>
     *
     * @return <p>
     *         To include alternative transcriptions within your transcription
     *         output, include <code>ShowAlternatives</code> in your
     *         transcription request.
     *         </p>
     *         <p>
     *         If you include <code>ShowAlternatives</code>, you must also
     *         include <code>MaxAlternatives</code>, which is the maximum number
     *         of alternative transcriptions you want Amazon Transcribe Medical
     *         to generate.
     *         </p>
     *         <p>
     *         For more information, see <a href=
     *         "https://docs.aws.amazon.com/transcribe/latest/dg/how-alternatives.html"
     *         >Alternative transcriptions</a>.
     *         </p>
     */
    public Boolean getShowAlternatives() {
        return showAlternatives;
    }

    /**
     * <p>
     * To include alternative transcriptions within your transcription output,
     * include <code>ShowAlternatives</code> in your transcription request.
     * </p>
     * <p>
     * If you include <code>ShowAlternatives</code>, you must also include
     * <code>MaxAlternatives</code>, which is the maximum number of alternative
     * transcriptions you want Amazon Transcribe Medical to generate.
     * </p>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/transcribe/latest/dg/how-alternatives.html"
     * >Alternative transcriptions</a>.
     * </p>
     *
     * @param showAlternatives <p>
     *            To include alternative transcriptions within your
     *            transcription output, include <code>ShowAlternatives</code> in
     *            your transcription request.
     *            </p>
     *            <p>
     *            If you include <code>ShowAlternatives</code>, you must also
     *            include <code>MaxAlternatives</code>, which is the maximum
     *            number of alternative transcriptions you want Amazon
     *            Transcribe Medical to generate.
     *            </p>
     *            <p>
     *            For more information, see <a href=
     *            "https://docs.aws.amazon.com/transcribe/latest/dg/how-alternatives.html"
     *            >Alternative transcriptions</a>.
     *            </p>
     */
    public void setShowAlternatives(Boolean showAlternatives) {
        this.showAlternatives = showAlternatives;
    }

    /**
     * <p>
     * To include alternative transcriptions within your transcription output,
     * include <code>ShowAlternatives</code> in your transcription request.
     * </p>
     * <p>
     * If you include <code>ShowAlternatives</code>, you must also include
     * <code>MaxAlternatives</code>, which is the maximum number of alternative
     * transcriptions you want Amazon Transcribe Medical to generate.
     * </p>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/transcribe/latest/dg/how-alternatives.html"
     * >Alternative transcriptions</a>.
     * </p>
     * <p>
     * Returns a reference to this object so that method calls can be chained
     * together.
     *
     * @param showAlternatives <p>
     *            To include alternative transcriptions within your
     *            transcription output, include <code>ShowAlternatives</code> in
     *            your transcription request.
     *            </p>
     *            <p>
     *            If you include <code>ShowAlternatives</code>, you must also
     *            include <code>MaxAlternatives</code>, which is the maximum
     *            number of alternative transcriptions you want Amazon
     *            Transcribe Medical to generate.
     *            </p>
     *            <p>
     *            For more information, see <a href=
     *            "https://docs.aws.amazon.com/transcribe/latest/dg/how-alternatives.html"
     *            >Alternative transcriptions</a>.
     *            </p>
     * @return A reference to this updated object so that method calls can be
     *         chained together.
     */
    public MedicalTranscriptionSetting withShowAlternatives(Boolean showAlternatives) {
        this.showAlternatives = showAlternatives;
        return this;
    }

    /**
     * <p>
     * Indicate the maximum number of alternative transcriptions you want Amazon
     * Transcribe Medical to include in your transcript.
     * </p>
     * <p>
     * If you select a number greater than the number of alternative
     * transcriptions generated by Amazon Transcribe Medical, only the actual
     * number of alternative transcriptions are included.
     * </p>
     * <p>
     * If you include <code>MaxAlternatives</code> in your request, you must
     * also include <code>ShowAlternatives</code> with a value of
     * <code>true</code>.
     * </p>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/transcribe/latest/dg/how-alternatives.html"
     * >Alternative transcriptions</a>.
     * </p>
     * <p>
     * <b>Constraints:</b><br/>
     * <b>Range: </b>2 - 10<br/>
     *
     * @return <p>
     *         Indicate the maximum number of alternative transcriptions you
     *         want Amazon Transcribe Medical to include in your transcript.
     *         </p>
     *         <p>
     *         If you select a number greater than the number of alternative
     *         transcriptions generated by Amazon Transcribe Medical, only the
     *         actual number of alternative transcriptions are included.
     *         </p>
     *         <p>
     *         If you include <code>MaxAlternatives</code> in your request, you
     *         must also include <code>ShowAlternatives</code> with a value of
     *         <code>true</code>.
     *         </p>
     *         <p>
     *         For more information, see <a href=
     *         "https://docs.aws.amazon.com/transcribe/latest/dg/how-alternatives.html"
     *         >Alternative transcriptions</a>.
     *         </p>
     */
    public Integer getMaxAlternatives() {
        return maxAlternatives;
    }

    /**
     * <p>
     * Indicate the maximum number of alternative transcriptions you want Amazon
     * Transcribe Medical to include in your transcript.
     * </p>
     * <p>
     * If you select a number greater than the number of alternative
     * transcriptions generated by Amazon Transcribe Medical, only the actual
     * number of alternative transcriptions are included.
     * </p>
     * <p>
     * If you include <code>MaxAlternatives</code> in your request, you must
     * also include <code>ShowAlternatives</code> with a value of
     * <code>true</code>.
     * </p>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/transcribe/latest/dg/how-alternatives.html"
     * >Alternative transcriptions</a>.
     * </p>
     * <p>
     * <b>Constraints:</b><br/>
     * <b>Range: </b>2 - 10<br/>
     *
     * @param maxAlternatives <p>
     *            Indicate the maximum number of alternative transcriptions you
     *            want Amazon Transcribe Medical to include in your transcript.
     *            </p>
     *            <p>
     *            If you select a number greater than the number of alternative
     *            transcriptions generated by Amazon Transcribe Medical, only
     *            the actual number of alternative transcriptions are included.
     *            </p>
     *            <p>
     *            If you include <code>MaxAlternatives</code> in your request,
     *            you must also include <code>ShowAlternatives</code> with a
     *            value of <code>true</code>.
     *            </p>
     *            <p>
     *            For more information, see <a href=
     *            "https://docs.aws.amazon.com/transcribe/latest/dg/how-alternatives.html"
     *            >Alternative transcriptions</a>.
     *            </p>
     */
    public void setMaxAlternatives(Integer maxAlternatives) {
        this.maxAlternatives = maxAlternatives;
    }

    /**
     * <p>
     * Indicate the maximum number of alternative transcriptions you want Amazon
     * Transcribe Medical to include in your transcript.
     * </p>
     * <p>
     * If you select a number greater than the number of alternative
     * transcriptions generated by Amazon Transcribe Medical, only the actual
     * number of alternative transcriptions are included.
     * </p>
     * <p>
     * If you include <code>MaxAlternatives</code> in your request, you must
     * also include <code>ShowAlternatives</code> with a value of
     * <code>true</code>.
     * </p>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/transcribe/latest/dg/how-alternatives.html"
     * >Alternative transcriptions</a>.
     * </p>
     * <p>
     * Returns a reference to this object so that method calls can be chained
     * together.
     * <p>
     * <b>Constraints:</b><br/>
     * <b>Range: </b>2 - 10<br/>
     *
     * @param maxAlternatives <p>
     *            Indicate the maximum number of alternative transcriptions you
     *            want Amazon Transcribe Medical to include in your transcript.
     *            </p>
     *            <p>
     *            If you select a number greater than the number of alternative
     *            transcriptions generated by Amazon Transcribe Medical, only
     *            the actual number of alternative transcriptions are included.
     *            </p>
     *            <p>
     *            If you include <code>MaxAlternatives</code> in your request,
     *            you must also include <code>ShowAlternatives</code> with a
     *            value of <code>true</code>.
     *            </p>
     *            <p>
     *            For more information, see <a href=
     *            "https://docs.aws.amazon.com/transcribe/latest/dg/how-alternatives.html"
     *            >Alternative transcriptions</a>.
     *            </p>
     * @return A reference to this updated object so that method calls can be
     *         chained together.
     */
    public MedicalTranscriptionSetting withMaxAlternatives(Integer maxAlternatives) {
        this.maxAlternatives = maxAlternatives;
        return this;
    }

    /**
     * <p>
     * The name of the custom vocabulary you want to use when processing your
     * medical transcription job. Vocabulary names are case sensitive.
     * </p>
     * <p>
     * The language of the specified vocabulary must match the language code you
     * specify in your transcription request. If the languages don't match, the
     * vocabulary isn't applied. There are no errors or warnings associated with
     * a language mismatch. US English (<code>en-US</code>) is the only valid
     * language for Amazon Transcribe Medical.
     * </p>
     * <p>
     * <b>Constraints:</b><br/>
     * <b>Length: </b>1 - 200<br/>
     * <b>Pattern: </b>^[0-9a-zA-Z._-]+<br/>
     *
     * @return <p>
     *         The name of the custom vocabulary you want to use when processing
     *         your medical transcription job. Vocabulary names are case
     *         sensitive.
     *         </p>
     *         <p>
     *         The language of the specified vocabulary must match the language
     *         code you specify in your transcription request. If the languages
     *         don't match, the vocabulary isn't applied. There are no errors or
     *         warnings associated with a language mismatch. US English (
     *         <code>en-US</code>) is the only valid language for Amazon
     *         Transcribe Medical.
     *         </p>
     */
    public String getVocabularyName() {
        return vocabularyName;
    }

    /**
     * <p>
     * The name of the custom vocabulary you want to use when processing your
     * medical transcription job. Vocabulary names are case sensitive.
     * </p>
     * <p>
     * The language of the specified vocabulary must match the language code you
     * specify in your transcription request. If the languages don't match, the
     * vocabulary isn't applied. There are no errors or warnings associated with
     * a language mismatch. US English (<code>en-US</code>) is the only valid
     * language for Amazon Transcribe Medical.
     * </p>
     * <p>
     * <b>Constraints:</b><br/>
     * <b>Length: </b>1 - 200<br/>
     * <b>Pattern: </b>^[0-9a-zA-Z._-]+<br/>
     *
     * @param vocabularyName <p>
     *            The name of the custom vocabulary you want to use when
     *            processing your medical transcription job. Vocabulary names
     *            are case sensitive.
     *            </p>
     *            <p>
     *            The language of the specified vocabulary must match the
     *            language code you specify in your transcription request. If
     *            the languages don't match, the vocabulary isn't applied. There
     *            are no errors or warnings associated with a language mismatch.
     *            US English (<code>en-US</code>) is the only valid language for
     *            Amazon Transcribe Medical.
     *            </p>
     */
    public void setVocabularyName(String vocabularyName) {
        this.vocabularyName = vocabularyName;
    }

    /**
     * <p>
     * The name of the custom vocabulary you want to use when processing your
     * medical transcription job. Vocabulary names are case sensitive.
     * </p>
     * <p>
     * The language of the specified vocabulary must match the language code you
     * specify in your transcription request. If the languages don't match, the
     * vocabulary isn't applied. There are no errors or warnings associated with
     * a language mismatch. US English (<code>en-US</code>) is the only valid
     * language for Amazon Transcribe Medical.
     * </p>
     * <p>
     * Returns a reference to this object so that method calls can be chained
     * together.
     * <p>
     * <b>Constraints:</b><br/>
     * <b>Length: </b>1 - 200<br/>
     * <b>Pattern: </b>^[0-9a-zA-Z._-]+<br/>
     *
     * @param vocabularyName <p>
     *            The name of the custom vocabulary you want to use when
     *            processing your medical transcription job. Vocabulary names
     *            are case sensitive.
     *            </p>
     *            <p>
     *            The language of the specified vocabulary must match the
     *            language code you specify in your transcription request. If
     *            the languages don't match, the vocabulary isn't applied. There
     *            are no errors or warnings associated with a language mismatch.
     *            US English (<code>en-US</code>) is the only valid language for
     *            Amazon Transcribe Medical.
     *            </p>
     * @return A reference to this updated object so that method calls can be
     *         chained together.
     */
    public MedicalTranscriptionSetting withVocabularyName(String vocabularyName) {
        this.vocabularyName = vocabularyName;
        return this;
    }

    /**
     * Returns a string representation of this object; useful for testing and
     * debugging.
     *
     * @return A string representation of this object.
     * @see java.lang.Object#toString()
     */
    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("{");
        if (getShowSpeakerLabels() != null)
            sb.append("ShowSpeakerLabels: " + getShowSpeakerLabels() + ",");
        if (getMaxSpeakerLabels() != null)
            sb.append("MaxSpeakerLabels: " + getMaxSpeakerLabels() + ",");
        if (getChannelIdentification() != null)
            sb.append("ChannelIdentification: " + getChannelIdentification() + ",");
        if (getShowAlternatives() != null)
            sb.append("ShowAlternatives: " + getShowAlternatives() + ",");
        if (getMaxAlternatives() != null)
            sb.append("MaxAlternatives: " + getMaxAlternatives() + ",");
        if (getVocabularyName() != null)
            sb.append("VocabularyName: " + getVocabularyName());
        sb.append("}");
        return sb.toString();
    }

    @Override
    public int hashCode() {
        final int prime = 31;
        int hashCode = 1;

        hashCode = prime * hashCode
                + ((getShowSpeakerLabels() == null) ? 0 : getShowSpeakerLabels().hashCode());
        hashCode = prime * hashCode
                + ((getMaxSpeakerLabels() == null) ? 0 : getMaxSpeakerLabels().hashCode());
        hashCode = prime
                * hashCode
                + ((getChannelIdentification() == null) ? 0 : getChannelIdentification().hashCode());
        hashCode = prime * hashCode
                + ((getShowAlternatives() == null) ? 0 : getShowAlternatives().hashCode());
        hashCode = prime * hashCode
                + ((getMaxAlternatives() == null) ? 0 : getMaxAlternatives().hashCode());
        hashCode = prime * hashCode
                + ((getVocabularyName() == null) ? 0 : getVocabularyName().hashCode());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj)
            return true;
        if (obj == null)
            return false;

        if (obj instanceof MedicalTranscriptionSetting == false)
            return false;
        MedicalTranscriptionSetting other = (MedicalTranscriptionSetting) obj;

        if (other.getShowSpeakerLabels() == null ^ this.getShowSpeakerLabels() == null)
            return false;
        if (other.getShowSpeakerLabels() != null
                && other.getShowSpeakerLabels().equals(this.getShowSpeakerLabels()) == false)
            return false;
        if (other.getMaxSpeakerLabels() == null ^ this.getMaxSpeakerLabels() == null)
            return false;
        if (other.getMaxSpeakerLabels() != null
                && other.getMaxSpeakerLabels().equals(this.getMaxSpeakerLabels()) == false)
            return false;
        if (other.getChannelIdentification() == null ^ this.getChannelIdentification() == null)
            return false;
        if (other.getChannelIdentification() != null
                && other.getChannelIdentification().equals(this.getChannelIdentification()) == false)
            return false;
        if (other.getShowAlternatives() == null ^ this.getShowAlternatives() == null)
            return false;
        if (other.getShowAlternatives() != null
                && other.getShowAlternatives().equals(this.getShowAlternatives()) == false)
            return false;
        if (other.getMaxAlternatives() == null ^ this.getMaxAlternatives() == null)
            return false;
        if (other.getMaxAlternatives() != null
                && other.getMaxAlternatives().equals(this.getMaxAlternatives()) == false)
            return false;
        if (other.getVocabularyName() == null ^ this.getVocabularyName() == null)
            return false;
        if (other.getVocabularyName() != null
                && other.getVocabularyName().equals(this.getVocabularyName()) == false)
            return false;
        return true;
    }
}
