/*
 * Decompiled with CFR 0.152.
 */
package com.amazonaws.athena.connector.lambda.data;

import com.amazonaws.athena.connector.lambda.data.TimeZoneKey;
import com.amazonaws.util.StringUtils;
import com.google.common.annotations.VisibleForTesting;
import java.math.BigDecimal;
import java.text.ParseException;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.time.format.DateTimeParseException;
import java.time.temporal.ChronoField;
import java.time.temporal.ChronoUnit;
import java.util.Date;
import java.util.Objects;
import java.util.concurrent.TimeUnit;
import org.apache.arrow.vector.holders.TimeStampMicroTZHolder;
import org.apache.arrow.vector.holders.TimeStampMilliTZHolder;
import org.apache.arrow.vector.types.pojo.ArrowType;
import org.apache.commons.lang3.time.DateFormatUtils;
import org.apache.commons.lang3.time.DateUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DateTimeFormatterUtil {
    private static final Logger logger = LoggerFactory.getLogger(DateTimeFormatterUtil.class);
    private static final String[] SUPPORTED_DATETIME_FORMAT = new String[]{DateFormatUtils.ISO_8601_EXTENDED_DATE_FORMAT.getPattern(), DateFormatUtils.ISO_8601_EXTENDED_DATETIME_FORMAT.getPattern(), DateFormatUtils.ISO_8601_EXTENDED_DATETIME_TIME_ZONE_FORMAT.getPattern(), "yyyyMMdd'T'HHmmss", "yyyyMMdd'T'HHmmssZZ"};
    private static final ZoneId UTC_ZONE_ID = ZoneId.of("UTC");
    private static final int TIME_ZONE_MASK = 4095;
    private static final int MILLIS_SHIFT = 12;
    private static boolean packTimezone = true;

    public static void disableTimezonePacking() {
        logger.info("Timezone packing disabled");
        packTimezone = false;
    }

    @VisibleForTesting
    static void enableTimezonePacking() {
        logger.info("Timezone packing enabled");
        packTimezone = true;
    }

    private DateTimeFormatterUtil() {
    }

    public static LocalDate stringToLocalDate(String value, String dateFormat, ZoneId defaultTimeZone) {
        if (StringUtils.isNullOrEmpty((String)dateFormat)) {
            logger.info("Unable to parse {} as Date type due to invalid dateformat", (Object)value);
            return null;
        }
        Date dateTransformedValue = DateTimeFormatterUtil.stringToDate(value, dateFormat);
        if (dateTransformedValue == null) {
            return null;
        }
        return dateTransformedValue.toInstant().atZone(defaultTimeZone).toLocalDate();
    }

    public static Object stringToZonedDateTime(String value, String dateFormat, ZoneId defaultTimeZone) {
        try {
            ZonedDateTime zdt = ZonedDateTime.parse(value);
            logger.info("Parsed {} to ZonedDateTime {}", (Object)value, (Object)zdt.toString());
            return zdt;
        }
        catch (DateTimeParseException e) {
            logger.warn("Unable to parse {} as ZonedDateTime type due to invalid date format, Falling back to LocalDateTime using timezone {}", (Object)value, (Object)defaultTimeZone);
            return DateTimeFormatterUtil.stringToDateTime(value, dateFormat, defaultTimeZone);
        }
    }

    public static Object stringToDateTime(String value, String dateFormat, ZoneId defaultTimeZone) {
        if (StringUtils.isNullOrEmpty((String)dateFormat)) {
            logger.warn("Unable to parse {} as DateTime type due to invalid date format", (Object)value);
            return null;
        }
        Date dateTransformedValue = DateTimeFormatterUtil.stringToDate(value, dateFormat);
        if (dateTransformedValue == null) {
            return null;
        }
        return dateTransformedValue.toInstant().atZone(defaultTimeZone).toLocalDateTime();
    }

    public static LocalDate bigDecimalToLocalDate(BigDecimal value, ZoneId defaultTimeZone) {
        Date dateTransformedValue = DateTimeFormatterUtil.bigDecimalToDate(value);
        if (dateTransformedValue == null) {
            return null;
        }
        return dateTransformedValue.toInstant().atZone(defaultTimeZone).toLocalDate();
    }

    public static LocalDateTime bigDecimalToLocalDateTime(BigDecimal value, ZoneId defaultTimeZone) {
        Date dateTransformedValue = DateTimeFormatterUtil.bigDecimalToDate(value);
        if (dateTransformedValue == null) {
            return null;
        }
        return dateTransformedValue.toInstant().atZone(defaultTimeZone).toLocalDateTime();
    }

    public static String inferDateTimeFormat(String value) {
        for (String datetimeFormat : SUPPORTED_DATETIME_FORMAT) {
            try {
                DateUtils.parseDate((String)value, (String[])new String[]{datetimeFormat});
                logger.info("Inferred format {} for value {}", (Object)datetimeFormat, (Object)value);
                return datetimeFormat;
            }
            catch (ParseException ex) {
            }
        }
        logger.warn("Failed to infer format for {}", (Object)value);
        return null;
    }

    private static Date stringToDate(String value, String dateFormat) {
        try {
            return DateUtils.parseDate((String)value, (String[])new String[]{dateFormat});
        }
        catch (IllegalArgumentException | ParseException ex) {
            logger.warn("Encountered value {} that is not consistent with inferred date/datetime format {} - will skip in result", (Object)value, (Object)dateFormat);
            return null;
        }
    }

    private static Date bigDecimalToDate(BigDecimal value) {
        try {
            return new Date(value.longValue());
        }
        catch (RuntimeException ex) {
            logger.warn("Invalid long value to transform to Date object");
            return null;
        }
    }

    private static long packDateTimeWithZone(ZonedDateTime zdt) {
        String zoneId = zdt.getZone().getId();
        LocalDateTime zdtInUtc = zdt.withZoneSameInstant(UTC_ZONE_ID).toLocalDateTime();
        return DateTimeFormatterUtil.packDateTimeWithZone(zdtInUtc.atZone(UTC_ZONE_ID).toInstant().toEpochMilli(), zoneId);
    }

    private static long packDateTimeWithZone(long millisUtc, String zoneId) {
        TimeZoneKey timeZoneKey = TimeZoneKey.getTimeZoneKey(zoneId);
        Objects.requireNonNull(timeZoneKey, "timeZoneKey is null");
        return DateTimeFormatterUtil.pack(millisUtc, timeZoneKey.getKey());
    }

    private static long pack(long millisUtc, short timeZoneKey) {
        return millisUtc << 12 | (long)(timeZoneKey & 0xFFF);
    }

    private static long unpackMillisUtc(long dateTimeWithTimeZone) {
        return dateTimeWithTimeZone >> 12;
    }

    private static TimeZoneKey unpackZoneKey(long dateTimeWithTimeZone) {
        return TimeZoneKey.getTimeZoneKey((short)(dateTimeWithTimeZone & 0xFFFL));
    }

    public static ZonedDateTime constructZonedDateTime(long epochTimestamp, ArrowType.Timestamp arrowType) {
        org.apache.arrow.vector.types.TimeUnit timeunit = arrowType.getUnit();
        ZoneId timeZone = ZoneId.of(arrowType.getTimezone());
        if (packTimezone) {
            if (!org.apache.arrow.vector.types.TimeUnit.MILLISECOND.equals((Object)timeunit)) {
                throw new UnsupportedOperationException("Unpacking is only supported for milliseconds");
            }
            TimeZoneKey timeZoneKey = DateTimeFormatterUtil.unpackZoneKey(epochTimestamp);
            epochTimestamp = DateTimeFormatterUtil.unpackMillisUtc(epochTimestamp);
            timeZone = ZoneId.of(timeZoneKey.getId());
        }
        return Instant.EPOCH.plus(epochTimestamp, DateTimeFormatterUtil.arrowTimeUnitToChronoUnit(timeunit)).atZone(timeZone);
    }

    public static ChronoUnit arrowTimeUnitToChronoUnit(org.apache.arrow.vector.types.TimeUnit timeunit) {
        switch (timeunit) {
            case MICROSECOND: {
                return ChronoUnit.MICROS;
            }
            case MILLISECOND: {
                return ChronoUnit.MILLIS;
            }
            case NANOSECOND: {
                return ChronoUnit.NANOS;
            }
            case SECOND: {
                return ChronoUnit.SECONDS;
            }
        }
        throw new UnsupportedOperationException(String.format("Unsupported timeunit: %s", timeunit));
    }

    public static ZonedDateTime zonedDateTimeFromObject(Object value) {
        if (value instanceof ZonedDateTime) {
            return (ZonedDateTime)value;
        }
        if (value instanceof LocalDateTime) {
            return ((LocalDateTime)value).atZone(ZoneId.of("UTC"));
        }
        if (value instanceof Date) {
            return ((Date)value).toInstant().atZone(ZoneId.of("UTC"));
        }
        throw new UnsupportedOperationException(String.format("Type: %s not supported", value.getClass()));
    }

    public static TimeStampMilliTZHolder timestampMilliTzHolderFromObject(Object value, String targetTimeZoneId) {
        ZonedDateTime zdt = DateTimeFormatterUtil.zonedDateTimeFromObject(value);
        TimeStampMilliTZHolder holder = new TimeStampMilliTZHolder();
        holder.timezone = zdt.getZone().getId();
        holder.value = zdt.toInstant().toEpochMilli();
        if (packTimezone) {
            holder.value = DateTimeFormatterUtil.packDateTimeWithZone(holder.value, holder.timezone);
        }
        if (targetTimeZoneId != null) {
            holder.timezone = targetTimeZoneId;
        }
        return holder;
    }

    public static TimeStampMicroTZHolder timestampMicroTzHolderFromObject(Object value, String targetTimeZoneId) {
        if (packTimezone) {
            throw new UnsupportedOperationException("Packing for TimeStampMicroTZ is not currently supported");
        }
        ZonedDateTime zdt = DateTimeFormatterUtil.zonedDateTimeFromObject(value);
        String zone = targetTimeZoneId != null ? targetTimeZoneId : zdt.getZone().getId();
        Instant instant = zdt.toInstant();
        long epochMicros = TimeUnit.SECONDS.toMicros(instant.getEpochSecond()) + instant.getLong(ChronoField.MICRO_OF_SECOND);
        TimeStampMicroTZHolder holder = new TimeStampMicroTZHolder();
        holder.timezone = zone;
        holder.value = epochMicros;
        return holder;
    }
}

