/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License").
 * You may not use this file except in compliance with the License.
 * A copy of the License is located at
 *
 *  http://aws.amazon.com/apache2.0
 *
 * or in the "license" file accompanying this file. This file is distributed
 * on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 * express or implied. See the License for the specific language governing
 * permissions and limitations under the License.
 */

package com.amazonaws.xray.emitters;

import com.amazonaws.xray.config.DaemonConfiguration;
import com.amazonaws.xray.entities.Entity;
import com.amazonaws.xray.entities.Segment;
import com.amazonaws.xray.entities.Subsegment;
import java.net.DatagramPacket;
import java.net.DatagramSocket;
import java.net.SocketException;
import java.nio.charset.StandardCharsets;
import java.util.List;
import java.util.Optional;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * @deprecated Use {@link Emitter#create()}.
 */
@Deprecated
public class UDPEmitter extends Emitter {
    private static final Log logger = LogFactory.getLog(UDPEmitter.class);
    private static final int UDP_PACKET_LIMIT = 63 * 1024;

    private DatagramSocket daemonSocket;
    private DaemonConfiguration config;
    private byte[] sendBuffer = new byte[DAEMON_BUF_RECEIVE_SIZE];

    /**
     * Constructs a UDPEmitter. Sets the daemon address to the value of the {@code AWS_XRAY_DAEMON_ADDRESS} environment variable
     * or {@code com.amazonaws.xray.emitters.daemonAddress} system property, if either are set to a non-empty value. Otherwise,
     * points to {@code InetAddress.getLoopbackAddress()} at port {@code 2000}.
     *
     * @throws SocketException
     *             if an error occurs while instantiating a {@code DatagramSocket}.
     */
    public UDPEmitter() throws SocketException {
        this(new DaemonConfiguration());
    }

    /**
     * Constructs a UDPEmitter. This overload allows you to specify the configuration.
     *
     * @param config The {@link DaemonConfiguration} for the Emitter.
     * @throws SocketException
     *             if an error occurs while instantiating a {@code DatagramSocket}.
     */
    public UDPEmitter(DaemonConfiguration config) throws SocketException {
        this.config = config;
        try {
            daemonSocket = new DatagramSocket();
        } catch (SocketException e) {
            logger.error("Exception while instantiating daemon socket.", e);
            throw e;
        }
    }

    public String getUDPAddress() {
        return config.getUDPAddress();
    }

    /**
     * {@inheritDoc}
     *
     * @see Emitter#sendSegment(Segment)
     */
    @Override
    public boolean sendSegment(Segment segment) {
        if (logger.isDebugEnabled()) {
            logger.debug(segment.prettySerialize());
        }
        
        byte[] bytes = (PROTOCOL_HEADER + PROTOCOL_DELIMITER + segment.serialize()).getBytes(StandardCharsets.UTF_8);

        if (bytes.length > UDP_PACKET_LIMIT) {
            List<Subsegment> subsegments = segment.getSubsegmentsCopy();
            logger.debug("Segment too large, sending subsegments to daemon first. bytes " + bytes.length + " subsegemnts "
                        + subsegments.size());
            for (Subsegment subsegment : subsegments) {
                sendSubsegment(subsegment);
                segment.removeSubsegment(subsegment);
            }
            bytes = (PROTOCOL_HEADER + PROTOCOL_DELIMITER + segment.serialize()).getBytes(StandardCharsets.UTF_8);
            logger.debug("New segment size. bytes " + bytes.length);
        }
        return sendData(bytes, segment);
    }

    /**
     * {@inheritDoc}
     *
     * @see Emitter#sendSubsegment(Subsegment)
     */
    @Override
    public boolean sendSubsegment(Subsegment subsegment) {
        if (logger.isDebugEnabled()) {
            logger.debug(subsegment.prettyStreamSerialize());
        }
        return sendData((PROTOCOL_HEADER + PROTOCOL_DELIMITER + subsegment.streamSerialize()).getBytes(StandardCharsets.UTF_8),
                        subsegment);
    }

    private boolean sendData(byte[] data, Entity entity) {
        try {
            DatagramPacket packet = new DatagramPacket(sendBuffer, DAEMON_BUF_RECEIVE_SIZE, config.getAddressForEmitter());
            packet.setData(data);
            logger.debug("Sending UDP packet.");
            daemonSocket.send(packet);
        } catch (Exception e) {
            String segmentName = Optional.ofNullable(entity.getParent()).map(this::nameAndId).orElse("[no parent segment]");
            logger.error("Exception while sending segment (" + entity.getClass().getSimpleName() + ") over UDP for entity "
                         + nameAndId(entity) + " on segment " + segmentName + ". Bytes: " + data.length, e);
            return false;
        }
        return true;
    }

    private String nameAndId(Entity entity) {
        return entity.getName() + " [" + entity.getId() + "]";
    }
}
