package com.appsflyer.security.plugin

import com.android.build.api.variant.AndroidComponentsExtension
import com.android.build.api.variant.ApplicationVariant
import com.android.build.api.variant.ComponentIdentity
import com.android.build.api.variant.Variant
import com.android.build.gradle.api.AndroidBasePlugin
import com.appsflyer.security.plugin.exc.AppsFlyerGradleException
import com.appsflyer.security.plugin.tasks.DownloadAppsFlyerSecuritySDK
import com.appsflyer.security.plugin.utils.APPLICATION_PLUGIN_NAME
import com.appsflyer.security.plugin.utils.APPSFLYER_PLUGIN_NAME
import com.appsflyer.security.plugin.utils.ASSEMBLE_TASK_PREFIX
import com.appsflyer.security.plugin.utils.BUNDLE_TASK_PREFIX
import com.appsflyer.security.plugin.utils.COLLECT_DEPENDENCIES_TASK_PREFIX
import com.appsflyer.security.plugin.utils.CONSUME_TASK_PREFIX
import com.appsflyer.security.plugin.utils.DATA_BINDING_TASK_PREFIX
import com.appsflyer.security.plugin.utils.DOWNLOAD_TASK_PREFIX
import com.appsflyer.security.plugin.utils.GRADLE_VERSION_ERR
import com.appsflyer.security.plugin.utils.IMPLEMENTATION_PREFIX
import com.appsflyer.security.plugin.utils.JWE_TOKEN_REGEX
import com.appsflyer.security.plugin.utils.MIN_GRADLE_VERSION
import com.appsflyer.security.plugin.utils.MISSING_APPLICATION_PLUGIN_ERR
import com.appsflyer.security.plugin.utils.MISSING_AUTH_TOKEN
import com.appsflyer.security.plugin.utils.MISSING_CERTIFICATE_HASHES
import com.appsflyer.security.plugin.utils.PRE_BUILD_TASK_PREFIX
import com.appsflyer.security.plugin.utils.PROCESS_MANIFEST_TASK_PREFIX
import com.appsflyer.security.plugin.utils.WRONG_AUTH_TOKEN_FORMAT
import com.appsflyer.security.plugin.utils.baseAarPath
import com.appsflyer.security.plugin.utils.capitalizeName
import com.appsflyer.security.plugin.utils.getAarFileName
import com.appsflyer.security.plugin.utils.versionName
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.SupervisorJob
import org.gradle.api.Plugin
import org.gradle.api.Project
import org.gradle.api.tasks.TaskContainer
import org.gradle.util.GradleVersion


/**
 * AppsFlyerSecurityPlugin is a class that applies the AppsFlyer security plugin to a Gradle Project.
 * The plugin creates and configures tasks to download and consume the AppsFlyer Security SDK during build.
 *
 * @property job SupervisorJob to control coroutine lifecycle.
 * @property scope CoroutineScope to execute coroutines in IO dispatcher and bound to job.
 */
class AppsFlyerSecurityPlugin : Plugin<Project> {
    private val job = SupervisorJob()
    private val scope = CoroutineScope(Dispatchers.IO + job)

    /**
     * Applies the plugin to the target project. It also verifies if the Android app plugin is applied
     * and the Gradle version is compatible. Then it sets up related tasks for the security SDK.
     *
     * @param project The Gradle project that the plugin should be applied to.
     */
    override fun apply(project: Project) = with(project) {
        val pluginExtension = extensions.create(
            APPSFLYER_PLUGIN_NAME,
            AppsFlyerSecurityPluginExtension::class.java
        )

        verifyAndroidApplicationPluginApplied()
        verifyGradleVersion()

        plugins.withType(AndroidBasePlugin::class.java).configureEach {
            val androidComponentsExtension =
                extensions.getByType(AndroidComponentsExtension::class.java)
            androidComponentsExtension.onVariants { variant ->
                // register tasks only for flavors that are not excluded by configuration
                if (!pluginExtension.shouldSkipFlavor(variant)) {
                    val flavorConfig = pluginExtension.getFlavorConfig(variant)
                    flavorConfig.validateConfiguration()
                    with(TaskNames(variant, flavorConfig)) {

                        val downloadTask = registerDownloadTask(
                            tasks,
                            variant as ApplicationVariant,
                            flavorConfig,
                            project,
                            pluginExtension.timeoutConfiguration
                        )
                        addAarDependency(project, downloadTask, implementation)
                        afterEvaluate(project, downloadTask)
                    }
                }
            }
        }
    }

    /**
     * Adds an AAR dependency to the given project.
     *
     * Due to Gradle's lifecycle model, all dependencies of a project should be known and available
     * during the configuration phase, before execution starts. For an AAR file that's downloaded
     * during the execution phase, this poses a problem since the file isn't available yet during
     * the configuration phase.
     *
     * To work around this, if the AAR file downloaded by [downloadTask] does not exist
     * (meaning the execution phase where it would have been downloaded has not yet occurred),
     * an empty file is written to the intended output path to serve as placeholder. Though empty, this file
     * is enough to satisfy Gradle's requirement of the file existing during configuration.
     *
     * When the execution phase begins (and [downloadTask] runs), the real AAR file is downloaded
     * and replaces the placeholder file.
     *
     * The AAR file (whether downloaded or placeholder) is then added as a file dependency
     * to the configuration specified by [implementation].
     *
     * @param project The Gradle project to which the dependency is to be added.
     * @param downloadTask The task downloading the AAR file. Its output file is added as a dependency.
     * @param implementation The configuration to which the AAR file dependency will be added.
     */
    private fun addAarDependency(
        project: Project,
        downloadTask: DownloadAppsFlyerSecuritySDK,
        implementation: String,
    ) {
        if (!downloadTask.outputFile.exists()) {
            downloadTask.outputFile.writeBytes(byteArrayOf())
        }
        // set dependency to the downloaded file
        val addedDependency = project.dependencies.add(
            implementation,
            project.files(downloadTask.outputFile.path)
        )
        if (addedDependency == null) {
            throw AppsFlyerGradleException(
                "Failed to add AAR dependency %s files('%s')".format(
                    implementation,
                    downloadTask.outputFile.path
                )
            )
        }
    }

    /**
     * Gets configuration options for a specific variant. It checks for a flavor-specific configuration first,
     * then defaults to the global configuration if none is found.
     *
     * @param variant The variant that we need the plugin configuration for.
     * @return The [FlavorConfig] instance that holds the configuration.
     */
    private fun AppsFlyerSecurityPluginExtension.getFlavorConfig(variant: Variant): FlavorConfig {
        val options = variant.filtersOptions
        return flavorConfigs.firstOrNull {
            it.name in options
        } ?: defaultConfig
    }

    /**
     * Determines if [AppsFlyerSecurityPluginExtension] should skip the current flavor variant.
     *
     * @param variant the build variant
     * @return true if the flavor variant should be skipped, false otherwise
     */
    private fun AppsFlyerSecurityPluginExtension.shouldSkipFlavor(variant: Variant): Boolean {
        val options = variant.filtersOptions
        return ignoreFlavors?.any { it in options } ?: false
    }


    /**
     * Extension property for [Variant] to get a set of filter options.
     *
     * Represents a set of parameters from a variant that are commonly used to filter out or identify a specific variant.
     * @return a set of strings representing a variant's name, flavor name, and build type.
     */
    private val Variant.filtersOptions: Set<String?>
        get() = setOf(name, flavorName, buildType)

    // region Extensions of TaskNames for creating specific tasks.

    /**
     * Extension function to create and register a DownloadAppsFlyerSecuritySDK task.
     * The function first checks if a task with the same name already exists
     * and returns that if true. Else it creates, configures and registers a new task.
     *
     * @param tasks The gradle TaskContainer where this task is registered.
     * @param variant Reference to the current variant to extract info like applicationId, versionName, etc
     * @param flavorConfig Reference to the current flavor's FlavorConfig instance
     * @return Returns the created (or found) DownloadAppsFlyerSecuritySDK task.
     */
    private fun TaskNames.registerDownloadTask(
        tasks: TaskContainer,
        variant: ApplicationVariant,
        flavorConfig: FlavorConfig,
        project: Project,
        timeoutConfiguration: TimeoutConfiguration
    ): DownloadAppsFlyerSecuritySDK =
        tasks.findByName(downloadTaskName) as? DownloadAppsFlyerSecuritySDK
            ?: tasks.register(
                downloadTaskName,
                DownloadAppsFlyerSecuritySDK::class.java,
            ) {
                with(variant) {
                    it.applicationId.set(getAppsFlyerAppId(flavorConfig))
                    it.authToken.set(flavorConfig.authToken)
                    it.versionName.set(versionName)
                    it.baseAarPath.set(baseAarPath)
                    flavorConfig.certificateHashes?.let { cer ->
                        it.certificateHashes.set(cer)

                    } ?: throw AppsFlyerGradleException(
                        MISSING_CERTIFICATE_HASHES
                    )
                    it.certificateHashes.set(flavorConfig.certificateHashes)
                    it.aarFileName.set(getAarFileName(flavorConfig))
                    it.projectDir.set(project.projectDir)
                    it.timeoutConfiguration.set(timeoutConfiguration)
                }
            }.get()


    /**
     * Returns the AppsFlyer App ID for the given ApplicationVariant and FlavorConfig.
     *
     * This function retrieves the AppsFlyer App ID based on the variant's specific configuration.
     * If the `applicationId` is specified in the `flavorConfig` object, it will be used as the App ID.
     * Otherwise, it will concatenate the `applicationId` obtained from the variant with the `channel`
     * from the `flavorConfig` to form the AppsFlyer App ID.
     *
     * @param flavorConfig The configuration object for the flavor associated with the variant.
     * @return The AppsFlyer App ID for the variant.
     */
    private fun ApplicationVariant.getAppsFlyerAppId(flavorConfig: FlavorConfig): String =
        flavorConfig.applicationId ?: "${applicationId.get()}${flavorConfig.channel}"

    /**
     * Set up dependencies of relevant tasks to depend on downloadTask and consumeTask.
     * These dependencies are set after the project is evaluated.
     *
     * @param project Base project of the plugin
     * @param downloadTask The download task which other tasks may depend upon.
     */
    private fun TaskNames.afterEvaluate(
        project: Project,
        downloadTask: DownloadAppsFlyerSecuritySDK,
    ) {
        project.afterEvaluate { pro ->
            listOf(
                assembleTaskName,
                bundleTaskName,
                processManifest,
                collectDependencies,
                preBuild
            ).forEach {
                pro.tasks.findByName(it)?.dependsOn(downloadTask)
            }
        }
    }

    //  end region
    // region Extensions for project verification.
    /**
     * Verifies that the 'com.android.application' plugin is applied to the project.
     * If the plugin isn't applied, this method throws a AppsFlyerGradleException with an appropriate error message.
     *
     * @throws AppsFlyerGradleException If 'com.android.application' plugin hasn't been applied.
     */
    private fun Project.verifyAndroidApplicationPluginApplied() {
        // Check if 'com.android.application' plugin is applied
        if (!plugins.hasPlugin(APPLICATION_PLUGIN_NAME)) {
            // 'com.android.application' plugin is not applied.
            throw AppsFlyerGradleException(MISSING_APPLICATION_PLUGIN_ERR)
        }
    }

    /**
     * Verifies that the current Gradle version matches or exceeds a certain minimum version.
     * If the current Gradle version is less than the minimum, this method throws a AppsFlyerGradleException
     * with an appropriate error message.
     *
     * @throws AppsFlyerGradleException If the current Gradle version is less than the specified minimum.
     */
    private fun verifyGradleVersion() {
        if (GradleVersion.current() < GradleVersion.version(MIN_GRADLE_VERSION)) {
            throw AppsFlyerGradleException(GRADLE_VERSION_ERR)
        }
    }
}

private fun FlavorConfig.validateConfiguration() {
    val jweRegex = JWE_TOKEN_REGEX.toRegex()
    if (authToken.isBlank()) {
        throw AppsFlyerGradleException(MISSING_AUTH_TOKEN)
    }
    if (!jweRegex.matches(authToken)) {
        throw AppsFlyerGradleException(WRONG_AUTH_TOKEN_FORMAT)
    }
    if (certificateHashes.isNullOrEmpty()) {
        throw AppsFlyerGradleException(MISSING_CERTIFICATE_HASHES)
    }
}

//  end region


/**
 * This class generates names for tasks based on the build variant and flavor configuration.
 *
 * @constructor initialize the TaskNames object and sets the relevant task names.
 * @param variant The build variant for which tasks should be named.
 * @param flavorConfig The flavor configuration used in naming.
 */
private class TaskNames(variant: ComponentIdentity, flavorConfig: FlavorConfig) {
    val downloadTaskName: String
    val consumeTaskName: String
    val assembleTaskName: String
    val bundleTaskName: String
    val processManifest: String
    val dataBinding: String
    val preBuild: String
    val collectDependencies: String
    val implementation: String

    init {
        with(variant as ApplicationVariant) {
            downloadTaskName =
                DOWNLOAD_TASK_PREFIX.format("${applicationId.get()}${flavorConfig.channel}")
            implementation = IMPLEMENTATION_PREFIX.format(name)
        }

        with(variant.capitalizeName) {
            consumeTaskName = CONSUME_TASK_PREFIX.format(this)
            assembleTaskName = ASSEMBLE_TASK_PREFIX.format(this)
            bundleTaskName = BUNDLE_TASK_PREFIX.format(this)
            processManifest = PROCESS_MANIFEST_TASK_PREFIX.format(this)
            collectDependencies = COLLECT_DEPENDENCIES_TASK_PREFIX.format(this)
            dataBinding = DATA_BINDING_TASK_PREFIX.format(this)
            preBuild = PRE_BUILD_TASK_PREFIX.format(this)
        }
    }
}
