package com.appsflyer.security.plugin

import org.gradle.api.Action
import org.gradle.api.Project
import org.gradle.api.NamedDomainObjectContainer

/**
 * This open class named "AppsFlyerSecurityPluginExtension" allows for the configuration of the AppsFlyer Security Plugin.
 * Configurations can be specified on a global default level or on a flavor-specific level.
 *
 * @property defaultConfig Default configuration that applies to all flavors, unless overridden in flavor-specific configurations.
 * @property flavorConfigs Container for flavor-specific configurations.
 * @property ignoreFlavors set of flavors that the plugin should ignore
 * @property timeoutConfiguration An instance of TimeoutConfiguration class providing timeout settings for the plugin, such as HTTP request timeout, connection timeout and socket timeout.
 * @constructor Initializes the AppsFlyerSecurityPluginExtension with a reference to the [Project] object.
 * @param project The Gradle project instance.
 */
open class AppsFlyerSecurityPluginExtension(project: Project) {
    var defaultConfig = FlavorConfig("default")
    val flavorConfigs: NamedDomainObjectContainer<FlavorConfig> =
        project.container(FlavorConfig::class.java)
    var ignoreFlavors: Set<String>? = null
    var timeoutConfiguration =  TimeoutConfiguration()

    /**
     * Define the default configuration. Any properties specified here will apply to all flavors
     * unless overridden in a specific flavor configuration.
     * @param configure A configuration action that sets up the `FlavorConfig` object.
     */
    fun defaultConfig(configure: Action<FlavorConfig>) {
        configure.execute(defaultConfig)
    }

    /**
     * Define specific configurations for different flavors. Properties specified here override the
     * values provided in the 'defaultConfig'.
     * @param action A configuration action that sets up the flavor-specific configurations.
     */
    fun flavorConfigs(action: Action<in NamedDomainObjectContainer<FlavorConfig>>) {
        action.execute(flavorConfigs)
    }

    /**
     * Updates the TimeoutConfiguration settings through a provided action.
     *
     * @param action An Action that will be applied to the current TimeoutConfiguration.
     * This enables changing timeout settings in a lambda expression.
     */
    fun timeoutConfiguration(action: Action<TimeoutConfiguration>){
        action.execute(timeoutConfiguration)
    }
}
