package com.appsflyer.security.plugin

open class FlavorConfig(val name: String) {
    private companion object {
        private val IGNORABLE_CHANNELS = listOf(
            "googleplay",
            "playstore",
            "googleplaystore"
        )
    }

    /**
     * This field specifies a list of the application signing keys that will be used to validate the Application Integrity.
     *
     * AppsFlyer cannot automatically determine which signing key will be used to sign your application.
     * It is therefore required that you supply the certificate hash for your singing key manually.
     * The certificate hash can only be provided without the colons.
     * We only support the SHA-256 hash.
     *
     * **Attention**
     *
     * When using app signing by [Google Play](https://developer.android.com/studio/publish/app-signing#google-play-app-signing),
     * Google manages and protects your app's signing key for you and signs your APKs on your behalf.
     * In this case it is required that you provide the certificate hash for the signing key **used by Google** using this option.
     * This is **always** the case when you distribute Android app bundles.
     */
     var certificateHashes: List<String>? = null

    /**
     * Authentication
     * An API V2 token (bearer token) is required to use the plugin.
     * The admin gets the V2 token in the [dashboard](https://support.appsflyer.com/hc/en-us/articles/360004562377)
     *
     */
    var authToken: String = ""

    /**
     * When adding the App in the AppsFlyer dashboard using the Android Out-of-Store APK option,
     * set the [channel]. The channel identifies the store.
     * The combination of Android package name and channel uniquely identifies each AppsFlyer dashboard.
     *
     * This option is relevant only when using the dashboard per store option.
     *
     * Example: The Android package name is `com.abc.def` and the channel is `xyzdevicestore`.
     * The unique dashboard identifier is `com.abc.def-xyzdevicestore`.
     *
     * For more info, please consult the official [AppsFlyer support documentation](https://support.appsflyer.com/hc/en-us/articles/207447023#single-dashboard-vs-dashboard-per-store).
     */
    var channel: String = ""
        get() = if (field.isBlank() || field in IGNORABLE_CHANNELS ) "" else "-$field"


    /**
     * Represents the application ID specific to the flavor configuration.
     *
     * The `applicationId` property holds the unique identifier for the application associated
     * with this flavor configuration. It can be used to override the default behavior of deriving
     * the AppsFlyer App ID from the variant's package name and the `channel`.
     * If the `applicationId` is specified, it will be used as the AppsFlyer App ID for the
     * corresponding variant. If not specified, the default behavior will be followed, which involves
     * concatenating the variant's package name with the `channel` to form the AppsFlyer App ID.
     *
     * The `channel` is used to identify the store when adding the application in the AppsFlyer
     * dashboard using the Android Out-of-Store APK option. The combination of the Android package
     * name and `channel` uniquely identifies each AppsFlyer dashboard.
     *
     * For more control over the AppsFlyer App ID, you can specify the desired `applicationId`,
     * overriding the default concatenation behavior. If the `applicationId` is not specified or
     * is blank, it will be treated as empty, and the AppsFlyer App ID will be derived from the variant's
     * package name and the `channel`.
     *
     * Example: Given the Android package name as `com.abc.def` and the channel as `xyzdevicestore`,
     * the default AppsFlyer App ID would be `com.abc.def-xyzdevicestore`. However, if you provide
     * a specific `applicationId` such as `myapp`, the AppsFlyer App ID will be `myapp`.
     *
     */
    var applicationId: String? = null
}