package com.appsflyer.security.plugin.network

import com.appsflyer.security.plugin.exc.AppsFlyerGradleException
import com.appsflyer.security.plugin.TimeoutConfiguration
import com.appsflyer.security.plugin.utils.APPSFLYER_HQ_BASE_ENDPOINT
import io.ktor.client.HttpClient
import io.ktor.client.engine.okhttp.OkHttp
import io.ktor.client.plugins.DefaultRequest
import io.ktor.client.plugins.HttpResponseValidator
import io.ktor.client.plugins.HttpTimeout
import io.ktor.client.plugins.ResponseException
import io.ktor.client.plugins.auth.Auth
import io.ktor.client.plugins.auth.providers.BearerTokens
import io.ktor.client.plugins.auth.providers.bearer
import io.ktor.client.plugins.contentnegotiation.ContentNegotiation
import io.ktor.client.plugins.defaultRequest
import io.ktor.client.plugins.resources.Resources
import io.ktor.client.request.accept
import io.ktor.client.request.header
import io.ktor.http.ContentType
import io.ktor.http.HttpHeaders
import io.ktor.serialization.kotlinx.json.json
import kotlinx.serialization.json.Json

/**
 * A typealias for a print function that takes a string as parameter and returns Unit
 *
 * @property LogPrint represents a function that prints logs
 */
typealias LogPrint = (String) -> Unit

/**
 * Creates an HTTP client with predefined configurations for base URL, redirection, success expectation,
 * installed features like resources, content negotiation, authorization, response validation,
 * default request settings, timeout settings, and default request with some header details.
 *
 * @param bearerToken The Bearer token to be used for authorization
 * @param logPrint Function reference for logging
 * @return Preconfigured HttpClient
 */
fun createHttpClient(
    bearerToken: String,
    timeoutConfiguration: TimeoutConfiguration,
    followRedirects: Boolean = false,
    logPrint: LogPrint
) = HttpClient(OkHttp) {
    this.followRedirects = followRedirects
    expectSuccess = true
    install(Resources)
    install(ContentNegotiation) {
        json(Json {
            ignoreUnknownKeys = true
            encodeDefaults = true
        })
    }

    install(Auth) {
        bearer {
            // Configure bearer authentication
            loadTokens {
                // Load tokens from a local storage and return them as the 'BearerTokens' instance
                BearerTokens(bearerToken, bearerToken)
            }
        }
    }

    HttpResponseValidator {
        handleResponseExceptionWithRequest { exception, _ ->
            when (exception) {
                is ResponseException -> exception.logAndThrow(logPrint)
                else -> exception.logAndThrowUnexpected(logPrint)
            }
        }
    }

    defaultRequest {
        url(APPSFLYER_HQ_BASE_ENDPOINT)
    }

    install(HttpTimeout) {
        timeoutConfiguration.requestTimeoutMillis?.let {
            requestTimeoutMillis = it
        }
        timeoutConfiguration.connectTimeoutMillis?.let {
            connectTimeoutMillis = it
        }
        timeoutConfiguration.socketTimeoutMillis?.let {
            socketTimeoutMillis = it
        }
    }

    install(DefaultRequest) {
        header(HttpHeaders.ContentType, ContentType.Application.Json)
        accept(ContentType.Application.Json)
    }
}


/**
 * Method to log a [ResponseException] and throw it as [AppsFlyerGradleException].
 *
 * @param logPrint Function reference for logging
 */
fun ResponseException.logAndThrow(logPrint: LogPrint) {
    val msg = "Failed to download Resources.\n\t* Reason: %s".format(
        this.message,
    )
    logPrint(msg)
    throw AppsFlyerGradleException(msg, this)
}

/**
 * Method to handle, log and throw unexpected exceptions as [AppsFlyerGradleException].
 *
 * @param logPrint Function reference for logging
 */
fun Throwable.logAndThrowUnexpected(logPrint: LogPrint) {
    val msg = "Unexpected Exception - Failed to download Resources.\n\t* Reason: %s".format(
        this.message,
    )
    logPrint(msg)
    throw AppsFlyerGradleException(msg, this)
}