package com.appsflyer.security.plugin.network

import com.appsflyer.security.plugin.TimeoutConfiguration
import com.appsflyer.security.plugin.exc.AppsFlyerGradleException
import com.appsflyer.security.plugin.network.models.requests.GenerateAarBody
import com.appsflyer.security.plugin.network.models.responses.GenerateAarResponse
import com.appsflyer.security.plugin.network.models.responses.DownloadAarResponse
import com.appsflyer.security.plugin.network.models.responses.CurrentBuildResponse
import com.appsflyer.security.plugin.network.models.responses.Status
import com.appsflyer.security.plugin.network.resources.AppId
import com.appsflyer.security.plugin.utils.BODY_FAILED_DECODING
import com.appsflyer.security.plugin.utils.BUILDING_IN_PROGRESS_LOG
import com.appsflyer.security.plugin.utils.DELAY_MILLIS
import com.appsflyer.security.plugin.utils.MAX_RETRIES
import com.appsflyer.security.plugin.utils.MAX_RETRIES_EXCEEDED_ERR
import io.ktor.client.HttpClient
import io.ktor.client.call.body
import io.ktor.client.plugins.resources.get
import io.ktor.client.plugins.resources.post
import io.ktor.client.request.setBody
import io.ktor.client.statement.HttpResponse
import io.ktor.client.statement.bodyAsChannel
import io.ktor.client.statement.bodyAsText
import io.ktor.http.HttpStatusCode
import io.ktor.util.cio.writeChannel
import io.ktor.utils.io.charsets.MalformedInputException
import io.ktor.utils.io.copyAndClose
import kotlinx.coroutines.delay
import org.gradle.api.logging.Logger
import java.io.Closeable
import java.io.File

/**
 * NetworkRepository class provides the functionalities to handle network requests.
 *
 * @property bearerToken used for HTTP authentication
 * @property logger to handle logs
 */
class NetworkRepository(bearerToken: String, private val logger: Logger, timeoutConfiguration: TimeoutConfiguration) : Closeable {
    private val ktorClient: HttpClient = createHttpClient(bearerToken, timeoutConfiguration) {
        logger.lifecycle(it)
    }
    private val ktorRedirectsClient: HttpClient =
        createHttpClient(bearerToken, timeoutConfiguration, followRedirects = true) {
            logger.lifecycle(it)
        }

    /**
     * Generates an .aar (Android Archive) file.
     *
     * @param appId The app id
     * @param appVersion The app version
     * @param sdkVersion The sdk version
     * @param body The body for the request to generate the .aar file
     * @return GenerateAarResponse
     */
    suspend fun generateAar(
        appId: String,
        appVersion: String,
        sdkVersion: String,
        body: GenerateAarBody
    ): GenerateAarResponse =
        ktorClient.post(AppId.GenerateAar(appId, appVersion, sdkVersion)) {
            setBody(body)
        }.also {
            if (it.status != HttpStatusCode.OK) {
                val exceptionResponseText = try {
                    it.bodyAsText()
                } catch (_: MalformedInputException) {
                    BODY_FAILED_DECODING
                }
                throw AppsFlyerGradleException(
                    "Unexpected Status Code",
                    UnexpectedStatusCodeException(it.status, exceptionResponseText, logger)
                )
            }
        }.body()

    /**
     * Gets the current build status
     *
     * @param appId The app id
     * @param appVersion The app version
     * @param sdkVersion The sdk version
     * @return HttpResponse
     */
    suspend fun getCurrentBuildStatus(
        appId: String,
        appVersion: String,
        sdkVersion: String
    ): HttpResponse = ktorClient.get(AppId.CurrentBuildStatus(appId, appVersion, sdkVersion)).body()


    /**
     * Continuously checks the status of a request until it is no longer in progress (i.e., not returning a 202 status).
     * Makes a repeated attempt to retrieve the status of a particular HTTP request identified by app id, app version, and sdk version.
     *
     * It repeats the process [MAX_RETRIES] number of times. On each attempt, if the response status is [HttpStatusCode.Accepted] (202), delay for [DELAY_MILLIS] milliseconds before making the next attempt.
     * If the response status is [HttpStatusCode.OK] (200), returns the response. Else, logs the unsuccessful status and throws a AppsFlyerGradleException.
     *
     * @param appId The app id
     * @param appVersion The app version
     * @param sdkVersion The sdk version
     * @throws AppsFlyerGradleException on non-successful HTTP calls (non-200 and non-202 status) or if the number of retries exceeds [MAX_RETRIES]
     * @return GetStatusResponse
     */
    suspend fun getStatusUntilNotInProgress(
        appId: String,
        appVersion: String,
        sdkVersion: String
    ): CurrentBuildResponse {
        repeat(MAX_RETRIES) {
            val response: HttpResponse = getCurrentBuildStatus(appId, appVersion, sdkVersion)
            when (response.status) {
                HttpStatusCode.Accepted -> {
                    val status = response.body<CurrentBuildResponse>().status
                    logger.lifecycle(BUILDING_IN_PROGRESS_LOG.format(it + 1, MAX_RETRIES, status))
                    delay(DELAY_MILLIS) // If status is IN_PROGRESS, delay for 10 seconds before next attempt
                }

                HttpStatusCode.OK -> return response.body()

                else -> {
                    val exceptionResponseText = try {
                        response.bodyAsText()
                    } catch (_: MalformedInputException) {
                        BODY_FAILED_DECODING
                    }
                    throw AppsFlyerGradleException(
                        "Unexpected Status Code",
                        UnexpectedStatusCodeException(
                            response.status,
                            exceptionResponseText,
                            logger
                        )
                    )
                }
            }
        }

        logger.lifecycle(MAX_RETRIES_EXCEEDED_ERR)
        return CurrentBuildResponse(
            status = Status.MAX_RETRIES_EXCEEDED,
            error = MAX_RETRIES_EXCEEDED_ERR
        )
    }

    /**
     * Downloads an .aar (Android Archive) file.
     *
     * @param appId The app id
     * @param appVersion The app version
     * @param sdkVersion The sdk version
     * @param outputFile The output file
     * @throws AppsFlyerGradleException on non-successful (non-200 status) HTTP calls and NoContent status (204 status).
     * In case of NoContent status, it implies that there is no file content to download.
     * @return GetAarResponse that contains information about the existence of the file. Asserts the existence of the requested .aar file.
     */
    suspend fun downloadAar(
        appId: String,
        appVersion: String,
        sdkVersion: String,
        outputFile: File
    ): DownloadAarResponse =
        with(ktorRedirectsClient.get(AppId.DownloadAar(appId, appVersion, sdkVersion))) {
            when (status) {
                HttpStatusCode.OK -> {
                    bodyAsChannel().copyAndClose(outputFile.writeChannel())
                    DownloadAarResponse(isFileExist = true)
                }

                HttpStatusCode.NoContent -> {
                    DownloadAarResponse(isFileExist = false)
                }

                else -> {
                    val exceptionResponseText = try {
                        bodyAsText()
                    } catch (_: MalformedInputException) {
                        BODY_FAILED_DECODING
                    }
                    throw AppsFlyerGradleException(
                        "Unexpected Status Code",
                        UnexpectedStatusCodeException(
                            status,
                            exceptionResponseText,
                            logger
                        )
                    )
                }
            }
        }

    /**
     * Closes the HTTP client
     */
    override fun close() {
        ktorClient.close()
        ktorRedirectsClient.close()
    }

    private class UnexpectedStatusCodeException(
        statusCode: HttpStatusCode,
        text: String,
        logger: Logger
    ) : IllegalStateException("Unexpected Status Code: $statusCode. Text: $text") {
        init {
            logger.lifecycle("Unexpected Status Code: $statusCode. Text: $text")
        }
    }
}