package com.appsflyer.security.plugin.tasks


import com.appsflyer.security.plugin.TimeoutConfiguration
import com.appsflyer.security.plugin.exc.AppsFlyerGradleException
import com.appsflyer.security.plugin.tasks.impl.Downloader
import com.appsflyer.security.plugin.utils.APPSFLYER_BETA_DEPENDENCY
import com.appsflyer.security.plugin.utils.APPSFLYER_GROUP
import com.appsflyer.security.plugin.utils.APPSFLYER_PROD_DEPENDENCY
import com.appsflyer.security.plugin.utils.CHECKING_DEPENDENCY_LOG
import com.appsflyer.security.plugin.utils.FOUND_VERSION_LOG
import com.appsflyer.security.plugin.utils.MISSING_CERTIFICATE_HASHES
import com.appsflyer.security.plugin.utils.SDK_ALREADY_AVAILABLE_LOG
import com.appsflyer.security.plugin.utils.existsAndValid
import com.appsflyer.security.plugin.utils.moveTo
import org.gradle.api.DefaultTask
import org.gradle.api.provider.ListProperty
import org.gradle.api.provider.Property
import org.gradle.api.tasks.Input
import org.gradle.api.tasks.OutputFile
import org.gradle.api.tasks.TaskAction
import java.io.File


/**
 * An abstract class that extends `DefaultTask` of gradle to download the AppsFlyer security SDK.
 *
 * @property authToken Authorization token required for HTTP requests.
 * @property baseAarPath Path for saving the .aar (Android Archive) file in the project directory.
 * @property aarFileName Name of the .aar (Android Archive) file.
 * @property applicationId An id that uniquely identifies your application.
 * @property versionName Version name of your application.
 * @property certificateHashes List of certificate hashes.
 */
abstract class DownloadAppsFlyerSecuritySDK : DefaultTask() {

    @get:Input
    abstract val authToken: Property<String>

    @get:Input
    abstract val baseAarPath: Property<String>

    @get:Input
    abstract val aarFileName: Property<String>

    @get:Input
    abstract val applicationId: Property<String>

    @get:Input
    abstract val versionName: Property<String>

    @get:Input
    abstract val certificateHashes: ListProperty<String>

    @get:Input
    abstract val projectDir: Property<File>

    @get:Input
    val appsFlyerSdkVersion: Property<String> =
        project.objects.property(String::class.java).also { prop ->
            prop.set(getAppsFlyerSdkVersion())
        }
    @get:Input
    abstract val timeoutConfiguration: Property<TimeoutConfiguration>

    private val libDir: File
        get() = File(projectDir.get(), baseAarPath.get()).apply { if (!exists()) mkdirs() }

    @get:OutputFile
    val outputFile: File
        get() = File(libDir, aarFileName.get())

    /**
     * Initiates the security SDK download
     *
     * This task action begins the download process. If the output file already exists, it logs a message and returns,
     * avoiding needless repeat downloads. Otherwise, it creates an instance of the Downloader with the necessary parameters
     * and commences the download process.
     *
     * @throws AppsFlyerGradleException if the process encounters any errors or if list of certificateHashes is null.
     */
    @TaskAction
    fun download() {
        if (outputFile.existsAndValid) {
            logger.lifecycle(SDK_ALREADY_AVAILABLE_LOG)
            return
        }
        val certificates: List<String> =
            certificateHashes.orNull ?: throw AppsFlyerGradleException(MISSING_CERTIFICATE_HASHES)

        Downloader(
            authToken.get(),
            certificates,
            logger,
            applicationId.get(),
            versionName.get(),
            appsFlyerSdkVersion.get(),
            timeoutConfiguration.get()
        ).downloadSecuritySdk()
            .moveTo(outputFile, overwrite = true)

        if (!outputFile.existsAndValid) {
            throw RuntimeException("Required AAR file dependency is not exist or empty: " + outputFile.absolutePath)
        }
    }

    /**
     * Retrieves the AppsFlyer SDK version.
     *
     * Involves checking all dependencies of the project for the presence of AppsFlyer dependencies (Production or Beta).
     * If an AppsFlyer dependency is located, it later informs about the version of the found dependency.
     *
     * @return String representing the AppsFlyer SDK version.
     */
    private fun getAppsFlyerSdkVersion(): String {
        var version: String? = null
        project.configurations.all {
            val tmpVersion = it.allDependencies.firstOrNull { dependency ->
                logger.debug(CHECKING_DEPENDENCY_LOG.format(dependency.toString()))
                (dependency.group == APPSFLYER_GROUP && dependency.name in listOf(
                    APPSFLYER_BETA_DEPENDENCY,
                    APPSFLYER_PROD_DEPENDENCY
                ))
            }?.version
            if (!tmpVersion.isNullOrBlank()) {
                version = tmpVersion
            }
        }
        logger.debug(FOUND_VERSION_LOG.format(version))
        return version.orEmpty()
    }
}
