package com.appsflyer.security.plugin.tasks.impl

import com.appsflyer.security.plugin.exc.AppsFlyerGradleException
import com.appsflyer.security.plugin.TimeoutConfiguration
import com.appsflyer.security.plugin.network.NetworkRepository
import com.appsflyer.security.plugin.network.models.requests.GenerateAarBody
import com.appsflyer.security.plugin.network.models.responses.DownloadAarResponse
import com.appsflyer.security.plugin.network.models.responses.Status
import com.appsflyer.security.plugin.utils.AAR_FILE_EXT
import com.appsflyer.security.plugin.utils.GET_STATUS_ERR
import com.appsflyer.security.plugin.utils.TEMP_AAR_FILE_NAME
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.SupervisorJob
import kotlinx.coroutines.runBlocking
import org.gradle.api.logging.Logger
import java.io.File

/**
 * A helper class that comprises all the functionalities for downloading and managing security SDK.
 *
 * @property authToken The authorization token required for network operations.
 * @property certificateHashes A list of certificate hashes to be bundled in the SDK.
 * @property logger Logger instance for logging events and errors.
 * @property appId An id uniquely identifying the application for pinging network services.
 * @property vName Version name of your application, used while making network requests.
 * @property sdkVersion Version of the currently installed SDK, used while making network requests.
 */
internal class Downloader(
    private val authToken: String,
    private val certificateHashes: List<String>,
    private val logger: Logger,
    private val appId: String,
    private val vName: String,
    private val sdkVersion: String,
    private val timeoutConfiguration: TimeoutConfiguration
) {
    private val job = SupervisorJob()
    private val coroutineScope = CoroutineScope(Dispatchers.IO + job)

    /**
     * Downloads the security SDK by making a network request.
     *
     * This function begins by creating a temporary file for the impending download. It then initiates checking or downloading the .aar file.
     * If the .aar file exists, it gets downloaded. Otherwise, a request for generation gets sent. It checks the status and retries downloading post generation.
     * It throws an exception on failure of the above operations.
     *
     * @return File The downloaded .aar (Android Archive) file represented as a `File`.
     * @throws AppsFlyerGradleException if the generation and download of Resources is unsuccessful.
     */
    fun downloadSecuritySdk(

    ): File {
        // Create temp file
        val tempFile = File.createTempFile(TEMP_AAR_FILE_NAME, AAR_FILE_EXT)
        NetworkRepository(authToken, logger, timeoutConfiguration).use { networkRepository ->

            if (!downloadAar(networkRepository, tempFile)) {
                // if aar is not exist in S3 yet, we should create it
                generateAar(networkRepository)
                getBuildStatus(networkRepository)
                if (!downloadAar(networkRepository, tempFile)) {
                    "Failed to generate and download Resources.".logAndThrow()
                }
            }
        }
        return tempFile
    }

    /**
     * Sends a request via the network repository to generate an .aar (Android Archive) file.
     *
     * @param networkRepository The repository through which the request to generate .aar file gets sent.
     */
    private fun generateAar(networkRepository: NetworkRepository) {
        val genBody = GenerateAarBody(certificateHashes)
        blocking {
            networkRepository.generateAar(appId, vName, sdkVersion, genBody)
        }
    }

    /**
     * Retrieves the current build status by sending a request through the network repository.
     *
     * The function keeps polling the server until the status is not "in progress" which means the .aar generation is complete.
     * Throws an exception if the received status is different than 'DONE'.
     *
     * @param networkRepository Network repository through which status request is sent.
     * @throws AppsFlyerGradleException if the response status doesn't equal DONE.
     */
    private fun getBuildStatus(networkRepository: NetworkRepository) {
        val statusResponse =
            blocking { networkRepository.getStatusUntilNotInProgress(appId, vName, sdkVersion) }
        if (statusResponse.status != Status.DONE) {
            GET_STATUS_ERR.format(statusResponse.status, statusResponse.error).logAndThrow()
        }
    }

    /**
     * Downloads the .aar (Android Archive) file via the passed NetworkRepository.
     *
     * After downloading, writes it to the passed outputFile. If download is unsuccessful, it doesn't throw an error, instead returns false.
     *
     * @param networkRepository The repository through which .aar file gets downloaded.
     * @param outputFile The output file where the .aar file needs to be written.
     * @return Boolean indicating whether download operation was successful or not.
     */
    private fun downloadAar(networkRepository: NetworkRepository, outputFile: File): Boolean {
        val downloadAarResponse: DownloadAarResponse = blocking {
            networkRepository.downloadAar(appId, vName, sdkVersion, outputFile)
        }

        return downloadAarResponse.isFileExist
    }

    private fun <T> blocking(block: suspend CoroutineScope.() -> T): T =
        runBlocking(coroutineScope.coroutineContext, block)

    private fun String.logAndThrow() {
        logger.error(this)
        throw AppsFlyerGradleException(this)
    }
}