package com.appsflyer.security.plugin.utils

import com.android.build.api.variant.ApplicationVariant
import com.android.build.api.variant.ComponentIdentity
import com.appsflyer.security.plugin.FlavorConfig
import java.io.File
import java.io.IOException
import java.util.Locale

/**
 * Extension property of ApplicationVariant for providing versionName.
 */
internal val ApplicationVariant.versionName: String
    get() = outputs.firstNotNullOf { it.versionName.get() }

/**
 * Extension property of ApplicationVariant for providing baseAarPath based on the appId.
 */
internal val ApplicationVariant.baseAarPath
    get() = "$BASE_AAR_PATH${applicationId.get().replace(".", "_")}"

/**
 * Extension function of ApplicationVariant for generating the .aar file name.
 *
 * @param flavorConfig Flavor configuration.
 * @return String of formatted .aar file name.
 */
internal fun ApplicationVariant.getAarFileName(flavorConfig: FlavorConfig?) =
    AAR_BASE_NAME.format(applicationId.get(), flavorConfig?.channel.orEmpty())

/**
 * Extension property of ComponentIdentity for providing capitalized name.
 */
internal val ComponentIdentity.capitalizeName
    get() = name.replaceFirstChar {
        if (it.isLowerCase()) it.titlecase(
            Locale.getDefault()
        ) else it.toString()
    }

/**
 * Extension function of File for moving the file to a different location, with an option to overwrite if the file already exists.
 *
 * @param destination The destination file.
 * @param overwrite A boolean flag indicating whether the destination file should be overwritten if it already exists.
 * @return Boolean indicating whether the file operation was successful.
 * @throws IOException If unable to create the parent directory of the destination file or if unable to delete existing file at the destination.
 */
internal fun File.moveTo(destination: File, overwrite: Boolean = false): Boolean {
    // Check if destination directory exists or not
    if (!destination.parentFile.exists()) {
        if (!destination.parentFile.mkdirs()) {
            throw IOException("Unable to create directory: ${destination.parentFile}")
        }
    }

    // If the destination file already exists, delete it based on the provided 'overwrite' parameter
    if (destination.exists()) {
        if (!overwrite) return false
        if (!destination.delete()) throw IOException("Failed to delete existing file: $destination")
    }

    // Try to rename (move) the file
    return if (renameTo(destination)) {
        // If rename successful, all done
        true
    } else {
        // If rename failed, try to copy the file and delete the original
        copyTo(destination, overwrite).exists() && delete()
    }
}

/**
 * A [File] extension property to check if a file both exists and is not empty.
 *
 * This property checks whether the file mentioned in the receiver [File] object exists,
 * and also ensures that it's not an empty file by testing whether the array of bytes
 * read from the file is not empty.
 *
 * @return `true` if the file exists and has at least one byte of data, `false` otherwise.
 */
internal val File.existsAndValid: Boolean
    get() = exists() && readBytes().isNotEmpty()