/*
 * Copyright 2021 Typelevel
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package bobcats

import cats.MonadThrow
import cats.effect.kernel.Async
import cats.syntax.all._
import scodec.bits.ByteVector

private[bobcats] trait HashCompanionPlatform {
  implicit def forAsyncOrMonadThrow[F[_]](
      implicit F: Priority[Async[F], MonadThrow[F]]): Hash[F] =
    if (facade.isNodeJSRuntime)
      new UnsealedHash[F] {
        override def digest(algorithm: HashAlgorithm, data: ByteVector): F[ByteVector] =
          F.join[MonadThrow[F]].catchNonFatal {
            val hash = facade.node.crypto.createHash(algorithm.toStringNodeJS)
            hash.update(data.toUint8Array)
            ByteVector.view(hash.digest())
          }
      }
    else
      F.getPreferred
        .map { implicit F: Async[F] =>
          new UnsealedHash[F] {
            import facade.browser._
            override def digest(algorithm: HashAlgorithm, data: ByteVector): F[ByteVector] =
              F.fromPromise(F.delay(
                crypto.subtle.digest(algorithm.toStringWebCrypto, data.toUint8Array.buffer)))
                .map(ByteVector.view)
          }
        }
        .getOrElse(throw new UnsupportedOperationException(
          "Hash[F] on browsers requires Async[F]"))

}
