/*
 * Copyright (c) 2008-2021 The Aspectran Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.aspectran.core.component.bean.proxy;

import com.aspectran.core.activity.Activity;
import com.aspectran.core.activity.InstantActivity;
import com.aspectran.core.activity.InstantActivityException;
import com.aspectran.core.component.aspect.AspectAdviceRuleRegistry;
import com.aspectran.core.context.ActivityContext;
import com.aspectran.core.context.rule.BeanRule;
import net.sf.cglib.proxy.Enhancer;
import net.sf.cglib.proxy.MethodInterceptor;
import net.sf.cglib.proxy.MethodProxy;

import java.lang.reflect.Method;

/**
 * Create an instance of the dynamic proxy bean using CGLIB.
 */
public class CglibDynamicProxyBean extends AbstractDynamicProxyBean implements MethodInterceptor {

    private final ActivityContext context;

    private final BeanRule beanRule;

    private CglibDynamicProxyBean(ActivityContext context, BeanRule beanRule) {
        super(context.getAspectRuleRegistry());
        this.context = context;
        this.beanRule = beanRule;
    }

    @Override
    public Object intercept(Object proxy, Method method, Object[] args, MethodProxy methodProxy) throws Throwable {
        if (isAvoidAdvice(method)) {
            return methodProxy.invokeSuper(proxy, args);
        }
        if (context.hasCurrentActivity()) {
            Activity activity = context.getCurrentActivity();
            return intercept(proxy, method, args, methodProxy, activity);
        } else {
            try {
                Activity activity = new InstantActivity(context);
                return activity.perform(() -> intercept(proxy, method, args, methodProxy, activity));
            } catch (Exception e) {
                throw new InstantActivityException(e);
            }
        }
    }

    public Object intercept(Object proxy, Method method, Object[] args, MethodProxy methodProxy, Activity activity) throws Throwable {
        String beanId = beanRule.getId();
        String className = beanRule.getClassName();
        String methodName = method.getName();
        AspectAdviceRuleRegistry aarr = getAspectAdviceRuleRegistry(activity, beanId, className, methodName);
        if (aarr == null) {
            return methodProxy.invokeSuper(proxy, args);
        }
        try {
            try {
                beforeAdvice(aarr.getBeforeAdviceRuleList(), beanRule, activity);
                Object result = methodProxy.invokeSuper(proxy, args);
                afterAdvice(aarr.getAfterAdviceRuleList(), beanRule, activity);
                return result;
            } finally {
                finallyAdvice(aarr.getFinallyAdviceRuleList(), beanRule, activity);
            }
        } catch (Exception e) {
            if (exception(aarr.getExceptionRuleList(), e, activity)) {
                return null;
            }
            throw e;
        }
    }

    /**
     * Creates a proxy class of bean and returns an instance of that class.
     * @param context the activity context
     * @param beanRule the bean rule
     * @param args the arguments passed to a constructor
     * @param argTypes the parameter types for a constructor
     * @return a new proxy bean object
     */
    public static Object newInstance(ActivityContext context, BeanRule beanRule, Object[] args, Class<?>[] argTypes) {
        Enhancer enhancer = new Enhancer();
        enhancer.setClassLoader(context.getApplicationAdapter().getClassLoader());
        enhancer.setSuperclass(beanRule.getBeanClass());
        enhancer.setCallback(new CglibDynamicProxyBean(context, beanRule));
        return enhancer.create(argTypes, args);
    }

}
