/*
 * Decompiled with CFR 0.152.
 */
package com.aspectran.core.component.session;

import com.aspectran.core.component.session.AbstractSessionStore;
import com.aspectran.core.component.session.SessionData;
import com.aspectran.core.component.session.UnreadableSessionDataException;
import com.aspectran.core.component.session.UnwritableSessionDataException;
import com.aspectran.utils.MultiException;
import com.aspectran.utils.StringUtils;
import com.aspectran.utils.ToStringBuilder;
import com.aspectran.utils.annotation.jsr305.NonNull;
import com.aspectran.utils.logging.Logger;
import com.aspectran.utils.logging.LoggerFactory;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Stream;

public class FileSessionStore
extends AbstractSessionStore {
    private static final Logger logger = LoggerFactory.getLogger(FileSessionStore.class);
    private final Map<String, String> sessionFileMap = new ConcurrentHashMap<String, String>();
    private File storeDir;
    private boolean deleteUnrestorableFiles = true;

    public FileSessionStore(File storeDir) {
        this.storeDir = storeDir;
    }

    public File getStoreDir() {
        return this.storeDir;
    }

    public boolean isDeleteUnrestorableFiles() {
        return this.deleteUnrestorableFiles;
    }

    public void setDeleteUnrestorableFiles(boolean deleteUnrestorableFiles) {
        this.checkInitializable();
        this.deleteUnrestorableFiles = deleteUnrestorableFiles;
    }

    @Override
    public SessionData load(String id) throws Exception {
        SessionData sessionData;
        String filename = this.sessionFileMap.get(id);
        if (filename == null) {
            if (logger.isTraceEnabled()) {
                logger.trace("Session id=" + id + " does not exist in session file map");
            }
            return null;
        }
        File file = new File(this.storeDir, filename);
        if (!file.exists()) {
            logger.warn("No such file " + filename + " for session id=" + id);
            return null;
        }
        FileInputStream in = new FileInputStream(file);
        try {
            SessionData data = SessionData.deserialize(in);
            data.setLastSaved(file.lastModified());
            sessionData = data;
        }
        catch (Throwable throwable) {
            try {
                try {
                    in.close();
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
            catch (Exception e) {
                throw new UnreadableSessionDataException(id, e);
            }
        }
        in.close();
        return sessionData;
    }

    @Override
    public boolean delete(String id) throws IOException {
        String filename = this.sessionFileMap.remove(id);
        if (filename == null) {
            return false;
        }
        return this.deleteFile(filename);
    }

    private boolean deleteFile(String filename) throws IOException {
        if (filename == null) {
            return false;
        }
        File file = new File(this.storeDir, filename);
        return Files.deleteIfExists(file.toPath());
    }

    @Override
    public boolean exists(String id) {
        String filename = this.sessionFileMap.get(id);
        if (filename == null) {
            return false;
        }
        long expiry = this.getExpiryFromFilename(filename);
        if (expiry == 0L) {
            return true;
        }
        if (expiry == -1L) {
            return false;
        }
        return expiry > System.currentTimeMillis();
    }

    @Override
    public void doSave(String id, SessionData data) throws Exception {
        try {
            this.delete(id);
        }
        catch (IOException e) {
            logger.warn("Failed to delete old data file for session " + id);
        }
        String filename = this.getIdWithExpiry(data);
        File file = new File(this.storeDir, filename);
        try (FileOutputStream fos = new FileOutputStream(file, false);){
            SessionData.serialize(data, fos, this.getNonPersistentAttributes());
            this.sessionFileMap.put(id, filename);
        }
        catch (Exception e) {
            file.delete();
            throw new UnwritableSessionDataException(id, e);
        }
    }

    @Override
    public Set<String> doGetExpired(long time) {
        HashSet<String> expired = new HashSet<String>();
        for (String filename : this.sessionFileMap.values()) {
            try {
                long expiry = this.getExpiryFromFilename(filename);
                if (expiry <= 0L || expiry > time) continue;
                expired.add(this.getIdFromFilename(filename));
            }
            catch (Exception e) {
                logger.warn("Error finding sessions expired before " + time, e);
            }
        }
        return expired;
    }

    @Override
    public void doCleanOrphans(long time) {
        this.sweepDisk(time, false);
    }

    @Override
    public Set<String> getAllSessions() {
        return new HashSet<String>(this.sessionFileMap.keySet());
    }

    private String getIdWithExpiry(@NonNull SessionData data) {
        if (data.getExpiry() > 0L) {
            return data.getExpiry() + "_" + data.getId();
        }
        return data.getId();
    }

    private long getExpiryFromFilename(@NonNull String filename) {
        int index = filename.indexOf(95);
        if (index == -1) {
            return 0L;
        }
        try {
            String s = filename.substring(0, index);
            return Long.parseLong(s);
        }
        catch (NumberFormatException e) {
            logger.warn("Not valid session filename: " + filename, e);
            return -1L;
        }
    }

    @NonNull
    private String getIdFromFilename(@NonNull String filename) {
        int index = filename.indexOf(95);
        if (index == -1) {
            return filename;
        }
        return filename.substring(index + 1);
    }

    private boolean isSessionFilename(String filename) {
        return StringUtils.hasText(filename) && !filename.startsWith(".");
    }

    private void sweepDisk(long time, boolean withManaged) {
        if (logger.isTraceEnabled()) {
            logger.trace("Sweeping " + String.valueOf(this.storeDir) + " for old session files at " + time);
        }
        try (Stream<Path> stream = Files.walk(this.storeDir.toPath(), 1, FileVisitOption.FOLLOW_LINKS);){
            stream.filter(p -> !Files.isDirectory(p, new LinkOption[0])).filter(p -> this.isSessionFilename(p.getFileName().toString())).filter(p -> withManaged || !this.sessionFileMap.containsValue(p.getFileName().toString())).forEach(p -> this.sweepFile(time, (Path)p));
        }
        catch (Exception e) {
            logger.warn("Unable to walk path " + String.valueOf(this.storeDir), e);
        }
    }

    private void sweepFile(long time, Path p) {
        block13: {
            if (p != null) {
                String filename = p.getFileName().toString();
                long expiry = this.getExpiryFromFilename(filename);
                if (expiry > 0L && expiry <= time) {
                    try {
                        if (Files.deleteIfExists(p)) {
                            if (logger.isDebugEnabled()) {
                                logger.debug("Sweep expired session file: " + filename);
                            }
                            break block13;
                        }
                        logger.warn("Could not delete " + filename);
                    }
                    catch (IOException e) {
                        logger.warn("Could not delete " + filename, e);
                    }
                } else if (expiry == -1L && this.isDeleteUnrestorableFiles()) {
                    try {
                        if (Files.deleteIfExists(p)) {
                            if (logger.isDebugEnabled()) {
                                logger.debug("Deleted unrestorable session file: " + String.valueOf(p.getFileName()));
                            }
                        } else {
                            logger.warn("Could not delete " + filename);
                        }
                    }
                    catch (IOException e) {
                        logger.warn("Could not delete " + filename, e);
                    }
                }
            }
        }
    }

    @Override
    protected void doInitialize() throws Exception {
        this.initializeStore();
    }

    @Override
    protected void doDestroy() {
        this.sessionFileMap.clear();
    }

    private void initializeStore() throws Exception {
        if (!this.storeDir.exists()) {
            this.storeDir.mkdirs();
            return;
        }
        if (!(this.storeDir.isDirectory() && this.storeDir.canWrite() && this.storeDir.canRead())) {
            throw new IllegalStateException(this.storeDir.getAbsolutePath() + " must be readable/writable directory");
        }
        MultiException me = new MultiException();
        long now = System.currentTimeMillis();
        try (Stream<Path> stream = Files.walk(this.storeDir.toPath(), 1, FileVisitOption.FOLLOW_LINKS);){
            stream.filter(p -> !Files.isDirectory(p, new LinkOption[0])).filter(p -> this.isSessionFilename(p.getFileName().toString())).forEach(p -> {
                String filename;
                String sessionId;
                String existing;
                this.sweepFile(now - this.getGracePeriodMillis(6.0f), (Path)p);
                if (Files.exists(p, new LinkOption[0]) && (existing = this.sessionFileMap.putIfAbsent(sessionId = this.getIdFromFilename(filename = p.getFileName().toString()), filename)) != null) {
                    try {
                        long existingExpiry = this.getExpiryFromFilename(existing);
                        long thisExpiry = this.getExpiryFromFilename(filename);
                        if (thisExpiry > existingExpiry) {
                            this.sessionFileMap.put(sessionId, filename);
                            Path existingFile = this.storeDir.toPath().resolve(existing);
                            Files.delete(existingFile);
                            if (logger.isDebugEnabled()) {
                                logger.debug("Replaced file " + existing + " with " + filename + " for session " + sessionId);
                            }
                        } else {
                            Files.delete(p);
                            if (logger.isDebugEnabled()) {
                                logger.debug("Deleted file " + filename + " for expired session " + sessionId);
                            }
                        }
                    }
                    catch (IOException e) {
                        me.add(e);
                    }
                }
            });
            me.ifExceptionThrow();
        }
    }

    public String toString() {
        ToStringBuilder tsb = new ToStringBuilder();
        tsb.append("storeDir", this.storeDir);
        tsb.appendForce("deleteUnrestorableFiles", this.deleteUnrestorableFiles);
        tsb.append("gracePeriodSecs", this.getGracePeriodSecs());
        tsb.append("savePeriodSecs", this.getSavePeriodSecs());
        tsb.append("nonPersistentAttributes", this.getNonPersistentAttributes());
        return tsb.toString();
    }
}

