/*
 * Decompiled with CFR 0.152.
 */
package com.authlete.common.util;

import com.authlete.common.api.AuthleteApi;
import com.authlete.common.api.AuthleteApiException;
import com.authlete.common.api.AuthleteApiFactory;
import com.authlete.common.dto.AuthorizedClientListResponse;
import com.authlete.common.dto.Client;
import com.authlete.common.dto.ClientAuthorizationGetListRequest;
import com.authlete.common.dto.ClientListResponse;
import com.authlete.common.dto.Service;
import com.authlete.common.dto.ServiceListResponse;
import com.authlete.common.util.Utils;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

public class CLI {
    private static final String HELP = "OVERVIEW:\n\n  Command line interface for Authlete API.\n\n\nUSAGE:\n\n  java -cp {classpath} com.authlete.common.util.CLI [options] {API name} [arguments]\n\n    or\n\n  mvn -q exec:java -Dexec.args='[options] {API name} [arguments]'\n\n    or\n\n  bin/authlete-cli.sh [options] {API name} [arguments]\n\n\nOPTIONS:\n\n  -h | --help : Show this help and exit.\n  -v | --verbose : Verbose output.\n\n\nAPI NAME AND ARGUMENTS:\n\n  getClient {clientId}\n  getClientAuthorizationList subject={subject} [developer={developer}] [start={start}] [end={end}]\n  getClientList [developer={developer}] [start={start}] [end={end}]\n  getService {serviceApiKey}\n  getServiceConfiguration [pretty={true|false}]\n  getServiceJwks [pretty={true|false}] [includePrivateKeys={true|false}]\n  getServiceList [start={start}] [end={end}]\n\n  # API name is case-insensitive.\n\n\nEXAMPLES:\n\n  $ bin/authlete-cli.sh --help\n  $ bin/authlete-cli.sh getClient 4326385670\n  $ bin/authlete-cli.sh getClientAuthorizationList subject=authlete_5526908833\n  $ bin/authlete-cli.sh getClientList developer=authlete_5526908833\n  $ bin/authlete-cli.sh getService 5526908833\n  $ bin/authlete-cli.sh getServiceConfiguration pretty=true\n  $ bin/authlete-cli.sh getServiceJwks pretty=true includePrivateKeys=true\n  $ bin/authlete-cli.sh getServiceList start=1\n\n\nNOTE:\n\n  'authlete.properties' must exist. See JavaDoc of AuthletePropertiesConfiguration.";

    public static void main(String[] args) {
        new CLI().execute(args);
    }

    private void execute(String[] args) {
        Settings settings = this.parseArguments(args);
        this.validateSettings(settings);
        AuthleteApi api = AuthleteApiFactory.getDefaultApi();
        try {
            this.executeApi(api, settings);
        }
        catch (AuthleteApiException e) {
            this.reportError(e);
        }
    }

    private Settings parseArguments(String[] args) {
        Settings settings = new Settings();
        int len = args.length;
        for (int i = 0; i < len; ++i) {
            String argv = args[i];
            if (argv.equals("-h") || argv.equals("--help")) {
                settings.help = true;
                break;
            }
            if (argv.equals("-v") || argv.equals("--verbose")) {
                ++settings.verbose;
                continue;
            }
            if (argv.startsWith("-")) {
                settings.setError("Unknown option: '%s'", argv);
                break;
            }
            settings.apiName = ApiName.parse(argv);
            settings.setParameters(args, i + 1);
            if (settings.apiName != null) break;
            settings.setError("Unknown API name: '%s'", argv);
            break;
        }
        return settings;
    }

    private void validateSettings(Settings settings) {
        if (settings.help) {
            System.out.println(HELP);
            System.exit(0);
        }
        if (settings.error) {
            this.showErrorAndExit(settings.errorMessage);
        }
        if (settings.apiName == null) {
            this.showErrorAndExit("API name is not given.");
        }
    }

    private void showErrorAndExit(String message) {
        System.err.println("ERROR: " + message);
        System.err.println("HELP: Use --help option.");
        System.exit(1);
    }

    private void executeApi(AuthleteApi api, Settings settings) {
        switch (settings.apiName) {
            case GET_CLIENT: {
                this.executeGetClientApi(api, settings);
                break;
            }
            case GET_CLIENT_AUTHORIZATION_LIST: {
                this.executeGetClientAuthorizationListApi(api, settings);
                break;
            }
            case GET_CLIENT_LIST: {
                this.executeGetClientListApi(api, settings);
                break;
            }
            case GET_SERVICE: {
                this.executeGetServiceApi(api, settings);
                break;
            }
            case GET_SERVICE_CONFIGURATION: {
                this.executeGetServiceConfigurationApi(api, settings);
                break;
            }
            case GET_SERVICE_JWKS: {
                this.executeGetServiceJwksApi(api, settings);
                break;
            }
            case GET_SERVICE_LIST: {
                this.executeGetServiceListApi(api, settings);
                break;
            }
            default: {
                throw new AssertionError((Object)settings.apiName.name());
            }
        }
    }

    private void reportError(AuthleteApiException exception) {
        System.err.println("ERROR: " + exception.getMessage());
        System.err.format("%d %s\n", exception.getStatusCode(), exception.getStatusMessage());
        this.reportHeaders(exception);
        this.reportResponseBody(exception);
    }

    private void reportHeaders(AuthleteApiException exception) {
        Map<String, List<String>> headers = exception.getResponseHeaders();
        if (headers == null) {
            return;
        }
        for (Map.Entry<String, List<String>> entry : headers.entrySet()) {
            String name = entry.getKey();
            List<String> list = entry.getValue();
            if (name == null) continue;
            if (list.size() == 0) {
                System.err.format("%s:\n", name);
                continue;
            }
            for (String value : list) {
                System.err.format("%s: %s\n", name, value);
            }
        }
    }

    private void reportResponseBody(AuthleteApiException exception) {
        String responseBody = exception.getResponseBody();
        if (responseBody == null) {
            return;
        }
        System.err.println();
        System.err.print(responseBody);
    }

    private int getIntegerOrExitIfNeeded(Map<String, String> parameters, String key, int defaultValue) {
        if (!parameters.containsKey(key)) {
            return defaultValue;
        }
        String value = parameters.get(key);
        if (value == null || value.length() == 0) {
            this.showErrorAndExit(String.format("'%s' needs its value.", key));
        }
        try {
            return Integer.parseInt(value);
        }
        catch (Exception e) {
            this.showErrorAndExit(String.format("Failed to parse the value of '%s' as an integer.", key));
            return defaultValue;
        }
    }

    private boolean getBoolean(Map<String, String> parameters, String key, boolean defaultValue) {
        if (!parameters.containsKey(key)) {
            return defaultValue;
        }
        String value = parameters.get(key);
        if (value == null || value.length() == 0) {
            return true;
        }
        return Boolean.parseBoolean(value);
    }

    private String getFirstKeyOrExit(Map<String, String> parameters, String apiName, String parameterName) {
        int size = parameters.size();
        if (size == 0) {
            this.showErrorAndExit(String.format("%s requires {%s}.", apiName, parameterName));
        } else if (size != 1) {
            this.showErrorAndExit(String.format("Too many arguments for %s.", apiName));
        }
        return this.getFirstKey(parameters);
    }

    private String getFirstKey(Map<String, String> parameters) {
        try {
            return parameters.keySet().iterator().next();
        }
        catch (Exception e) {
            return null;
        }
    }

    private void verbose(Settings settings, String format, Object ... args) {
        if (settings.verbose <= 0) {
            return;
        }
        System.out.format(format, args);
        System.out.println();
    }

    private void executeGetClientApi(AuthleteApi api, Settings settings) {
        long clientId;
        String value = this.getFirstKeyOrExit(settings.parameters, "getClient", "clientId");
        try {
            clientId = Long.parseLong(value);
        }
        catch (Exception e) {
            this.showErrorAndExit("The value of {clientId} is invalid.");
            return;
        }
        this.verbose(settings, "Calling getClient(clientId=%d)", clientId);
        Client client = api.getClient(clientId);
        System.out.println(Utils.toJson(client, true));
    }

    private void executeGetClientAuthorizationListApi(AuthleteApi api, Settings settings) {
        ClientAuthorizationGetListRequest request = new ClientAuthorizationGetListRequest();
        String subject = settings.parameters.get("subject");
        if (subject == null || subject.length() == 0) {
            this.showErrorAndExit("getClientAuthorizationList requires a 'subject' value.");
            return;
        }
        request.setSubject(subject);
        if (settings.parameters.containsKey("developer")) {
            request.setDeveloper(settings.parameters.get("developer"));
        }
        if (settings.parameters.containsKey("start")) {
            request.setStart(this.getIntegerOrExitIfNeeded(settings.parameters, "start", 0));
        }
        if (settings.parameters.containsKey("end")) {
            request.setStart(this.getIntegerOrExitIfNeeded(settings.parameters, "end", 5));
        }
        this.verbose(settings, "Calling getClientAuthorizationList(request) (subject=%s, developer=%s, start=%d, end=%d)", request.getSubject(), request.getDeveloper(), request.getStart(), request.getEnd());
        AuthorizedClientListResponse response = api.getClientAuthorizationList(request);
        System.out.println(Utils.toJson(response, true));
    }

    private void executeGetClientListApi(AuthleteApi api, Settings settings) {
        ClientListResponse response;
        String developer;
        boolean useRange = false;
        int start = 0;
        int end = 5;
        if (settings.parameters.containsKey("start") || settings.parameters.containsKey("end")) {
            start = this.getIntegerOrExitIfNeeded(settings.parameters, "start", 0);
            end = this.getIntegerOrExitIfNeeded(settings.parameters, "end", 5);
            useRange = true;
        }
        if ((developer = settings.parameters.get("developer")) != null) {
            if (useRange) {
                this.verbose(settings, "Calling getClientList(developer=%s, start=%d, end=%d)", developer, start, end);
                response = api.getClientList(developer, start, end);
            } else {
                this.verbose(settings, "Calling getClientList(developer=%s)", developer);
                response = api.getClientList(developer);
            }
        } else if (useRange) {
            this.verbose(settings, "Calling getClientList(start=%d, end=%d)", start, end);
            response = api.getClientList(start, end);
        } else {
            this.verbose(settings, "Calling getClientList()", new Object[0]);
            response = api.getClientList();
        }
        System.out.println(Utils.toJson(response, true));
    }

    private void executeGetServiceConfigurationApi(AuthleteApi api, Settings settings) {
        boolean pretty = this.getBoolean(settings.parameters, "pretty", true);
        this.verbose(settings, "Calling getServiceConfiguration(pretty=%s)", pretty);
        String configuration = api.getServiceConfiguration(pretty);
        System.out.println(configuration);
    }

    private void executeGetServiceJwksApi(AuthleteApi api, Settings settings) {
        boolean pretty = this.getBoolean(settings.parameters, "pretty", true);
        boolean include = this.getBoolean(settings.parameters, "includePrivateKeys", false);
        this.verbose(settings, "Calling getServiceJwks(pretty=%s, includePrivateKeys=%s)", pretty, include);
        String jwks = api.getServiceJwks(pretty, include);
        System.out.println(jwks);
    }

    private void executeGetServiceApi(AuthleteApi api, Settings settings) {
        long serviceApiKey;
        String value = this.getFirstKeyOrExit(settings.parameters, "getService", "serviceApiKey");
        try {
            serviceApiKey = Long.parseLong(value);
        }
        catch (Exception e) {
            this.showErrorAndExit("The value of {serviceApiKey} is invalid.");
            return;
        }
        this.verbose(settings, "Calling getService(serviceApiKey=%d)", serviceApiKey);
        Service service = api.getService(serviceApiKey);
        System.out.println(Utils.toJson(service, true));
    }

    private void executeGetServiceListApi(AuthleteApi api, Settings settings) {
        ServiceListResponse response;
        if (settings.parameters.containsKey("start") || settings.parameters.containsKey("end")) {
            int start = this.getIntegerOrExitIfNeeded(settings.parameters, "start", 0);
            int end = this.getIntegerOrExitIfNeeded(settings.parameters, "end", 5);
            this.verbose(settings, "Calling getServiceList(start=%d, end=%d)", start, end);
            response = api.getServiceList(start, end);
        } else {
            this.verbose(settings, "Calling getServiceList()", new Object[0]);
            response = api.getServiceList();
        }
        System.out.println(Utils.toJson(response, true));
    }

    private static class Settings {
        public boolean help;
        public boolean error;
        public int verbose;
        public String errorMessage;
        public ApiName apiName;
        public Map<String, String> parameters = new LinkedHashMap<String, String>();

        private Settings() {
        }

        public void setError(String format, Object ... args) {
            this.error = true;
            this.errorMessage = String.format(format, args);
        }

        public void setParameters(String[] args, int from) {
            block4: for (int i = from; i < args.length; ++i) {
                String[] pair = args[i].split("=", 2);
                String key = null;
                String value = null;
                switch (pair.length) {
                    case 2: {
                        value = pair[1];
                    }
                    case 1: {
                        key = pair[0];
                        break;
                    }
                    default: {
                        continue block4;
                    }
                }
                this.parameters.put(key, value);
            }
        }
    }

    private static enum ApiName {
        GET_CLIENT("getClient"),
        GET_CLIENT_AUTHORIZATION_LIST("getClientAuthorizationList"),
        GET_CLIENT_LIST("getClientList"),
        GET_SERVICE("getService"),
        GET_SERVICE_CONFIGURATION("getServiceConfiguration"),
        GET_SERVICE_JWKS("getServiceJwks"),
        GET_SERVICE_LIST("getServiceList");

        private final String name;

        private ApiName(String name) {
            this.name = name;
        }

        public static ApiName parse(String name) {
            for (ApiName api : ApiName.values()) {
                if (!api.name.equalsIgnoreCase(name)) continue;
                return api;
            }
            return null;
        }
    }
}

